# API設計書

## 概要

本ドキュメントは、Redox OS ビルドシステム（cookbook）のAPI設計書です。本プロジェクトはHTTP REST APIではなく、RustライブラリAPIおよびCLIインターフェースを提供するビルドツールです。Redox OSのパッケージのフェッチ、ビルド、パッケージングを行うための機能を提供します。

## 共通仕様

### ライブラリクレート情報

```toml
[lib]
name = "cookbook"
path = "src/lib.rs"
```

### 公開定数

| 定数名 | 型 | 値 | 説明 |
| --- | --- | --- | --- |
| WALK_DEPTH | usize | 16 | 依存関係ツリーの最大探索深度 |
| REMOTE_PKG_SOURCE | &str | "https://static.redox-os.org/pkg" | リモートパッケージソースのデフォルトURL |

### 共通エラー型

本ライブラリは主に以下のエラー型を使用します：

| エラー型 | 説明 |
| --- | --- |
| String | 多くの関数で使用される汎用エラー型 |
| PackageError | パッケージ関連エラー（pkg crateから） |
| anyhow::Error | CLIバイナリで使用されるエラー型 |

## API一覧

### ライブラリAPI（公開モジュール）

| モジュール | パス | 説明 |
| --- | --- | --- |
| blake3 | cookbook::blake3 | BLAKE3ハッシュ計算機能 |
| config | cookbook::config | 設定管理機能 |
| cook | cookbook::cook | ビルド・フェッチ・パッケージング機能 |
| recipe | cookbook::recipe | レシピ定義・解析機能 |

### CLIバイナリ

| バイナリ名 | パス | 説明 |
| --- | --- | --- |
| repo | src/bin/repo.rs | メインのレポジトリ管理CLI |
| cookbook_redoxer | src/bin/cookbook_redoxer.rs | Redoxer統合ラッパー |
| repo_builder | src/bin/repo_builder.rs | パッケージ公開ツール |

## 各APIエンドポイント定義

### blake3モジュール

#### 1 blake3_progress

ファイルのBLAKE3ハッシュを進捗表示付きで計算します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn blake3_progress<P: AsRef<Path>>(path: P) -> Result<String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| path | P: AsRef\<Path\> | ○ | ハッシュを計算するファイルパス |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Result\<String\> | 成功時: BLAKE3ハッシュ文字列（16進数）、失敗時: IOエラー |

---

#### 2 blake3_silent

ファイルのBLAKE3ハッシュを進捗表示なしで計算します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn blake3_silent<P: AsRef<Path>>(path: P) -> Result<String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| path | P: AsRef\<Path\> | ○ | ハッシュを計算するファイルパス |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Result\<String\> | 成功時: BLAKE3ハッシュ文字列（16進数）、失敗時: IOエラー |

---

#### 3 hash_to_hex

BLAKE3ハッシュバイト配列を16進文字列に変換します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn hash_to_hex(h: [u8; 32]) -> String` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| h | [u8; 32] | ○ | 32バイトのBLAKE3ハッシュ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| String | 16進数文字列 |

---

### configモジュール

#### 4 init_config

設定を初期化します。`cookbook.toml`ファイルがあれば読み込み、なければデフォルト値を使用します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn init_config()` |
| 可視性 | public |

**環境変数**

| 環境変数名 | デフォルト | 説明 |
| --- | --- | --- |
| CI | - | 設定されていればTUIを無効化 |
| COOKBOOK_MAKE_JOBS | nproc | ビルドジョブ数 |
| COOKBOOK_LOGS | CI依存 | ログ出力を有効化 |
| COOKBOOK_OFFLINE | false | オフラインモード |
| COOKBOOK_VERBOSE | true | 詳細ログ |
| COOKBOOK_NONSTOP | false | エラー時も続行 |
| COOKBOOK_CLEAN_BUILD | false | ビルド前にビルドディレクトリをクリーン |
| COOKBOOK_CLEAN_TARGET | false | ビルド後にターゲットディレクトリをクリーン |

---

#### 5 get_config

初期化済みの設定を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn get_config() -> &'static CookbookConfig` |
| 可視性 | public |
| 前提条件 | init_config()が事前に呼ばれていること |

**戻り値**

| 型 | 説明 |
| --- | --- |
| &'static CookbookConfig | 静的な設定への参照 |

---

#### 6 translate_mirror

URLをミラーURLに変換します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn translate_mirror(original_url: &str) -> String` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| original_url | &str | ○ | 変換元URL |

**戻り値**

| 型 | 説明 |
| --- | --- |
| String | ミラー設定にマッチすれば変換後URL、なければ元のURL |

---

### recipeモジュール

#### 7 Recipe::new

レシピファイルからRecipe構造体を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn new(file: &PathBuf) -> Result<Recipe, PackageError>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| file | &PathBuf | ○ | recipe.tomlファイルへのパス |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Result\<Recipe, PackageError\> | 成功時: パースされたRecipe、失敗時: パースエラー |

---

#### 8 CookRecipe::from_name

パッケージ名からCookRecipeを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn from_name(name: PackageName) -> Result<Self, PackageError>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | PackageName | ○ | パッケージ名 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Result\<CookRecipe, PackageError\> | 成功時: CookRecipe、失敗時: パッケージが見つからない等のエラー |

---

#### 9 CookRecipe::from_path

ディレクトリパスからCookRecipeを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn from_path(dir: &Path, read_recipe: bool, is_host: bool) -> Result<Self, PackageError>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dir | &Path | ○ | レシピディレクトリへのパス |
| read_recipe | bool | ○ | recipe.tomlを読み込むか |
| is_host | bool | ○ | ホストパッケージかどうか |

---

#### 10 CookRecipe::get_build_deps_recursive

ビルド依存関係を再帰的に取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn get_build_deps_recursive(names: &[PackageName], include_dev: bool) -> Result<Vec<Self>, PackageError>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| names | &[PackageName] | ○ | パッケージ名のスライス |
| include_dev | bool | ○ | 開発依存も含めるか |

---

#### 11 CookRecipe::get_package_deps_recursive

パッケージ依存関係を再帰的に取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn get_package_deps_recursive(names: &[PackageName], include_names: bool) -> Result<Vec<PackageName>, PackageError>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| names | &[PackageName] | ○ | パッケージ名のスライス |
| include_names | bool | ○ | 元のパッケージ名も結果に含めるか |

---

### cook::fetchモジュール

#### 12 fetch

レシピのソースをダウンロードします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn fetch(recipe: &CookRecipe, logger: &PtyOut) -> Result<PathBuf, String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| recipe | &CookRecipe | ○ | ビルドするレシピ |
| logger | &PtyOut | ○ | ログ出力先（Noneで標準エラー出力） |

**処理内容**

- Git/Tar/Path/SameAsソースタイプに対応
- サブモジュールの更新
- パッチの適用
- Cargoプロジェクトの依存フェッチ

---

#### 13 fetch_offline

オフラインモードでソースを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn fetch_offline(recipe: &CookRecipe, logger: &PtyOut) -> Result<PathBuf, String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| recipe | &CookRecipe | ○ | ビルドするレシピ |
| logger | &PtyOut | ○ | ログ出力先 |

---

#### 14 fetch_remote

リモートからビルド済みパッケージをダウンロードします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn fetch_remote(recipe_dir: &Path, recipe: &CookRecipe, offline_mode: bool, logger: &PtyOut) -> Result<(), String>` |
| 可視性 | public |

---

### cook::cook_buildモジュール

#### 15 build

レシピをビルドします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn build(recipe_dir: &Path, source_dir: &Path, target_dir: &Path, name: &PackageName, recipe: &Recipe, cook_config: &CookConfig, check_source: bool, logger: &PtyOut) -> Result<(Vec<PathBuf>, BTreeSet<PackageName>), String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| recipe_dir | &Path | ○ | レシピディレクトリ |
| source_dir | &Path | ○ | ソースディレクトリ |
| target_dir | &Path | ○ | ターゲットディレクトリ |
| name | &PackageName | ○ | パッケージ名 |
| recipe | &Recipe | ○ | レシピ定義 |
| cook_config | &CookConfig | ○ | ビルド設定 |
| check_source | bool | ○ | ソースの変更をチェックするか |
| logger | &PtyOut | ○ | ログ出力先 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Result\<(Vec\<PathBuf\>, BTreeSet\<PackageName\>), String\> | 成功時: (ステージディレクトリ一覧, 自動検出された依存関係) |

**サポートするビルドテンプレート**

| テンプレート | 説明 |
| --- | --- |
| none | ビルドなし（メタパッケージ用） |
| remote | リモートからダウンロード |
| cargo | Cargo（Rust）ビルド |
| configure | configure + make |
| cmake | CMakeビルド |
| meson | Mesonビルド |
| custom | カスタムスクリプト |

---

#### 16 get_stage_dirs

ステージディレクトリのリストを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn get_stage_dirs(features: &Vec<OptionalPackageRecipe>, target_dir: &Path) -> Vec<PathBuf>` |
| 可視性 | public |

---

#### 17 remove_stage_dir

ステージディレクトリを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn remove_stage_dir(stage_dir: &PathBuf) -> Result<(), String>` |
| 可視性 | public |

---

### cook::packageモジュール

#### 18 package

レシピをパッケージ化します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn package(recipe: &CookRecipe, stage_dirs: &Vec<PathBuf>, auto_deps: &BTreeSet<PackageName>, logger: &PtyOut) -> Result<(), String>` |
| 可視性 | public |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| recipe | &CookRecipe | ○ | レシピ |
| stage_dirs | &Vec\<PathBuf\> | ○ | ステージディレクトリ一覧 |
| auto_deps | &BTreeSet\<PackageName\> | ○ | 自動検出された依存関係 |
| logger | &PtyOut | ○ | ログ出力先 |

**処理内容**

- pkgar形式でアーカイブを作成
- Ed25519鍵で署名
- メタデータ（.toml）を生成

---

#### 19 package_target

パッケージのターゲットトリプルを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn package_target(name: &PackageName) -> &'static str` |
| 可視性 | public |

---

#### 20 package_stage_paths

ステージパス情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `pub fn package_stage_paths(package: Option<&OptionalPackageRecipe>, target_dir: &Path) -> (PathBuf, PathBuf, PathBuf)` |
| 可視性 | public |

**戻り値**

| 型 | 説明 |
| --- | --- |
| (PathBuf, PathBuf, PathBuf) | (ステージディレクトリ, pkgarファイルパス, tomlファイルパス) |

---

## CLIコマンド仕様

### repoコマンド

```
Usage: repo <command> [flags] <recipe1> <recipe2> ...
```

#### コマンド一覧

| コマンド | 説明 |
| --- | --- |
| fetch | レシピソースをダウンロード |
| cook | レシピパッケージをビルド |
| unfetch | レシピソースを削除 |
| clean | レシピアーティファクトを削除 |
| push | パッケージをsysrootに展開 |
| find | レシピパッケージのパスを表示 |
| tree | レシピパッケージのツリーを表示 |

#### 共通フラグ

| フラグ | 説明 |
| --- | --- |
| --cookbook=\<dir\> | recipesフォルダのパス（デフォルト: $PWD/recipes） |
| --repo=\<dir\> | repoフォルダのパス（デフォルト: $PWD/repo） |
| --sysroot=\<dir\> | sysrootフォルダのパス |
| --with-package-deps | パッケージ依存を含める |
| --all | 全レシピに適用 |
| --category=\<category\> | 指定カテゴリのレシピに適用 |
| --filesystem=\<file\> | インストーラ設定ファイルで上書き |
| --repo-binary | バイナリリポジトリを使用 |

---

## データ構造

### CookConfig

ビルド設定を表す構造体。

```rust
pub struct CookConfig {
    pub offline: bool,        // オフラインモード
    pub jobs: usize,          // 並列ジョブ数
    pub tui: bool,            // TUIを使用するか
    pub logs: bool,           // ログを保存するか
    pub nonstop: bool,        // エラー時も続行するか
    pub verbose: bool,        // 詳細ログを出力するか
    pub clean_build: bool,    // ビルド前にクリーン
    pub clean_target: bool,   // ビルド後にクリーン
}
```

### Recipe

レシピ定義を表す構造体。

```rust
pub struct Recipe {
    pub source: Option<SourceRecipe>,           // ソース定義
    pub build: BuildRecipe,                     // ビルド定義
    pub package: PackageRecipe,                 // パッケージ定義
    pub optional_packages: Vec<OptionalPackageRecipe>, // オプショナルパッケージ
}
```

### SourceRecipe

ソースの取得方法を表すenum。

```rust
pub enum SourceRecipe {
    SameAs { same_as: String },           // 他パッケージのソースを共用
    Path { path: String },                 // ローカルパス
    Git {                                  // Gitリポジトリ
        git: String,
        upstream: Option<String>,
        branch: Option<String>,
        rev: Option<String>,
        shallow_clone: Option<bool>,
        patches: Vec<String>,
        script: Option<String>,
    },
    Tar {                                  // tarアーカイブ
        tar: String,
        blake3: Option<String>,
        patches: Vec<String>,
        script: Option<String>,
    },
}
```

### BuildKind

ビルドテンプレートを表すenum。

```rust
pub enum BuildKind {
    None,                                      // ビルドなし
    Remote,                                    // リモートダウンロード
    Cargo { package_path: Option<String>, cargoflags: String },
    Configure { configureflags: Vec<String> },
    Cmake { cmakeflags: Vec<String> },
    Meson { mesonflags: Vec<String> },
    Custom { script: String },
}
```

---

## 備考

- 本プロジェクトはRedox OSのビルドシステムであり、HTTP APIは提供していません
- CLIとライブラリの両方のインターフェースを提供しています
- pkgar形式（Ed25519署名付き）でパッケージを作成します
- 依存関係の自動検出（動的リンクされたライブラリから）をサポートしています
- TUIモードではマルチスレッドでフェッチとビルドを並行して実行できます
