# アーキテクチャ構成図

## システム全体構成図

```mermaid
graph TB
    subgraph "External Systems"
        GIT[Git Repositories<br/>GitLab/GitHub]
        TAR[HTTP/FTP Servers<br/>Tarball Sources]
        CARGO[Cargo Registry<br/>crates.io]
        PKGAR_REMOTE[Remote pkgar Repository<br/>static.redox-os.org]
    end

    subgraph "Redox Cookbook System"
        subgraph "CLI Layer"
            REPO[repo<br/>Main CLI]
            REPO_BUILDER[repo_builder<br/>Repository Generator]
            COOKBOOK_REDOXER[cookbook_redoxer<br/>Redoxer Wrapper]
        end

        subgraph "Application Layer"
            FETCH[fetch.rs<br/>Source Fetcher]
            BUILD[cook_build.rs<br/>Build Controller]
            PACKAGE[package.rs<br/>Packager]
            TREE[tree.rs<br/>Dependency Tree]
        end

        subgraph "Domain Layer"
            RECIPE[recipe.rs<br/>Recipe Models]
            CONFIG[config.rs<br/>Configuration]
        end

        subgraph "Infrastructure Layer"
            FS[fs.rs<br/>File System Utils]
            SCRIPT[script.rs<br/>Build Scripts]
            PTY[pty.rs<br/>PTY Control]
            IDENT[ident.rs<br/>Build Identity]
            BLAKE3[blake3.rs<br/>Hash Utils]
        end
    end

    subgraph "Local Storage"
        RECIPES_DIR[(recipes/<br/>Recipe Definitions)]
        SOURCE_DIR[(source/<br/>Source Cache)]
        TARGET_DIR[(target/<br/>Build Artifacts)]
        REPO_DIR[(repo/<br/>Package Repository)]
    end

    %% External connections
    GIT --> FETCH
    TAR --> FETCH
    CARGO --> BUILD
    PKGAR_REMOTE --> FETCH

    %% CLI to Application
    REPO --> FETCH
    REPO --> BUILD
    REPO --> PACKAGE
    REPO --> TREE
    REPO_BUILDER --> PACKAGE

    %% Application to Domain
    FETCH --> RECIPE
    FETCH --> CONFIG
    BUILD --> RECIPE
    BUILD --> CONFIG
    PACKAGE --> RECIPE

    %% Application to Infrastructure
    FETCH --> FS
    FETCH --> SCRIPT
    BUILD --> FS
    BUILD --> SCRIPT
    BUILD --> PTY
    PACKAGE --> FS
    PACKAGE --> BLAKE3
    FETCH --> IDENT
    BUILD --> IDENT

    %% Infrastructure to Storage
    FS --> RECIPES_DIR
    FS --> SOURCE_DIR
    FS --> TARGET_DIR
    FS --> REPO_DIR
```

## レイヤー構成図

```mermaid
graph TD
    subgraph "Presentation Layer"
        TUI[TUI Interface<br/>ratatui]
        CLI_PARSE[CLI Parser<br/>Arguments Handling]
    end

    subgraph "Application Layer"
        COOK_CMD[Cook Command]
        FETCH_CMD[Fetch Command]
        CLEAN_CMD[Clean Command]
        PUSH_CMD[Push Command]
        TREE_CMD[Tree Command]
    end

    subgraph "Domain Layer"
        RECIPE_MODEL[Recipe<br/>SourceRecipe<br/>BuildRecipe]
        COOK_RECIPE[CookRecipe]
        BUILD_KIND[BuildKind<br/>Cargo/Configure/CMake/Meson/Custom]
        COOK_CONFIG[CookConfig<br/>CookbookConfig]
    end

    subgraph "Infrastructure Layer"
        FS_OPS[File Operations<br/>create_dir, remove_all, rename]
        CMD_EXEC[Command Execution<br/>run_command, run_command_stdin]
        PTY_CTRL[PTY Control<br/>setup_pty, flush_pty]
        HASH_CALC[Hash Calculation<br/>blake3]
    end

    TUI --> COOK_CMD
    CLI_PARSE --> COOK_CMD
    CLI_PARSE --> FETCH_CMD
    CLI_PARSE --> CLEAN_CMD
    CLI_PARSE --> PUSH_CMD
    CLI_PARSE --> TREE_CMD

    COOK_CMD --> RECIPE_MODEL
    COOK_CMD --> COOK_RECIPE
    FETCH_CMD --> RECIPE_MODEL
    FETCH_CMD --> BUILD_KIND

    RECIPE_MODEL --> FS_OPS
    COOK_RECIPE --> CMD_EXEC
    BUILD_KIND --> CMD_EXEC
    COOK_CONFIG --> FS_OPS

    CMD_EXEC --> PTY_CTRL
```

## ビルドパイプライン図

```mermaid
sequenceDiagram
    participant User
    participant CLI as repo CLI
    participant Fetch as fetch.rs
    participant Build as cook_build.rs
    participant Package as package.rs
    participant RepoBuilder as repo_builder

    User->>CLI: repo cook <recipes>
    CLI->>CLI: parse_args()
    CLI->>CLI: get_build_deps_recursive()

    loop for each recipe
        CLI->>Fetch: handle_fetch()
        alt Git Source
            Fetch->>Fetch: git clone/fetch
        else Tar Source
            Fetch->>Fetch: wget + tar extract
        else Path Source
            Fetch->>Fetch: copy_dir_all
        else Remote Build
            Fetch->>Fetch: download pkgar
        end
        Fetch-->>CLI: source_dir

        CLI->>Build: handle_cook()
        Build->>Build: build_deps_dir (sysroot)
        Build->>Build: build_deps_dir (toolchain)

        alt Cargo Template
            Build->>Build: cookbook_cargo
        else Configure Template
            Build->>Build: cookbook_configure
        else CMake Template
            Build->>Build: cookbook_cmake
        else Meson Template
            Build->>Build: cookbook_meson
        else Custom Template
            Build->>Build: custom script
        end

        Build-->>CLI: stage_dirs, auto_deps

        CLI->>Package: package()
        Package->>Package: pkgar::create()
        Package->>Package: package_toml()
        Package-->>CLI: success
    end

    CLI->>RepoBuilder: publish_packages()
    RepoBuilder->>RepoBuilder: copy pkgar/toml to repo
    RepoBuilder->>RepoBuilder: generate repo.toml
    RepoBuilder-->>User: done
```

## データフロー図

```mermaid
flowchart LR
    subgraph "Input"
        RECIPE_TOML[recipe.toml]
        SOURCE[Source Code<br/>Git/Tar/Local]
        DEPS[Dependencies<br/>pkgar files]
    end

    subgraph "Processing"
        PARSE[Parse Recipe]
        FETCH_SRC[Fetch Source]
        BUILD_SYSROOT[Build Sysroot]
        COMPILE[Compile]
        STAGE[Stage Files]
    end

    subgraph "Output"
        STAGE_DIR[stage/<br/>Staged Files]
        PKGAR[.pkgar<br/>Package Archive]
        TOML[.toml<br/>Package Metadata]
        REPO_TOML[repo.toml<br/>Repository Index]
    end

    RECIPE_TOML --> PARSE
    PARSE --> FETCH_SRC
    SOURCE --> FETCH_SRC
    FETCH_SRC --> BUILD_SYSROOT
    DEPS --> BUILD_SYSROOT
    BUILD_SYSROOT --> COMPILE
    COMPILE --> STAGE
    STAGE --> STAGE_DIR
    STAGE_DIR --> PKGAR
    STAGE_DIR --> TOML
    PKGAR --> REPO_TOML
    TOML --> REPO_TOML
```

## TUIモード スレッド構成図

```mermaid
graph TB
    subgraph "Main Thread"
        TUI_RENDER[TUI Render Loop]
        INPUT_HANDLER[Input Handler]
        STATUS_RECEIVER[Status Receiver]
    end

    subgraph "Fetcher Thread"
        FETCH_LOOP[Fetch Loop]
        GIT_OPS[Git Operations]
        TAR_OPS[Tar Operations]
    end

    subgraph "Cooker Thread"
        COOK_LOOP[Cook Loop]
        BUILD_OPS[Build Operations]
        PACKAGE_OPS[Package Operations]
    end

    subgraph "Input Thread"
        KEY_EVENTS[Keyboard Events]
    end

    subgraph "Channels"
        STATUS_TX[status_tx<br/>mpsc::Sender]
        WORK_TX[work_tx<br/>mpsc::Sender]
        INPUT_TX[input_tx<br/>mpsc::Sender]
    end

    KEY_EVENTS -->|Event| INPUT_TX
    INPUT_TX -->|Event| INPUT_HANDLER

    FETCH_LOOP -->|StatusUpdate| STATUS_TX
    COOK_LOOP -->|StatusUpdate| STATUS_TX
    STATUS_TX -->|StatusUpdate| STATUS_RECEIVER

    FETCH_LOOP -->|CookRecipe, PathBuf| WORK_TX
    WORK_TX -->|CookRecipe, PathBuf| COOK_LOOP

    STATUS_RECEIVER --> TUI_RENDER
```

## ディレクトリ構成図

```mermaid
graph TD
    subgraph "Project Root"
        ROOT[redox-master/]
    end

    subgraph "Source Code"
        SRC[src/]
        SRC_LIB[lib.rs]
        SRC_CONFIG[config.rs]
        SRC_RECIPE[recipe.rs]
        SRC_BLAKE3[blake3.rs]

        SRC_COOK[cook/]
        COOK_BUILD[cook_build.rs]
        COOK_FETCH[fetch.rs]
        COOK_PKG[package.rs]
        COOK_FS[fs.rs]
        COOK_SCRIPT[script.rs]
        COOK_PTY[pty.rs]
        COOK_IDENT[ident.rs]
        COOK_TREE[tree.rs]

        SRC_BIN[bin/]
        BIN_REPO[repo.rs]
        BIN_BUILDER[repo_builder.rs]
        BIN_REDOXER[cookbook_redoxer.rs]
    end

    subgraph "Recipe Storage"
        RECIPES[recipes/]
        RECIPES_CORE[core/]
        RECIPES_LIBS[libs/]
        RECIPES_TOOLS[tools/]
        RECIPES_DEV[dev/]
    end

    subgraph "Build Output"
        BUILD[build/]
        BUILD_KEYS[id_ed25519.toml<br/>id_ed25519.pub.toml]
        BUILD_LOGS[logs/]
    end

    subgraph "Repository"
        REPO[repo/]
        REPO_ARCH[x86_64-unknown-redox/]
        REPO_PKGAR[*.pkgar]
        REPO_INDEX[repo.toml]
    end

    ROOT --> SRC
    ROOT --> RECIPES
    ROOT --> BUILD
    ROOT --> REPO

    SRC --> SRC_LIB
    SRC --> SRC_CONFIG
    SRC --> SRC_RECIPE
    SRC --> SRC_BLAKE3
    SRC --> SRC_COOK
    SRC --> SRC_BIN

    SRC_COOK --> COOK_BUILD
    SRC_COOK --> COOK_FETCH
    SRC_COOK --> COOK_PKG
    SRC_COOK --> COOK_FS
    SRC_COOK --> COOK_SCRIPT
    SRC_COOK --> COOK_PTY
    SRC_COOK --> COOK_IDENT
    SRC_COOK --> COOK_TREE

    SRC_BIN --> BIN_REPO
    SRC_BIN --> BIN_BUILDER
    SRC_BIN --> BIN_REDOXER

    RECIPES --> RECIPES_CORE
    RECIPES --> RECIPES_LIBS
    RECIPES --> RECIPES_TOOLS
    RECIPES --> RECIPES_DEV

    BUILD --> BUILD_KEYS
    BUILD --> BUILD_LOGS

    REPO --> REPO_ARCH
    REPO_ARCH --> REPO_PKGAR
    REPO_ARCH --> REPO_INDEX
```

## 依存関係解決図

```mermaid
flowchart TD
    subgraph "Recipe Resolution"
        INPUT_NAMES[Input Package Names]
        GET_BUILD_DEPS[get_build_deps_recursive]
        GET_PKG_DEPS[get_package_deps_recursive]
        RECIPE_LIST[Resolved Recipe List]
    end

    subgraph "Dependency Types"
        BUILD_DEPS[build.dependencies<br/>Compile-time deps]
        DEV_DEPS[build.dev-dependencies<br/>Development deps]
        PKG_DEPS[package.dependencies<br/>Runtime deps]
        AUTO_DEPS[auto_deps<br/>Dynamic linking deps]
    end

    subgraph "Resolution Process"
        NEW_RECURSIVE[new_recursive]
        FLATTEN[recipes_flatten_package_names]
        MARK_DEPS[recipes_mark_as_deps]
    end

    INPUT_NAMES --> GET_BUILD_DEPS
    GET_BUILD_DEPS --> NEW_RECURSIVE
    NEW_RECURSIVE --> BUILD_DEPS
    NEW_RECURSIVE --> DEV_DEPS
    BUILD_DEPS --> NEW_RECURSIVE
    DEV_DEPS --> NEW_RECURSIVE

    INPUT_NAMES --> GET_PKG_DEPS
    GET_PKG_DEPS --> PKG_DEPS
    PKG_DEPS --> GET_PKG_DEPS

    NEW_RECURSIVE --> FLATTEN
    FLATTEN --> MARK_DEPS
    MARK_DEPS --> RECIPE_LIST

    AUTO_DEPS --> |Discovered at build time| RECIPE_LIST
```
