# アーキテクチャ設計書

## 概要

本ドキュメントは、Redox OS Build System（redox_cookbook）のアーキテクチャ設計について記載する。本システムはRedox OSのビルドシステムであり、レシピ（パッケージ定義）に基づいてソースコードの取得、ビルド、パッケージングを行うCLIツールである。Rust言語で実装されており、モジュラーアーキテクチャを採用している。

## システム全体構成

### アーキテクチャ概要図

詳細なアーキテクチャ構成図は [アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| Git リポジトリ（GitLab等） | HTTPS/SSH | ソースコードの取得（clone/fetch） |
| HTTP/FTPサーバー | HTTP/HTTPS | tarballソースの取得 |
| Cargo レジストリ（crates.io） | HTTPS | Rustパッケージ依存関係の解決 |
| pkgar リポジトリ | HTTPS | リモートパッケージのダウンロード |
| ホストOS（Linux/Redox） | システムコール | ビルドツールチェーンの実行 |

## レイヤー構成

### アーキテクチャスタイル

モジュラーアーキテクチャ / レイヤードアーキテクチャ

本システムは明確なレイヤー分離を持つモジュラー設計を採用している。CLIアプリケーションとしての特性上、Presentation層は薄く、主にApplication層とDomain層で構成される。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Presentation | CLI入力解析、TUI表示、ユーザーインタラクション | `src/bin/repo.rs`（TUI、引数パース） |
| Application | ビルドワークフロー制御、コマンド実行 | `src/cook/cook_build.rs`, `src/cook/fetch.rs`, `src/cook/package.rs` |
| Domain | レシピ定義、パッケージモデル、設定管理 | `src/recipe.rs`, `src/config.rs` |
| Infrastructure | ファイルシステム操作、外部コマンド実行、PTY | `src/cook/fs.rs`, `src/cook/script.rs`, `src/cook/pty.rs` |

### レイヤー間の依存関係ルール

- Presentation層はApplication層とDomain層に依存する
- Application層はDomain層とInfrastructure層に依存する
- Domain層は他のレイヤーに依存しない（純粋なデータモデル）
- Infrastructure層はDomain層に依存する
- 循環依存は禁止されている

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| レシピ管理 | パッケージビルド定義の解析・管理 | `recipe.rs` |
| 設定管理 | グローバル設定、ミラー設定 | `config.rs` |
| ソース取得 | Git/Tar/Pathからのソース取得 | `cook/fetch.rs` |
| ビルド制御 | ビルドプロセス、依存関係解決 | `cook/cook_build.rs` |
| パッケージング | pkgarアーカイブ生成、メタデータ作成 | `cook/package.rs` |
| リポジトリ管理 | パッケージリポジトリの生成・更新 | `bin/repo_builder.rs` |

### パッケージ構造

```
src/
├── lib.rs              # ライブラリエントリポイント、定数定義
├── config.rs           # グローバル設定、ミラー設定
├── recipe.rs           # レシピ定義（SourceRecipe, BuildKind, Recipe）
├── blake3.rs           # ハッシュ計算ユーティリティ
├── progress_bar.rs     # プログレスバー表示
├── cook/               # ビルド関連モジュール群
│   ├── cook_build.rs   # ビルドロジック、依存解決
│   ├── fetch.rs        # ソースコード取得（Git/Tar/Path）
│   ├── package.rs      # パッケージング、pkgar生成
│   ├── fs.rs           # ファイルシステム操作ユーティリティ
│   ├── script.rs       # ビルドスクリプトテンプレート
│   ├── pty.rs          # 疑似端末制御
│   ├── ident.rs        # ビルド識別子管理
│   └── tree.rs         # 依存関係ツリー表示
└── bin/
    ├── repo.rs         # メインCLI（fetch/cook/clean等）
    ├── repo_builder.rs # リポジトリ生成ツール
    └── cookbook_redoxer.rs # Redoxer統合ラッパー
```

### コンポーネント依存関係

主要コンポーネント間の依存関係:

1. **repo.rs（CLI）** -> config, cook/*, recipe
2. **cook_build.rs** -> recipe, config, cook/fs, cook/script, cook/package
3. **fetch.rs** -> recipe, config, cook/fs, cook/script
4. **package.rs** -> recipe, cook/fs, blake3
5. **recipe.rs** -> pkg（外部クレート）

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| ファイルベース | TOML | - | レシピ定義、設定ファイル、パッケージメタデータ |

※ 本システムは従来のRDBMSを使用せず、ファイルシステムベースでデータを管理する。

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| ファイルシステム | - | ビルドキャッシュ（sysroot, toolchain） | ソース変更まで |
| ファイルシステム | - | パッケージキャッシュ（stage, pkgar） | ソース変更まで |

キャッシュ無効化ロジック:
- ソースファイルの更新時刻がキャッシュより新しい場合に再ビルド
- 依存パッケージが更新された場合に再ビルド
- `COOKBOOK_CLEAN_BUILD`/`COOKBOOK_CLEAN_TARGET`環境変数による明示的クリア

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| mpsc（Rust標準） | - | TUIスレッド間通信（StatusUpdate） |

### 検索エンジン

該当なし（ファイルシステムベースのパターンマッチング `globset` を使用）

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| pkgar | Git依存 | パッケージアーカイブ形式 |
| pkgar-keys | Git依存 | 署名鍵管理 |
| redoxer | Git依存 | クロスコンパイル環境 |
| ratatui | 0.29.0 | TUI（Terminal User Interface）描画 |
| termion | 4 | ターミナル制御 |

## データフロー

### リクエスト処理の流れ

**cookコマンド実行時の処理フロー:**

1. `repo.rs::main()` でCLI引数をパース
2. `parse_args()` でレシピ一覧と設定を構築
3. `CookRecipe::get_build_deps_recursive()` で依存関係を再帰的に解決
4. 各レシピに対して以下を順次実行:
   - `handle_fetch()`: ソースコード取得
   - `handle_cook()`: ビルド実行
5. `publish_packages()` でリポジトリに公開

### 非同期処理の流れ

**TUIモード時の並列処理フロー:**

1. メインスレッド: TUI描画、入力処理
2. Fetcherスレッド: ソースコード取得（1並列）
3. Cookerスレッド: ビルド実行（1並列）
4. 各スレッドは `mpsc::channel` でStatusUpdateを送信
5. メインスレッドがStatusUpdateを受信しTUI更新

```
[Main Thread]  <---mpsc--- [Fetcher Thread]
      |                          |
      |                    mpsc::channel(work_tx)
      |                          |
      +------mpsc------> [Cooker Thread]
```

### データ永続化の流れ

1. **レシピ読み込み**: `recipes/<category>/<name>/recipe.toml` -> `Recipe`構造体
2. **ソースキャッシュ**: 取得したソースを `recipes/<name>/source/` に保存
3. **ビルド成果物**: `recipes/<name>/target/<arch>/stage/` に出力
4. **パッケージ**: `recipes/<name>/target/<arch>/stage.pkgar` として保存
5. **リポジトリ公開**: `repo/<arch>/<name>.pkgar` と `repo/<arch>/<name>.toml` にコピー

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| Ed25519署名 | `cook/package.rs`, pkgar-keys | パッケージの署名・検証 |
| SSH認証 | 外部Git | プライベートリポジトリへのアクセス |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| ビルドログ | PTY出力キャプチャ | `build/logs/<arch>/<name>.log` |
| 進捗表示 | TUI/標準出力 | コンソール（リアルタイム） |
| デバッグログ | `log_to_pty!`マクロ | 同上（COOKBOOK_VERBOSE=true時） |

### エラーハンドリング

エラーハンドリングの方針: `anyhow::Result` による統一的なエラー伝播

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| レシピ解析エラー | `PackageError` | 処理中止、エラーメッセージ表示 |
| ソース取得エラー | `anyhow::Error` | リトライプロンプト（TUI）または中止 |
| ビルドエラー | `anyhow::Error` | リトライ/スキップ/終了の選択（TUI） |
| 依存解決エラー | `PackageError::Recursion` | 循環依存検出、処理中止 |

`COOKBOOK_NONSTOP=true` 設定時はエラー発生時もスキップして継続する。

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| ソース取得 | アトミック移動（source.tmp -> source） | ファイルレベル |
| ビルド成果物 | アトミック移動（stage.tmp -> stage） | ファイルレベル |
| 依存展開 | アトミック移動（sysroot.tmp -> sysroot） | ファイルレベル |

一時ディレクトリで処理を行い、成功時に `rename()` でアトミックに移動することで、部分的な状態を防止している。

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 単一責任原則（SRP） | モジュール分割 | fetch.rs（取得）、cook_build.rs（ビルド）、package.rs（パッケージング）の分離 |
| 依存性逆転原則（DIP） | Recipe構造体 | レシピ定義がビルド種別（Cargo/Configure/CMake等）を抽象化 |
| Open/Closed原則 | BuildKind enum | 新しいビルドテンプレート追加時にenumバリアントを追加するだけで拡張可能 |
| 不変性の優先 | CookConfig | 設定は初期化後に変更されない（OnceLock使用） |

### コーディング規約

- Rust Edition 2024使用
- `anyhow::Result` による統一的なエラー処理
- `serde` によるTOML/JSON シリアライズ
- `#[derive(Debug, Clone, Deserialize, Serialize)]` の一貫した使用
- `snake_case` による命名（Rust標準）
- マクロ `log_to_pty!` によるロギング統一
- パスは `PathBuf` / `&Path` を使用（文字列操作を避ける）

## 備考

- 本システムはRedox OS専用のビルドシステムであるが、Linuxホスト上でも動作する
- クロスコンパイル機能は `redoxer` クレートに依存している
- パッケージ形式は `pkgar` 独自フォーマットを使用
- ビルドスクリプトはシェルスクリプト（bash）として実行される
- TUI機能は `tui` feature flagで制御可能（CI環境では自動無効化）
