# Redox OS Build System コードリーディングガイドライン

## はじめに

このガイドラインは、Redox OSビルドシステム（Cookbook）のコードベースを効率的に理解するための手引きです。
Rust言語に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Rust言語の基本構文と概念を解説します。

### 1.1 プログラム構造

Rustプログラムは、モジュール（`mod`）とクレート（パッケージ）で構成されます。エントリーポイントは`main()`関数です。

```rust
// ファイル: src/bin/repo.rs:174-180
fn main() {
    init_config();
    if let Err(e) = main_inner() {
        eprintln!("{:?}", e);
        process::exit(1);
    };
}
```

**解説:**
- `fn main()`: プログラムのエントリーポイント
- `init_config()`: 設定の初期化
- `if let Err(e)`: エラーハンドリングのパターンマッチ
- `eprintln!`: 標準エラー出力へのマクロ

### 1.2 データ型と変数

Rustは静的型付け言語で、`let`で変数を宣言し、`mut`で可変性を指定します。

```rust
// ファイル: src/config.rs:29-39
#[derive(Debug, Default, Clone, Deserialize, PartialEq, Serialize)]
pub struct CookConfig {
    pub offline: bool,
    pub jobs: usize,
    pub tui: bool,
    pub logs: bool,
    pub nonstop: bool,
    pub verbose: bool,
    pub clean_build: bool,
    pub clean_target: bool,
}
```

**解説:**
- `#[derive(...)]`: コンパイラによる自動実装（トレイトの導出）
- `pub struct`: 公開構造体の定義
- `bool`, `usize`: プリミティブ型

### 1.3 制御構造

条件分岐、ループ、パターンマッチを使用します。

```rust
// ファイル: src/bin/repo.rs:291-354
Ok(match *command {
    CliCommand::Fetch | CliCommand::Cook => {
        // 処理
    }
    CliCommand::Unfetch => handle_clean(recipe, config, true, true)?,
    CliCommand::Clean => handle_clean(recipe, config, false, true)?,
    CliCommand::Push => unreachable!(),
    CliCommand::Tree => unreachable!(),
    CliCommand::Find => println!("{}", recipe.dir.display()),
})
```

**解説:**
- `match`: パターンマッチング（網羅的なswitch文）
- `|`: 複数パターンのOR結合
- `?`: エラー伝播演算子
- `unreachable!()`: 到達不可能コードを示すマクロ

### 1.4 関数/メソッド定義

関数は`fn`キーワードで定義し、`->`で戻り値型を指定します。

```rust
// ファイル: src/recipe.rs:206-216
impl Recipe {
    pub fn new(file: &PathBuf) -> Result<Recipe, PackageError> {
        if !file.is_file() {
            return Err(PackageError::FileMissing(file.clone()));
        }
        let toml = fs::read_to_string(&file)
            .map_err(|err| PackageError::Parse(DeError::custom(err), Some(file.clone())))?;
        let recipe: Recipe = toml::from_str(&toml)
            .map_err(|err| PackageError::Parse(DeError::custom(err), Some(file.clone())))?;
        Ok(recipe)
    }
}
```

**解説:**
- `impl Recipe`: Recipe構造体のメソッド実装ブロック
- `&PathBuf`: 参照渡し（所有権を移動しない）
- `Result<T, E>`: 成功または失敗を表す列挙型
- `map_err`: エラー型の変換

### 1.5 モジュール/インポート

モジュールシステムでコードを整理し、`use`でインポートします。

```rust
// ファイル: src/lib.rs:1-6
pub mod blake3;
pub mod config;
pub mod cook;
pub mod recipe;

mod progress_bar;
```

```rust
// ファイル: src/cook.rs:1-9
// avoid confusion with build.rs
pub mod cook_build;
pub mod fetch;
pub mod fs;
pub mod ident;
pub mod package;
pub mod pty;
pub mod script;
pub mod tree;
```

**解説:**
- `pub mod`: 公開モジュールの宣言
- `mod`: 非公開モジュールの宣言
- モジュールは同名のファイルまたはディレクトリに対応

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

このプロジェクトは、Redox OSのビルドシステム「Cookbook」です。主要な概念：

**Recipe（レシピ）:**
パッケージのビルド方法を定義するTOML設定ファイル。

```rust
// ファイル: src/recipe.rs:162-174
#[derive(Debug, Clone, Default, Deserialize, PartialEq, Serialize)]
#[serde(default)]
pub struct Recipe {
    /// Specifies how to download the source for this recipe
    pub source: Option<SourceRecipe>,
    /// Specifies how to build this recipe
    pub build: BuildRecipe,
    /// Specifies how to package this recipe
    pub package: PackageRecipe,
    /// Specifies optional packages based from this recipe
    #[serde(rename = "optional-packages")]
    pub optional_packages: Vec<OptionalPackageRecipe>,
}
```

**BuildKind（ビルド種別）:**
ビルドテンプレートの種類を定義。

```rust
// ファイル: src/recipe.rs:89-128
#[derive(Debug, Clone, Deserialize, PartialEq, Serialize)]
#[serde(tag = "template")]
pub enum BuildKind {
    #[serde(rename = "none")]
    None,
    #[serde(rename = "remote")]
    Remote,
    #[serde(rename = "cargo")]
    Cargo { ... },
    #[serde(rename = "configure")]
    Configure { ... },
    #[serde(rename = "cmake")]
    Cmake { ... },
    #[serde(rename = "meson")]
    Meson { ... },
    #[serde(rename = "custom")]
    Custom { script: String },
}
```

### 2.2 プロジェクト独自のパターン

**PtyOut（疑似端末出力）:**
ビルドログの出力管理パターン。

```rust
// ファイル: src/cook/pty.rs で定義
pub type PtyOut<'a> = Option<(&'a mut UnixSlavePty, &'a mut std::io::PipeWriter)>;
```

**log_to_pty!マクロ:**
条件付きログ出力マクロ。

```rust
// 使用例: src/cook/cook_build.rs:101
log_to_pty!(logger, "DEBUG: {} needs {}", relative_path.display(), name);
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `*.rs` | Rustソースファイル | `config.rs`, `recipe.rs` |
| `*.toml` | TOML設定ファイル | `recipe.toml`, `Cargo.toml` |
| `recipe.toml` | パッケージビルド定義 | `recipes/core/kernel/recipe.toml` |
| `recipes/<category>/<name>/` | レシピディレクトリ | `recipes/core/kernel/` |
| `src/bin/` | バイナリエントリーポイント | `src/bin/repo.rs` |
| `src/cook/` | cook機能モジュール群 | `src/cook/fetch.rs` |
| `mk/` | Makefileフラグメント | `mk/config.mk` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Cli*` | CLIコマンド関連 | `CliConfig`, `CliCommand` |
| `*Recipe` | レシピ関連データ構造 | `CookRecipe`, `SourceRecipe` |
| `*Config` | 設定関連 | `CookConfig`, `CookbookConfig` |
| `handle_*` | イベント/コマンドハンドラ | `handle_fetch`, `handle_cook` |
| `fetch_*` | ソース取得関連 | `fetch_offline`, `fetch_remote` |
| `build_*` | ビルド処理関連 | `build`, `build_remote` |
| `*_dir` | ディレクトリパス変数 | `source_dir`, `target_dir` |
| `*_path` | ファイルパス変数 | `archive_path`, `log_path` |

### 3.3 プログラム分類一覧

| バイナリ名 | 役割 |
|-----------|------|
| `repo` | メインCLIツール（fetch/cook/clean等） |
| `repo_builder` | パッケージリポジトリ構築ツール |
| `cookbook_redoxer` | Redoxクロスコンパイル環境ラッパー |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
redox-master/
├── Cargo.toml          # Rustパッケージマニフェスト
├── Cargo.lock          # 依存関係ロックファイル
├── Makefile            # ビルドシステムエントリーポイント
├── build.sh            # ビルドスクリプト
├── src/                # Rustソースコード
│   ├── lib.rs          # ライブラリルート
│   ├── config.rs       # 設定管理
│   ├── recipe.rs       # レシピデータ構造
│   ├── cook.rs         # cookモジュールルート
│   ├── cook/           # cook機能サブモジュール
│   │   ├── cook_build.rs   # ビルド処理
│   │   ├── fetch.rs        # ソース取得
│   │   ├── fs.rs           # ファイルシステム操作
│   │   ├── package.rs      # パッケージング
│   │   ├── script.rs       # スクリプト実行
│   │   └── ...
│   └── bin/            # バイナリエントリーポイント
│       ├── repo.rs         # メインCLI
│       ├── repo_builder.rs # リポジトリビルダー
│       └── cookbook_redoxer.rs
├── recipes/            # パッケージレシピ群
│   ├── core/           # コアシステムコンポーネント
│   ├── libs/           # ライブラリ
│   ├── dev/            # 開発ツール
│   ├── games/          # ゲーム
│   └── ...
├── config/             # ビルド設定プロファイル
│   ├── x86_64/         # x86_64アーキテクチャ設定
│   ├── aarch64/        # ARM64アーキテクチャ設定
│   └── *.toml          # 設定プロファイル
├── mk/                 # Makefileフラグメント
├── scripts/            # ユーティリティスクリプト
└── podman/             # Podmanコンテナ設定
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | Rustソースコード | `lib.rs`, `recipe.rs`, `config.rs` |
| `src/bin/` | CLIバイナリ | `repo.rs`, `repo_builder.rs` |
| `src/cook/` | ビルド・パッケージング機能 | `cook_build.rs`, `fetch.rs` |
| `recipes/` | パッケージレシピ定義 | `recipe.toml` |
| `config/` | ビルドプロファイル | `desktop.toml`, `server.toml` |
| `mk/` | Makefileモジュール | `config.mk`, `repo.mk` |
| `scripts/` | 補助スクリプト | `*.sh` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

このプロジェクトは**モジュラーCLIアーキテクチャ**を採用しています。

```
┌─────────────────────────────────────────────────────────────┐
│                        CLI Layer                            │
│  ┌─────────────┐  ┌──────────────┐  ┌───────────────────┐  │
│  │    repo     │  │ repo_builder │  │ cookbook_redoxer  │  │
│  └──────┬──────┘  └──────┬───────┘  └─────────┬─────────┘  │
└─────────┼────────────────┼────────────────────┼────────────┘
          │                │                    │
┌─────────▼────────────────▼────────────────────▼────────────┐
│                     Library Layer (cookbook)               │
│  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌───────────┐  │
│  │  config  │  │  recipe  │  │   cook   │  │  blake3   │  │
│  └──────────┘  └──────────┘  └────┬─────┘  └───────────┘  │
└───────────────────────────────────┼────────────────────────┘
                                    │
┌───────────────────────────────────▼────────────────────────┐
│                     Cook Submodules                        │
│  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌───────────┐  │
│  │  fetch   │  │cook_build│  │ package  │  │   pty     │  │
│  └──────────┘  └──────────┘  └──────────┘  └───────────┘  │
│  ┌──────────┐  ┌──────────┐  ┌──────────┐                 │
│  │    fs    │  │  script  │  │   tree   │                 │
│  └──────────┘  └──────────┘  └──────────┘                 │
└────────────────────────────────────────────────────────────┘
          │
┌─────────▼──────────────────────────────────────────────────┐
│                    External Layer                          │
│  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌───────────┐  │
│  │  pkgar   │  │ redoxer  │  │installer │  │  pkgutils │  │
│  └──────────┘  └──────────┘  └──────────┘  └───────────┘  │
└────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| CLI Layer | ユーザーインターフェース、コマンド解析 | `src/bin/repo.rs` |
| Library Layer | ビジネスロジック、データ構造 | `src/recipe.rs`, `src/config.rs` |
| Cook Submodules | ビルド・パッケージング処理 | `src/cook/*.rs` |
| External Layer | 外部クレート依存 | pkgar, redoxer, installer |

### 5.3 データフロー

```
[recipe.toml] → [Recipe構造体] → [Fetch処理] → [Build処理] → [Package処理] → [.pkgar]
                     │
                     ▼
              [CookConfig]
                (設定情報)
```

1. `recipe.toml`がパース され`Recipe`構造体に変換
2. `CookConfig`でビルド設定を適用
3. `fetch`でソースコードを取得
4. `cook_build`でビルド実行
5. `package`でpkgarアーカイブを生成

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

メインCLIツール`repo`のエントリーポイント:

```rust
// ファイル: src/bin/repo.rs:174-180
fn main() {
    init_config();
    if let Err(e) = main_inner() {
        eprintln!("{:?}", e);
        process::exit(1);
    };
}
```

コマンド処理の流れ:
```rust
// ファイル: src/bin/repo.rs:182-249
fn main_inner() -> anyhow::Result<()> {
    let args: Vec<String> = env::args().skip(1).collect();
    // ...
    let (config, command, recipe_names) = parse_args(args)?;
    // ...
}
```

### 6.2 ビジネスロジック

**レシピ処理 (`src/recipe.rs`):**

```rust
// ファイル: src/recipe.rs:227-260
impl CookRecipe {
    pub fn new(name: PackageName, dir: PathBuf, mut recipe: Recipe) -> Result<Self, PackageError> {
        let target = cook_package::package_target(&name);
        // ...
        Ok(Self {
            name,
            dir,
            recipe,
            target,
            is_deps: false,
            rule: "".into(),
        })
    }
}
```

**ビルド処理 (`src/cook/cook_build.rs`):**

```rust
// ファイル: src/cook/cook_build.rs:170-179
pub fn build(
    recipe_dir: &Path,
    source_dir: &Path,
    target_dir: &Path,
    name: &PackageName,
    recipe: &Recipe,
    cook_config: &CookConfig,
    check_source: bool,
    logger: &PtyOut,
) -> Result<(Vec<PathBuf>, BTreeSet<PackageName>), String> {
    // ...
}
```

### 6.3 データアクセス

**設定ファイル読み込み (`src/config.rs`):**

```rust
// ファイル: src/config.rs:68-78
pub fn init_config() {
    let mut config: CookbookConfig = if fs::exists("cookbook.toml").unwrap_or(false) {
        let toml_content = fs::read_to_string("cookbook.toml")
            .map_err(|e| format!("Unable to read config: {:?}", e))
            .unwrap();
        toml::from_str(&toml_content)
            .map_err(|e| format!("Unable to parse config: {:?}", e))
            .unwrap()
    } else {
        CookbookConfig::default()
    };
    // ...
}
```

### 6.4 ユーティリティ/共通機能

**ファイルシステム操作 (`src/cook/fs.rs`):**
- ディレクトリ作成・削除
- ファイルコピー・移動
- シンボリックリンク作成
- 更新時刻チェック

**スクリプト実行 (`src/cook/script.rs`):**
- シェルスクリプト実行
- ビルドテンプレート処理

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Result型によるエラー処理 | 成功/失敗を型で表現 | 高 | 全ファイル |
| Serde派生マクロ | TOML/JSON直列化 | 高 | `recipe.rs`, `config.rs` |
| ビルダーパターン | 段階的なオブジェクト構築 | 中 | `recipe.rs` |
| TUIアプリパターン | 対話的UI | 中 | `repo.rs` |

### 各パターンの詳細

#### パターン1: Result型によるエラー処理

**目的:** 型安全なエラーハンドリングを実現

**実装例:**
```rust
// ファイル: src/cook/fetch.rs:117-131
pub fn fetch(recipe: &CookRecipe, logger: &PtyOut) -> Result<PathBuf, String> {
    let recipe_dir = &recipe.dir;
    let source_dir = recipe_dir.join("source");
    match recipe.recipe.build.kind {
        BuildKind::None => {
            fetch_apply_source_info(recipe, "".to_string())?;
            return Ok(source_dir);
        }
        BuildKind::Remote => {
            fetch_remote(recipe_dir, recipe, false, logger)?;
            return Ok(source_dir);
        }
        _ => {}
    }
    // ...
}
```

**解説:** `Result<T, E>`を使用し、`?`演算子でエラーを伝播させる。成功時は`Ok(値)`、失敗時は`Err(エラー)`を返す。

#### パターン2: Serde派生マクロによるTOML解析

**目的:** 設定ファイルの型安全な読み込み

**実装例:**
```rust
// ファイル: src/recipe.rs:136-144
#[derive(Debug, Clone, Default, Deserialize, PartialEq, Serialize)]
#[serde(default)]
pub struct BuildRecipe {
    #[serde(flatten)]
    pub kind: BuildKind,
    pub dependencies: Vec<PackageName>,
    #[serde(rename = "dev-dependencies")]
    pub dev_dependencies: Vec<PackageName>,
}
```

**解説:** `#[derive(Deserialize, Serialize)]`でTOML/JSONの自動変換を実装。`#[serde(rename = "...")]`でフィールド名をカスタマイズ。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: パッケージビルドフロー (`repo cook <recipe>`)

**概要:** レシピ名を指定してパッケージをビルドするフロー

**処理フロー:**
```
main() → main_inner() → parse_args() → repo_inner() → handle_fetch() → handle_cook() → package()
```

**詳細な追跡:**

1. **エントリーポイント** (`src/bin/repo.rs:174-180`)
   ```rust
   fn main() {
       init_config();
       if let Err(e) = main_inner() {
           eprintln!("{:?}", e);
           process::exit(1);
       };
   }
   ```

2. **引数解析** (`src/bin/repo.rs:373-551`)
   ```rust
   fn parse_args(args: Vec<String>) -> anyhow::Result<(CliConfig, CliCommand, Vec<CookRecipe>)> {
       let mut config = CliConfig::new()?;
       let mut command: Option<String> = None;
       let mut recipe_names: Vec<PackageName> = Vec::new();
       // ...
   }
   ```

3. **フェッチ処理** (`src/bin/repo.rs:553-566`)
   ```rust
   fn handle_fetch(
       recipe: &CookRecipe,
       config: &CliConfig,
       allow_offline: bool,
       logger: &PtyOut,
   ) -> anyhow::Result<PathBuf> {
       let source_dir = match config.cook.offline && allow_offline {
           true => fetch_offline(&recipe, logger),
           false => fetch(&recipe, logger),
       }
       .map_err(|e| anyhow!("failed to fetch: {:?}", e))?;
       Ok(source_dir)
   }
   ```

4. **ビルド処理** (`src/bin/repo.rs:568-606`)
   ```rust
   fn handle_cook(
       recipe: &CookRecipe,
       config: &CliConfig,
       source_dir: PathBuf,
       is_deps: bool,
       logger: &PtyOut,
   ) -> anyhow::Result<()> {
       // ...
       let (stage_dirs, auto_deps) = build(
           recipe_dir,
           &source_dir,
           &target_dir,
           &recipe.name,
           &recipe.recipe,
           &config.cook,
           !is_deps,
           logger,
       )
       .map_err(|err| anyhow!("failed to build: {:?}", err))?;
       // ...
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `CONTRIBUTING.md` - 開発ガイドライン
3. `Cargo.toml` - 依存関係とプロジェクト構成
4. `src/lib.rs` - ライブラリ構造

#### 特定機能を理解したい場合
1. 関連する`src/cook/*.rs`モジュール
2. `src/recipe.rs` - データ構造定義
3. `recipes/`配下の実際のrecipe.toml例

#### 改修作業を行う場合
1. 対象機能の該当モジュール
2. 関連する設定（`src/config.rs`）
3. テストコード（各モジュール内の`#[cfg(test)]`セクション）

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要 | 最初に読む |
| CONTRIBUTING.md | 開発ガイドライン | 開発参加時 |
| HARDWARE.md | ハードウェア互換性 | 動作確認時 |
| Redox Book (外部) | 詳細ドキュメント | 深い理解が必要な時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `?`演算子が理解できない
A: `?`はResult/Optionのエラーを呼び出し元に伝播させる演算子です。`match`を使った明示的なエラー処理と等価です。
参照: [Rust Book - Error Handling](https://doc.rust-lang.org/book/ch09-02-recoverable-errors-with-result.html)

#### Q: `#[derive(...)]`の意味は？
A: コンパイラに自動実装を依頼するアトリビュートです。`Debug`はデバッグ出力、`Clone`はコピー可能、`Deserialize`/`Serialize`はシリアライズ機能を自動実装します。

#### Q: `mod`と`use`の違いは？
A: `mod`はモジュールを宣言/定義し、`use`は既存のアイテムをスコープにインポートします。

### デバッグのヒント

1. **COOKBOOK_VERBOSE=true**を設定すると詳細ログが出力される
2. **--nonstop**フラグでエラー時も継続実行できる
3. `build/logs/`ディレクトリにビルドログが保存される
4. `eprintln!`や`log_to_pty!`を追加してデバッグ出力を確認

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Recipe | パッケージのビルド方法を定義するTOMLファイル |
| Cookbook | Redox OSのビルドシステム名称 |
| pkgar | Redox OSのパッケージアーカイブ形式 |
| sysroot | ビルド時の依存ライブラリを配置するディレクトリ |
| stage | ビルド成果物を配置する一時ディレクトリ |
| fetch | ソースコードをダウンロードする処理 |
| cook | ソースをビルドしてパッケージを作成する処理 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/lib.rs` | ライブラリルート | モジュール宣言、定数定義 |
| `src/config.rs` | 設定管理 | CookConfig、初期化処理 |
| `src/recipe.rs` | レシピ構造 | Recipe、CookRecipe、BuildKind |
| `src/cook/cook_build.rs` | ビルド処理 | build()、依存解決 |
| `src/cook/fetch.rs` | ソース取得 | fetch()、git/tar処理 |
| `src/cook/package.rs` | パッケージング | package()、pkgar生成 |
| `src/bin/repo.rs` | メインCLI | コマンド解析、TUI |
| `src/bin/repo_builder.rs` | リポジトリ構築 | パッケージ公開処理 |

### C. 参考資料

- [The Rust Programming Language](https://doc.rust-lang.org/book/) - Rust公式ドキュメント
- [Redox Book](https://doc.redox-os.org/book/) - Redox OS公式ドキュメント
- [Serde](https://serde.rs/) - シリアライゼーションフレームワーク
- [anyhow](https://docs.rs/anyhow/) - エラーハンドリングクレート
