# セキュリティ設計書

## 概要

本ドキュメントは、Redox OS Build Systemのセキュリティ設計に関する技術文書です。本システムはRust言語で実装されたOSビルドシステムであり、パッケージのビルド、配布、検証に関するセキュリティ機能を提供します。

## 認証設計

### 認証方式

本システムはパッケージ署名ベースの認証を採用しています。

| 認証方式 | 用途 | 実装 |
| --- | --- | --- |
| Ed25519公開鍵暗号 | パッケージ署名・検証 | `pkgar-keys` クレート |
| 鍵ファイル認証 | ビルド時のパッケージ署名 | `id_ed25519.toml` / `id_ed25519.pub.toml` |

#### 鍵ペア管理

```
build/
├── id_ed25519.toml        # 秘密鍵（ローカルビルド用）
└── id_ed25519.pub.toml    # 公開鍵（パッケージ検証用）
```

鍵ペアは `pkgar_keys::SecretKeyFile::new()` により自動生成され、ビルドディレクトリに保存されます。

### セッション管理

本システムはOSビルドツールであり、従来のWebアプリケーション型セッション管理は採用していません。

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | N/A | ビルドプロセス単位で完結 |
| セッション固定化対策 | N/A | ステートレスな設計 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| ビルドシステム | パッケージ署名権限 | 秘密鍵を使用したパッケージへの署名 |
| インストーラ | パッケージ検証権限 | 公開鍵を使用した署名検証 |
| リモートサーバ | パッケージ配布権限 | 署名済みパッケージの配布 |

### アクセス制御

ファイルシステムベースのアクセス制御を採用しています。

- **秘密鍵へのアクセス**: ビルドプロセスのみがアクセス可能
- **公開鍵の配布**: リモートサーバから取得（`id_ed25519.pub.toml`）
- **パッケージアーカイブ**: 署名検証後のみ展開許可

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | リモートパッケージ取得時にHTTPS使用（`https://static.redox-os.org/pkg`） |
| ミラーURL変換 | HTTPからHTTPSへの自動アップグレードサポート |
| ソースダウンロード | wget経由での継続ダウンロードサポート（HTTPS優先） |

### 通信フロー

```
[ビルドシステム] --HTTPS--> [static.redox-os.org]
                              ├── パッケージ (.pkgar)
                              ├── メタデータ (.toml)
                              └── 公開鍵 (id_ed25519.pub.toml)
```

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| パッケージ署名 | Ed25519（pkgar-keys） |
| ハッシュ検証 | BLAKE3（256ビット） |
| 通信 | TLS（HTTPS経由） |

### ハッシュ検証の実装

```rust
// src/blake3.rs
pub fn blake3_progress<P: AsRef<Path>>(path: P) -> Result<String> {
    let hash = Hasher::new().update_reader(&mut pbr)?.finalize();
    format!("{}", hash.to_hex())
}
```

### 機密情報管理

| 機密情報 | 保管場所 | 保護方式 |
| --- | --- | --- |
| 署名秘密鍵 | `build/id_ed25519.toml` | ファイルシステム権限 |
| 公開鍵 | `build/id_ed25519.pub.toml` | 公開情報 |
| tarアーカイブのハッシュ | `recipe.toml` | BLAKE3で検証 |

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| 改ざんされたパッケージ | Ed25519署名による検証 |
| 不正なtarファイル | BLAKE3ハッシュによる整合性検証 |
| パス・トラバーサル | `pkgar::ext::EntryExt::check_path()` による検証 |
| シンボリックリンク攻撃 | ループ検出（深さ50までの制限） |
| ビルド環境汚染 | サンドボックス化されたsysroot環境 |

### パッケージ検証フロー

```
1. 公開鍵の取得（リモートまたはローカル）
2. パッケージファイル（.pkgar）のダウンロード
3. 署名の検証
4. 展開前のパス検証
5. 安全な展開
```

### tarファイル検証

```rust
// src/cook/fetch.rs
if let Some(blake3) = blake3 {
    if source_tar_blake3 != *blake3 {
        return Err(format!(
            "The downloaded tar blake3 '{source_tar_blake3}' is not equal to blake3 in recipe.toml."
        ));
    }
}
```

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| ビルドログ | ビルドプロセスの詳細 | `build/logs/`配下、設定による |
| ソース識別子 | コミットハッシュ、タイムスタンプ | `source_info.toml`に永続化 |
| 依存関係追跡 | 自動検出された依存関係 | `auto_deps.toml`に永続化 |

### ソース追跡機能

```rust
// src/cook/ident.rs
pub struct IdentifierConfig {
    pub commit: String,  // Gitコミットハッシュ
    pub time: String,    // ISO8601形式のタイムスタンプ
}
```

## 依存関係のセキュリティ

### 使用クレートのセキュリティ評価

| クレート | 用途 | セキュリティ特性 |
| --- | --- | --- |
| `blake3` | ハッシュ計算 | 高速・安全な暗号ハッシュ |
| `pkgar` | パッケージ管理 | Ed25519署名サポート |
| `pkgar-keys` | 鍵管理 | 安全な鍵生成・保存 |
| `serde` | シリアライズ | 型安全なデータ処理 |
| `toml` | 設定解析 | 安全な設定ファイル処理 |

### Rust言語によるメモリ安全性

- 所有権システムによるメモリ安全性の保証
- `unwrap()`の使用は限定的で、エラーハンドリングを優先
- `unsafe`コードの不使用（本コードベース内）

## 備考

### セキュリティ上の考慮事項

1. **オフラインモード**: ネットワーク分離環境でのビルドをサポート（`COOKBOOK_OFFLINE`環境変数）
2. **ミラー設定**: 信頼できるミラーサーバの設定が可能（`cookbook.toml`）
3. **クリーンビルド**: `clean_build`/`clean_target`オプションによる汚染防止

### 推奨されるセキュリティプラクティス

1. 秘密鍵ファイル（`id_ed25519.toml`）のアクセス権限を適切に設定
2. tarファイルには必ずBLAKE3ハッシュを指定
3. 本番環境では信頼できるミラーサーバのみを使用
4. 定期的な依存クレートのセキュリティアップデート
