# データベース設計書

## 概要

本ドキュメントは、Redox OS Cookbook（ビルドシステム）におけるデータ永続化とデータモデルの設計を記述します。本プロジェクトは従来のSQLデータベースを使用せず、TOML形式の設定ファイルとRustの構造体によるファイルベースのデータ永続化を採用しています。

### データ永続化の方式

| 方式 | 用途 | 特徴 |
| --- | --- | --- |
| TOMLファイル | 設定・レシピ・メタデータ | 人間可読、バージョン管理対応 |
| pkgarアーカイブ | パッケージバイナリ | 署名付きアーカイブ形式 |
| メモリ内データ構造 | ビルド時の一時データ | Rustの構造体・列挙型 |

## テーブル一覧

以下は、本プロジェクトにおける主要なデータエンティティとその対応する設定ファイル/構造体の一覧です。

| テーブル名（論理名） | 対応エンティティ | 説明 |
| --- | --- | --- |
| CookbookConfig | `cookbook.toml` / `CookbookConfig`構造体 | ビルドシステム全体の設定 |
| CookConfig | `[cook]`セクション / `CookConfig`構造体 | ビルド実行時のオプション設定 |
| Recipe | `recipe.toml` / `Recipe`構造体 | パッケージのビルドレシピ定義 |
| SourceRecipe | `[source]`セクション / `SourceRecipe`列挙型 | ソースコードの取得方法 |
| BuildRecipe | `[build]`セクション / `BuildRecipe`構造体 | ビルド方法の設定 |
| PackageRecipe | `[package]`セクション / `PackageRecipe`構造体 | パッケージのメタ情報 |
| OptionalPackageRecipe | `[[optional-packages]]`配列 / `OptionalPackageRecipe`構造体 | オプションパッケージの設定 |
| Package | `stage.toml` / `Package`構造体 | ビルド成果物のメタデータ |

## 各テーブル定義

### 1. CookbookConfig

ビルドシステム全体の設定を管理するエンティティです。`cookbook.toml`ファイルに保存され、アプリケーション起動時に読み込まれます。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| cook | CookConfig | NO | ビルド実行時のオプション設定 |
| mirrors | HashMap<String, String> | NO | URLミラーのマッピング（キー: 元のURL、値: ミラーURL） |

**対応するRust構造体**:
```rust
pub struct CookbookConfig {
    pub cook: CookConfig,
    pub mirrors: HashMap<String, String>,
}
```

**保存先**: `cookbook.toml`

---

### 2. CookConfig

ビルド実行時のオプション設定を管理するエンティティです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| offline | bool | NO | オフラインモードで実行するか |
| jobs | usize | NO | 並列ジョブ数（デフォルト: nproc） |
| tui | bool | NO | TUIを使用して並列ビルドを許可するか |
| logs | bool | NO | ログをbuild/logsディレクトリに出力するか |
| nonstop | bool | NO | ビルドエラーを無視して継続するか |
| verbose | bool | NO | 詳細ログを出力するか |
| clean_build | bool | NO | ビルド前にビルドディレクトリをクリーンするか |
| clean_target | bool | NO | ビルド後にターゲットディレクトリをクリーンするか |

**対応するRust構造体**:
```rust
pub struct CookConfig {
    pub offline: bool,
    pub jobs: usize,
    pub tui: bool,
    pub logs: bool,
    pub nonstop: bool,
    pub verbose: bool,
    pub clean_build: bool,
    pub clean_target: bool,
}
```

**環境変数によるデフォルト値**:
- `COOKBOOK_MAKE_JOBS`: 並列ジョブ数
- `COOKBOOK_OFFLINE`: オフラインモード
- `COOKBOOK_VERBOSE`: 詳細ログ
- `COOKBOOK_NONSTOP`: エラー継続
- `COOKBOOK_CLEAN_BUILD`: ビルド前クリーン
- `COOKBOOK_CLEAN_TARGET`: ビルド後クリーン
- `COOKBOOK_LOGS`: ログ出力
- `CI`: CI環境でのTUI無効化

---

### 3. Recipe

パッケージのビルドレシピを定義するエンティティです。各パッケージディレクトリの`recipe.toml`ファイルに保存されます。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| source | SourceRecipe | YES | ソースコードの取得方法 |
| build | BuildRecipe | NO | ビルド方法の設定 |
| package | PackageRecipe | NO | パッケージのメタ情報 |
| optional_packages | Vec<OptionalPackageRecipe> | NO | オプションパッケージのリスト |

**対応するRust構造体**:
```rust
pub struct Recipe {
    pub source: Option<SourceRecipe>,
    pub build: BuildRecipe,
    pub package: PackageRecipe,
    pub optional_packages: Vec<OptionalPackageRecipe>,
}
```

**保存先**: `recipes/{category}/{name}/recipe.toml`

---

### 4. SourceRecipe

ソースコードの取得方法を定義する列挙型エンティティです。以下のバリアントがあります。

#### 4.1 Git ソース

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| git | String | NO | Gitリポジトリの URL |
| upstream | String | YES | アップストリームリポジトリのURL |
| branch | String | YES | 追跡するブランチ名 |
| rev | String | YES | 使用するリビジョン（コミットハッシュ） |
| shallow_clone | bool | YES | ツリーレスクローンを使用するか |
| patches | Vec<String> | NO | 適用するパッチファイルのリスト |
| script | String | YES | ソース準備用のスクリプト |

#### 4.2 Tar ソース

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| tar | String | NO | tarファイルのURL |
| blake3 | String | YES | tarファイルのBLAKE3ハッシュ |
| patches | Vec<String> | NO | 適用するパッチファイルのリスト |
| script | String | YES | ソース準備用のスクリプト（例: ./autogen.sh） |

#### 4.3 Path ソース

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| path | String | NO | ソースへのパス |

#### 4.4 SameAs ソース

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| same_as | String | NO | ソースを共有するパッケージへの相対パス |

**対応するRust列挙型**:
```rust
pub enum SourceRecipe {
    SameAs { same_as: String },
    Path { path: String },
    Git { git: String, upstream: Option<String>, branch: Option<String>, rev: Option<String>, shallow_clone: Option<bool>, patches: Vec<String>, script: Option<String> },
    Tar { tar: String, blake3: Option<String>, patches: Vec<String>, script: Option<String> },
}
```

---

### 5. BuildRecipe

ビルド方法を定義するエンティティです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| kind | BuildKind | NO | ビルドテンプレートの種類 |
| dependencies | Vec<PackageName> | NO | ビルド時依存パッケージのリスト |
| dev_dependencies | Vec<PackageName> | NO | 開発用依存パッケージのリスト |

**BuildKind（ビルドテンプレート）の種類**:

| 値 | 説明 | 追加フィールド |
| --- | --- | --- |
| none | ビルドしない（メタパッケージ用） | なし |
| remote | リモートからビルド済みパッケージをダウンロード | なし |
| cargo | Cargoを使用してビルド | package_path, cargoflags |
| configure | configure + makeを使用してビルド | configureflags |
| cmake | CMakeを使用してビルド | cmakeflags |
| meson | Mesonを使用してビルド | mesonflags |
| custom | カスタムスクリプトを使用してビルド | script |

**対応するRust構造体**:
```rust
pub struct BuildRecipe {
    pub kind: BuildKind,
    pub dependencies: Vec<PackageName>,
    pub dev_dependencies: Vec<PackageName>,
}
```

---

### 6. PackageRecipe

パッケージのメタ情報を定義するエンティティです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| dependencies | Vec<PackageName> | NO | ランタイム依存パッケージのリスト |
| version | String | YES | パッケージのバージョン |

**対応するRust構造体**:
```rust
pub struct PackageRecipe {
    pub dependencies: Vec<PackageName>,
    pub version: Option<String>,
}
```

---

### 7. OptionalPackageRecipe

オプションパッケージを定義するエンティティです。1つのレシピから複数のパッケージを生成する場合に使用します。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | String | NO | オプションパッケージの名前 |
| dependencies | Vec<PackageName> | NO | 依存パッケージのリスト |
| files | Vec<String> | NO | 含めるファイルのパターンリスト |

**対応するRust構造体**:
```rust
pub struct OptionalPackageRecipe {
    pub name: String,
    pub dependencies: Vec<PackageName>,
    pub files: Vec<String>,
}
```

---

### 8. Package（ビルド成果物メタデータ）

ビルド成果物のメタデータを表すエンティティです。`stage.toml`ファイルとして保存されます。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | PackageName | NO | パッケージ名 |
| version | String | NO | パッケージバージョン |
| target | String | NO | ターゲットアーキテクチャ |
| blake3 | String | NO | パッケージのBLAKE3ハッシュ |
| network_size | u64 | NO | ネットワーク転送サイズ |
| storage_size | u64 | NO | ストレージサイズ |
| depends | Vec<PackageName> | NO | 依存パッケージのリスト |
| commit_identifier | String | NO | コミット識別子 |
| source_identifier | String | NO | ソース識別子 |
| time_identifier | String | NO | 時間識別子 |

**保存先**: `recipes/{category}/{name}/target/{arch}/stage.toml`

---

### 9. CookRecipe（ビルド時データ）

ビルド実行時に使用される拡張レシピデータです。メモリ内のみで使用されます。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | PackageName | NO | パッケージ名 |
| dir | PathBuf | NO | レシピディレクトリのパス |
| recipe | Recipe | NO | 読み込まれたレシピ |
| target | &'static str | NO | ターゲットアーキテクチャ |
| is_deps | bool | NO | 依存として追加されたかどうか |
| rule | String | NO | 適用されたビルドルール（source/local/binary/ignore） |

**対応するRust構造体**:
```rust
pub struct CookRecipe {
    pub name: PackageName,
    pub dir: PathBuf,
    pub recipe: Recipe,
    pub target: &'static str,
    pub is_deps: bool,
    pub rule: String,
}
```

---

### 10. IdentifierConfig

ソースの識別情報を管理するエンティティです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| commit_identifier | String | NO | コミットハッシュまたは識別子 |
| source_identifier | String | NO | ソースURL/パス識別子 |
| time_identifier | String | NO | 取得時刻の識別子 |

**対応するRust構造体**:
```rust
pub struct IdentifierConfig {
    pub commit_identifier: String,
    pub source_identifier: String,
    pub time_identifier: String,
}
```

---

### 11. AutoDeps

自動検出された依存関係を格納するエンティティです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| packages | BTreeSet<PackageName> | NO | 自動検出されたパッケージ依存のセット |

**対応するRust構造体**:
```rust
pub struct AutoDeps {
    pub packages: BTreeSet<PackageName>,
}
```

## 備考

### ファイル構造

```
redox-master/
├── cookbook.toml                    # グローバル設定
├── build/
│   ├── id_ed25519.toml              # 署名用秘密鍵
│   └── id_ed25519.pub.toml          # 署名用公開鍵
├── config/
│   ├── base.toml                    # ベース設定
│   ├── {arch}/                      # アーキテクチャ別設定
│   │   ├── ci.toml
│   │   ├── demo.toml
│   │   └── ...
│   └── ...
└── recipes/
    └── {category}/
        └── {name}/
            ├── recipe.toml           # レシピ定義
            ├── *.patch               # パッチファイル
            └── target/
                └── {arch}/
                    ├── source/       # ソースコード
                    ├── stage/        # ビルド成果物
                    ├── stage.pkgar   # パッケージアーカイブ
                    └── stage.toml    # パッケージメタデータ
```

### データ整合性

- **BLAKE3ハッシュ**: パッケージの整合性検証に使用
- **Ed25519署名**: パッケージの署名と検証に使用
- **シングルトン設定**: `OnceLock`を使用したスレッドセーフな設定管理

### データの永続化タイミング

| エンティティ | 読み込みタイミング | 書き込みタイミング |
| --- | --- | --- |
| CookbookConfig | アプリケーション起動時 | 手動編集のみ |
| Recipe | ビルド開始時 | 手動編集のみ |
| Package | パッケージング時 | パッケージング完了時 |
| pkgarアーカイブ | インストール時 | パッケージング完了時 |
