# 機能設計書 14-依存関係解決

## 概要

本ドキュメントは、Cookbook（Redox OSパッケージビルドシステム）における依存関係解決機能の設計を記述する。パッケージのビルド依存関係と実行時依存関係を再帰的に解決し、正しいビルド順序を決定する機能を提供する。

### 本機能の処理概要

依存関係解決機能は、パッケージのrecipe.tomlに記述されたビルド依存関係（build.dependencies）、開発依存関係（build.dev-dependencies）、および実行時依存関係（package.dependencies）を再帰的に解析し、必要なすべてのパッケージを正しい順序で収集する機能である。

**業務上の目的・背景**：ソフトウェアパッケージは他のパッケージ（ライブラリ、ツール等）に依存することが一般的である。ビルドを成功させるためには、依存パッケージが先にビルド・インストールされている必要がある。本機能により、開発者は依存関係を宣言するだけで、システムが自動的に正しいビルド順序を決定する。

**機能の利用シーン**：`repo cook`コマンド実行時に自動的に呼び出される。また、`repo tree`コマンドで依存関係ツリーを可視化する際にも使用される。

**主要な処理内容**：
1. 指定パッケージのrecipe.tomlからdependenciesを抽出
2. 各依存パッケージに対して再帰的に依存関係を解析
3. 循環依存の検出と防止
4. ビルド依存と実行時依存の分離管理
5. ホストパッケージとターゲットパッケージの区別

**関連システム・外部連携**：レシピファイルシステム（recipes/配下のディレクトリ）と連携する。

**権限による制御**：特別な権限制御はなし。ファイル読み取り権限のみ必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | TUIメイン画面 | API連携 | ビルド対象パッケージの依存関係を再帰的に解決 |
| 4 | repo cookコマンド画面 | API連携 | ビルド依存関係と実行時依存関係の再帰的解決 |
| 7 | repo pushコマンド画面 | API連携 | --with-package-deps指定時の依存関係解決 |
| 8 | repo treeコマンド画面 | API連携 | パッケージ依存関係の再帰的解決 |

## 機能種別

データ解析処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| names | &[PackageName] | Yes | 解析対象のパッケージ名リスト | 空でないこと |
| recurse_build_deps | bool | Yes | ビルド依存を再帰解析するか | - |
| recurse_dev_build_deps | bool | Yes | 開発依存を再帰解析するか | - |
| recurse_package_deps | bool | Yes | 実行時依存を再帰解析するか | - |
| collect_build_deps | bool | Yes | ビルド依存を結果に含めるか | - |
| collect_package_deps | bool | Yes | 実行時依存を結果に含めるか | - |
| collect_self | bool | Yes | 自分自身を結果に含めるか | - |
| recursion | usize | Yes | 再帰深さ制限 | WALK_DEPTH以下 |

### 入力データソース

- recipe.toml: 各パッケージの依存関係定義
  - build.dependencies: ビルド時依存（ライブラリ等）
  - build.dev-dependencies: 開発時依存（ビルドツール等）
  - package.dependencies: 実行時依存

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| recipes | Vec<CookRecipe> | 依存関係順に並んだレシピリスト |
| names | Vec<PackageName> | 依存関係順に並んだパッケージ名リスト |

### 出力先

- メモリ上のデータ構造（呼び出し元に返却）

## 処理フロー

### 処理シーケンス

```
1. 入力パッケージリストを走査
   └─ 各パッケージに対して以下を実行
2. レシピ読み込み
   └─ CookRecipe::from_name()でrecipe.tomlを解析
3. ビルド依存の再帰解析（recurse_build_deps=true時）
   └─ build.dependenciesに対してnew_recursive()を再帰呼び出し
4. 開発依存の再帰解析（recurse_dev_build_deps=true時）
   └─ build.dev-dependenciesに対してnew_recursive()を再帰呼び出し
5. 実行時依存の再帰解析（recurse_package_deps=true時）
   └─ package.dependenciesに対してnew_recursive()を再帰呼び出し
6. 重複排除
   └─ BTreeSetで既に収集済みのパッケージをスキップ
7. 自分自身の追加（collect_self=true時）
   └─ 依存パッケージの後に自分を追加
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パッケージリスト走査]
    B --> C[CookRecipe::from_name]
    C --> D{recurse_build_deps?}
    D -->|Yes| E[build.dependencies再帰解析]
    D -->|No| F{recurse_dev_build_deps?}
    E --> F
    F -->|Yes| G[dev-dependencies再帰解析]
    F -->|No| H{recurse_package_deps?}
    G --> H
    H -->|Yes| I[package.dependencies再帰解析]
    H -->|No| J{collect_self?}
    I --> J
    J -->|Yes| K[自分自身を追加]
    J -->|No| L{次のパッケージあり?}
    K --> L
    L -->|Yes| C
    L -->|No| M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-14-01 | 再帰深さ制限 | WALK_DEPTH（デフォルト128）を超える再帰はエラー | 全再帰解析 |
| BR-14-02 | 重複排除 | 同一パッケージは一度だけ結果に含める | 全解析 |
| BR-14-03 | ホスト依存変換 | ホストパッケージのビルド時、依存もホスト用に変換 | name.is_host()=true時 |
| BR-14-04 | 自己依存除外 | ホストパッケージ変換時、自分自身への依存は除外 | ホストパッケージ時 |
| BR-14-05 | 依存順序 | 依存パッケージは被依存パッケージより先に配置 | 結果リスト生成時 |

### 計算ロジック

再帰深さの管理:
```
recursion_new = recursion - 1
if recursion_new == 0 then エラー（Recursion）
```

## データベース操作仕様

該当なし（ファイルシステムからの読み取りのみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-DEP-01 | PackageNotFound | 依存パッケージが存在しない | パッケージ名の確認、レシピの追加 |
| E-DEP-02 | Recursion | 再帰深さがWALK_DEPTHを超過 | 循環依存の確認、依存構造の見直し |
| E-DEP-03 | FileMissing | recipe.tomlが見つからない | レシピファイルの配置確認 |
| E-DEP-04 | Parse | recipe.tomlの解析エラー | TOML構文の修正 |

### リトライ仕様

自動リトライなし。エラー発生時は依存チェーンをエラーメッセージに含めて報告。

## トランザクション仕様

該当なし（読み取り専用処理）

## パフォーマンス要件

- BTreeSetによる重複検出で計算量を抑制
- 再帰深さ制限で無限ループを防止

## セキュリティ考慮事項

- 再帰深さ制限による DoS 攻撃（悪意のある循環依存）の防止

## 備考

- WALK_DEPTHはlib.rsで定義（デフォルト128）
- ホストパッケージは`host.パッケージ名`形式で表現

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、依存関係を保持するデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | BuildRecipe構造体（136-144行目）のdependencies、dev_dependenciesフィールド |
| 1-2 | recipe.rs | `src/recipe.rs` | PackageRecipe構造体（146-151行目）のdependenciesフィールド |
| 1-3 | recipe.rs | `src/recipe.rs` | CookRecipe構造体（195-204行目）の全体構造 |

**読解のコツ**: `Vec<PackageName>`として依存関係が格納される。PackageNameはpkgクレートで定義されており、`host.`プレフィックスでホストパッケージを区別できる。

#### Step 2: エントリーポイントを理解する

依存関係解決のエントリーポイントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | recipe.rs | `src/recipe.rs` | get_build_deps_recursive関数（385-400行目） |
| 2-2 | recipe.rs | `src/recipe.rs` | get_package_deps_recursive関数（403-420行目） |

**主要処理フロー**:
1. **385-400行目**: ビルド依存を再帰的に取得（dev依存含むかのフラグ付き）
2. **403-420行目**: 実行時依存を再帰的に取得

#### Step 3: 再帰解析ロジックを理解する

コアとなる再帰解析処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | recipe.rs | `src/recipe.rs` | new_recursive関数（285-383行目） |

**主要処理フロー**:
- **295-296行目**: 再帰深さチェック
- **298-300行目**: 結果格納用のVecとBTreeSet
- **301-302行目**: 入力パッケージリストの走査開始
- **304-326行目**: ビルド依存の再帰解析
- **328-350行目**: 開発依存の再帰解析
- **352-374行目**: 実行時依存の再帰解析
- **376-379行目**: 自分自身の追加

#### Step 4: ホストパッケージ変換を理解する

ホストパッケージの依存関係変換ロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | recipe.rs | `src/recipe.rs` | CookRecipe::new関数（228-260行目） |

**主要処理フロー**:
- **230行目**: package_target()でターゲット判定
- **231-251行目**: ホストパッケージの場合、依存関係をホスト用に変換

### プログラム呼び出し階層図

```
repo cook / repo tree (CLI)
    │
    ├─ CookRecipe::get_build_deps_recursive() [recipe.rs:385]
    │      │
    │      └─ CookRecipe::new_recursive() [recipe.rs:285]
    │             │
    │             ├─ CookRecipe::from_name() [recipe.rs:262]
    │             │      │
    │             │      ├─ recipes::find() [pkg crate]
    │             │      │
    │             │      └─ Recipe::new() [recipe.rs:207]
    │             │
    │             └─ new_recursive() (再帰呼び出し)
    │                    └─ build.dependencies / dev_dependencies解析
    │
    └─ CookRecipe::get_package_deps_recursive() [recipe.rs:403]
           │
           └─ CookRecipe::new_recursive() [recipe.rs:285]
                  └─ package.dependencies解析
```

### データフロー図

```
[入力]                    [処理]                         [出力]

パッケージ名リスト ────▶ CookRecipe::from_name ────────▶ CookRecipe
                              │
                              ▼

recipe.toml ──────────▶ 依存関係抽出 ─────────────────▶ Vec<PackageName>
  build.dependencies          │                          - ビルド依存
  dev-dependencies            │                          - 開発依存
  package.dependencies        │                          - 実行時依存
                              │
                              ▼

                         new_recursive() ──────────────▶ 再帰呼び出し
                         （深さ制限付き）                   │
                              │                            │
                              ▼                            │
                                                           │
                         BTreeSet重複チェック ◀────────────┘
                              │
                              ▼

                         結果マージ ────────────────────▶ Vec<CookRecipe>
                         （依存順）                        または
                                                         Vec<PackageName>
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| recipe.rs | `src/recipe.rs` | ソース | 依存関係解決のメインロジック |
| lib.rs | `src/lib.rs` | ソース | WALK_DEPTH定数の定義 |
| cook_build.rs | `src/cook/cook_build.rs` | ソース | ビルド時の依存関係取得呼び出し |
| tree.rs | `src/cook/tree.rs` | ソース | 依存関係ツリー表示 |
