# 機能設計書 19-pkgarアーカイブ作成

## 概要

本ドキュメントは、Cookbook（Redox OSパッケージビルドシステム）におけるpkgarアーカイブ作成機能の設計を記述する。stageディレクトリの内容をpkgar形式でアーカイブ化する機能を提供する。

### 本機能の処理概要

pkgarアーカイブ作成機能は、ビルドが完了したstageディレクトリの内容を、Redox OS独自のパッケージ形式であるpkgar（Package Archive）にアーカイブ化する機能である。

**業務上の目的・背景**：Redox OSはpkgar形式のパッケージを使用してソフトウェアを配布・インストールする。pkgar形式はEd25519署名による整合性検証、blake3ハッシュによるファイル検証を備えた安全なパッケージ形式である。本機能により、ビルド成果物を配布可能なパッケージに変換できる。

**機能の利用シーン**：`repo cook`コマンドでパッケージをビルドした後、ステージディレクトリ作成に続いて自動的に呼び出される。生成されたpkgarファイルはリポジトリに公開される。

**主要な処理内容**：
1. Ed25519鍵ペアの生成または読み込み
2. ステージディレクトリの更新時刻チェック
3. 古いpkgarファイルの削除
4. pkgar::createによるアーカイブ作成
5. 署名付きpkgarファイルの生成

**関連システム・外部連携**：pkgarライブラリ、pkgar_keysライブラリ（鍵管理）と連携する。

**権限による制御**：特別な権限制御はなし。ファイル書き込み権限のみ必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | repo cookコマンド画面 | API連携 | stageディレクトリのpkgar形式アーカイブ化 |

## 機能種別

パッケージング処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| recipe | &CookRecipe | Yes | レシピ情報 | - |
| stage_dirs | &Vec<PathBuf> | Yes | ステージディレクトリリスト | 存在するディレクトリ |
| auto_deps | &BTreeSet<PackageName> | Yes | 自動検出された依存関係 | - |
| logger | &PtyOut | Yes | ログ出力先 | - |

### 入力データソース

- ステージディレクトリ: アーカイブ対象のファイル群
- build/id_ed25519.toml: 秘密鍵ファイル
- build/id_ed25519.pub.toml: 公開鍵ファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| stage.pkgar | File | 署名付きパッケージアーカイブ |
| stage.toml | File | パッケージメタデータ |

### 出力先

- pkgarファイル: `{recipe_dir}/target/{target}/stage.pkgar`
- オプショナルパッケージ: `{recipe_dir}/target/{target}/stage.{name}.pkgar`

## 処理フロー

### 処理シーケンス

```
1. BuildKindチェック
   └─ BuildKind::Noneはメタパッケージなのでpkgar作成不要
2. 鍵ファイル確認
   └─ 存在しなければEd25519鍵ペアを生成
3. ステージ更新時刻取得
   └─ modified_all()で全ステージの最新更新時刻を取得
4. パッケージリスト取得
   └─ get_packages_list()でオプショナル含む全パッケージを取得
5. 各パッケージに対してループ
   └─ 更新判定、pkgar作成、メタデータ生成
6. pkgar::create呼び出し
   └─ 秘密鍵でステージディレクトリをアーカイブ化
7. メタデータ生成
   └─ package_toml()でstage.tomlを生成
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{BuildKind::None?}
    B -->|Yes| Z[終了 - メタパッケージ]
    B -->|No| C{鍵ファイル存在?}
    C -->|No| D[Ed25519鍵ペア生成]
    C -->|Yes| E[ステージ更新時刻取得]
    D --> E
    E --> F[パッケージリスト取得]
    F --> G[パッケージループ開始]
    G --> H{pkgarより新しいstage?}
    H -->|Yes| I[古いpkgar削除]
    H -->|No| J{pkgar存在しない?}
    I --> K[pkgar::create実行]
    J -->|Yes| K
    J -->|No| L{toml存在しない?}
    K --> L
    L -->|Yes| M[package_toml生成]
    L -->|No| N{次のパッケージあり?}
    M --> N
    N -->|Yes| G
    N -->|No| O[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-19-01 | メタパッケージ除外 | BuildKind::Noneはpkgar作成しない | BuildKind判定時 |
| BR-19-02 | 鍵自動生成 | 鍵ファイルがなければ自動生成 | 初回ビルド時 |
| BR-19-03 | 更新時pkgar再作成 | stageが更新されていればpkgarを再作成 | パッケージング時 |
| BR-19-04 | オプショナル依存 | オプショナルパッケージはメインパッケージに依存 | オプショナル処理時 |
| BR-19-05 | clean_target対応 | clean_target時はpkgar存在確認でスキップ | clean_target設定時 |

### 計算ロジック

pkgar更新判定:
```
再作成必要 = (package_file.is_file() AND modified(package_file) < stage_modified)
           OR (NOT package_file.is_file())
```

## データベース操作仕様

該当なし（ファイルシステム操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-PKGAR-01 | 鍵生成失敗 | ファイル書き込み権限エラー | build/ディレクトリの権限確認 |
| E-PKGAR-02 | ステージ不存在 | stageディレクトリがない | ビルドの再実行 |
| E-PKGAR-03 | アーカイブ作成失敗 | pkgar::createエラー | ステージ内容の確認 |
| E-PKGAR-04 | メタデータ作成失敗 | シリアライズエラー | 依存関係の確認 |

### リトライ仕様

自動リトライなし。エラー時はビルド全体が失敗。

## トランザクション仕様

- pkgarファイルは直接作成（アトミック性は保証されない）
- 失敗時は不完全なファイルが残る可能性あり

## パフォーマンス要件

- ステージ更新判定で不要なアーカイブ化をスキップ
- 大きなパッケージでもメモリ効率的に処理

## セキュリティ考慮事項

- Ed25519秘密鍵はbuild/id_ed25519.tomlに保存
- 秘密鍵はリポジトリにコミットしないこと
- 署名によりパッケージの整合性と出所を保証

## 備考

- pkgar形式の詳細はpkgarクレートを参照
- 圧縮は現在サポートされていない（network_size == storage_size）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

pkgarパッケージング用のパス構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | package.rs | `src/cook/package.rs` | package_stage_paths関数（200-212行目） |

**読解のコツ**: `package_stage_paths`は(stage_dir, pkgar_file, toml_file)のタプルを返す。COOKBOOK_CROSS_TARGETによるパス変更にも対応。

#### Step 2: エントリーポイントを理解する

pkgar作成のエントリーポイントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | package.rs | `src/cook/package.rs` | package関数（15-118行目） |

**主要処理フロー**:
1. **23-33行目**: BuildKind::Noneの特別処理（メタパッケージ）
2. **35-48行目**: 鍵ファイルの確認と生成
3. **50-57行目**: ステージ更新時刻の取得
4. **59-115行目**: 各パッケージに対するpkgar作成とメタデータ生成

#### Step 3: pkgar作成処理を理解する

実際のアーカイブ作成処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | package.rs | `src/cook/package.rs` | pkgar::create呼び出し（72-78行目） |

**主要処理フロー**:
- **62-70行目**: 更新判定と古いファイル削除
- **72-78行目**: pkgar::createでアーカイブ作成
- **87-114行目**: package_tomlでメタデータ生成

#### Step 4: メタデータ生成を理解する

stage.tomlの生成ロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | package.rs | `src/cook/package.rs` | package_toml関数（120-174行目） |

**主要処理フロー**:
- **127-131行目**: auto_depsをpackage_depsにマージ
- **133-153行目**: pkgarからblake3ハッシュとサイズを取得
- **155-170行目**: Packageオブジェクトを構築してシリアライズ

### プログラム呼び出し階層図

```
cook (CLI entry point)
    │
    └─ package::package() [package.rs:15]
           │
           ├─ 鍵ファイル確認/生成 [package.rs:35-48]
           │      ├─ pkgar_keys::SecretKeyFile::new()
           │      ├─ public_key.save()
           │      └─ secret_key.save()
           │
           ├─ modified_all(stage_dirs) [package.rs:50]
           │
           ├─ get_packages_list() [recipe.rs:218]
           │
           └─ パッケージループ [package.rs:61-115]
                  │
                  ├─ package_stage_paths() [package.rs:200]
                  │
                  ├─ pkgar::create() [package.rs:72]
                  │      └─ 秘密鍵でステージを署名アーカイブ化
                  │
                  └─ package_toml() [package.rs:107]
                         │
                         ├─ fetch_get_source_info() - ソース識別子取得
                         │
                         └─ serialize_and_write() - TOML出力
```

### データフロー図

```
[入力]                    [処理]                         [出力]

build/id_ed25519.toml ─▶ 鍵読み込み ────────────────────▶ SecretKey
(または自動生成)              │
                              ▼

stage/ ─────────────────▶ modified_all ─────────────────▶ stage_modified
(ビルド成果物)                │
                              ▼

                         pkgar::create ─────────────────▶ stage.pkgar
                         (SecretKey, stage_dir)            (署名付きアーカイブ)
                              │
                              ▼

auto_deps ──────────────▶ package_toml ─────────────────▶ stage.toml
package.dependencies          │                            name: "pkg"
source_info                   │                            version: "1.0.0"
                              │                            blake3: "abc..."
                              │                            depends: [...]
                              │
                              ▼

                         serialize_and_write ───────────▶ TOML出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| package.rs | `src/cook/package.rs` | ソース | pkgar作成のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | get_packages_list等 |
| fetch.rs | `src/cook/fetch.rs` | ソース | fetch_get_source_info |
| fs.rs | `src/cook/fs.rs` | ソース | serialize_and_write |
