# 機能設計書 2-ソース取得（Git）

## 概要

本ドキュメントは、Redox OSビルドシステムにおけるGitリポジトリからのソースコード取得機能の設計を記載する。Gitリポジトリのクローン、フェッチ、チェックアウト処理について詳述する。

### 本機能の処理概要

ソース取得（Git）機能は、recipe.tomlで指定されたGitリポジトリからソースコードを取得し、ビルドに必要な状態に準備する機能である。クローン、フェッチ、特定リビジョンへのチェックアウト、サブモジュールの更新などを自動的に行う。

**業務上の目的・背景**：Redox OSを構成する多くのパッケージは、GitLabやGitHubなどのGitリポジトリで管理されている。これらのソースコードを効率的かつ再現可能な方法で取得することで、一貫性のあるビルド環境を実現する。特定のリビジョンを指定することで、ビルドの再現性を保証する。

**機能の利用シーン**：
- 新規パッケージのソースコードを初めて取得する際（git clone）
- 既存ソースコードを最新の状態に更新する際（git fetch + checkout）
- 特定のリビジョンやブランチをチェックアウトする際
- サブモジュールを含むプロジェクトのソースを取得する際

**主要な処理内容**：
1. ソースディレクトリの存在確認
2. 新規の場合：git cloneによるリポジトリのクローン
3. 既存の場合：git fetchによるリモートからの更新取得
4. 指定リビジョンまたはブランチへのチェックアウト
5. サブモジュールの同期と更新
6. パッチファイルの適用（存在する場合）
7. 準備スクリプトの実行（存在する場合）

**関連システム・外部連携**：
- gitコマンド：バージョン管理操作
- ミラーサーバー：cookbook.tomlで設定されたミラーURL変換

**権限による制御**：特になし（ファイルシステムの書き込み権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | TUIメイン画面 | 主機能 | Fetch Queueパネルでソースコードダウンロードの進捗表示 |
| 3 | repo fetchコマンド画面 | API連携 | Gitリポジトリからのソースクローン・フェッチ |

## 機能種別

データ連携 / 外部コマンド実行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| git | String | Yes | GitリポジトリのURL | 有効なURL形式 |
| upstream | Option<String> | No | upstreamリポジトリのURL | 有効なURL形式 |
| branch | Option<String> | No | チェックアウトするブランチ名 | - |
| rev | Option<String> | No | チェックアウトするリビジョン（コミットハッシュ） | - |
| shallow_clone | Option<bool> | No | treeless cloneを使用するか（デフォルト: revがある場合true） | - |
| patches | Vec<String> | No | 適用するパッチファイルのリスト | ファイルが存在すること |
| script | Option<String> | No | ソース準備スクリプト | - |

### 入力データソース

- recipe.toml: sourceセクションのGit設定
- cookbook.toml: ミラー設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_dir | PathBuf | ソースコードが配置されたディレクトリパス |
| source_identifier | String | チェックアウトされたコミットハッシュ |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/source/`

## 処理フロー

### 処理シーケンス

```
1. ソースディレクトリの存在確認
   └─ {レシピディレクトリ}/sourceディレクトリの存在チェック
2. 新規クローン処理（ソースディレクトリが存在しない場合）
   ├─ source.tmpディレクトリを作成
   ├─ git cloneコマンドを実行
   │   ├─ URLをミラーサーバーに変換
   │   ├─ --recursiveオプションでサブモジュールも取得
   │   ├─ --branchオプションでブランチ指定（指定時）
   │   └─ --filter=tree:0オプションでshallow clone（指定時）
   └─ source.tmpをsourceにアトミックリネーム
3. 既存更新処理（ソースディレクトリが存在する場合）
   ├─ .gitディレクトリの存在確認
   ├─ git remote set-url originでURLを更新
   ├─ git fetch originでリモートから更新を取得
   └─ 更新が必要かどうかを判定
4. リビジョンチェックアウト
   ├─ revが指定されている場合：git checkout {rev}
   └─ revが未指定の場合：ブランチをリモートに同期
5. サブモジュール処理
   ├─ git submodule syncでURL同期
   └─ git submodule update --init --recursiveで更新
6. パッチ・スクリプト適用
   ├─ patchesの適用（存在する場合）
   └─ scriptの実行（存在する場合）
7. ソース情報の記録
   └─ source_info.tomlに識別子を保存
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{sourceディレクトリが存在するか}
    B -->|No| C[source.tmpを作成]
    C --> D[git clone実行]
    D --> E{shallow_cloneか}
    E -->|Yes| F[--filter=tree:0オプション追加]
    E -->|No| G[通常clone]
    F --> H[source.tmpをsourceにリネーム]
    G --> H
    B -->|Yes| I{.gitディレクトリが存在するか}
    I -->|No| J[エラー: Gitリポジトリではない]
    I -->|Yes| K[git remote set-url origin]
    K --> L[git fetch origin]
    L --> M{更新が必要か}
    M -->|No| N[キャッシュを使用]
    M -->|Yes| O{revが指定されているか}
    O -->|Yes| P[git checkout rev]
    O -->|No| Q[ブランチをリモートに同期]
    P --> R[git submodule sync]
    Q --> R
    H --> R
    R --> S[git submodule update]
    S --> T{パッチが存在するか}
    T -->|Yes| U[パッチを適用]
    T -->|No| V{スクリプトが存在するか}
    U --> V
    V -->|Yes| W[スクリプトを実行]
    V -->|No| X[source_info.tomlを保存]
    W --> X
    N --> X
    X --> Y[終了]
    J --> Y
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ミラーURL変換 | cookbook.tomlで設定されたミラーがあればURLを変換 | 常時 |
| BR-02 | shallow clone判定 | revが指定されている場合、デフォルトでshallow_clone=true | revが指定されている場合 |
| BR-03 | 更新スキップ判定 | HEADがリモートと同じ場合は更新をスキップ | ソースディレクトリが存在する場合 |
| BR-04 | アトミックリネーム | クローン完了後にsource.tmpからsourceにリネーム | 新規クローン時 |
| BR-05 | パッチ適用前リセット | パッチまたはスクリプトがある場合、git reset --hardを実行 | patches/scriptが存在する場合 |

### 計算ロジック

**更新必要性判定ロジック**：
```
1. detached HEAD状態の場合：
   - revが指定されていれば、そのrevとHEADを比較
   - 一致すればスキップ可能

2. ブランチにいる場合：
   - リモートトラッキングブランチの情報を取得
   - branchが指定されていれば、それと比較
   - リモート名が"origin"かどうか確認
   - リモートURLがrecipeのgitと一致するか確認
   - FETCH_HEADとHEADを比較
   - すべて一致すればスキップ可能
```

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムとgitコマンドのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ディレクトリエラー | sourceが存在するが.gitがない | ディレクトリを削除して再クローン |
| - | コマンドエラー | git clone/fetch/checkout失敗 | ネットワーク接続確認、URL確認 |
| - | ブランチエラー | 指定されたブランチが存在しない | ブランチ名を確認 |
| - | リビジョンエラー | 指定されたrevが存在しない | リビジョンを確認 |

### リトライ仕様

ネットワークエラーの場合、TUIモードではユーザーにリトライ/スキップ/終了を選択させる。

## トランザクション仕様

- source.tmpへの一時クローン後にsourceへアトミックリネーム
- 失敗時はsource.tmpを削除

## パフォーマンス要件

- shallow clone対応により、大規模リポジトリでもメモリ・ディスク使用量を削減
- ネットワーク帯域はリポジトリサイズに依存

## セキュリティ考慮事項

- ミラーURL変換：信頼できるミラーサーバーのみ設定すること
- git credential：システムの認証情報を使用

## 備考

- treeless clone（--filter=tree:0）はGit 2.19以降で利用可能
- Redox OS上ではbashスクリプトによるブランチリセットが制限される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Gitソース設定を表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe::Git variantの定義 |

**読解のコツ**:
- `#[serde(untagged)]`によりTOMLのキー名から自動的にGitタイプと判定される
- gitフィールドが存在すればSourceRecipe::Gitとしてデシリアライズされる

**主要処理フロー**:
- **35-53行目**: SourceRecipe::Git variant定義
  - git: String - リポジトリURL
  - upstream: Option<String> - upstreamリポジトリURL
  - branch: Option<String> - ブランチ名
  - rev: Option<String> - リビジョン（コミットハッシュ）
  - shallow_clone: Option<bool> - treeless clone設定
  - patches: Vec<String> - パッチファイルリスト
  - script: Option<String> - 準備スクリプト

#### Step 2: エントリーポイントを理解する

ソース取得処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch()関数、fetch_offline()関数 |

**主要処理フロー**:
- **117-406行目**: fetch()関数 - メインのソース取得処理
- **37-115行目**: fetch_offline()関数 - オフラインモードでのソース取得
- **161-317行目**: Git処理の分岐（fetch関数内）

#### Step 3: Git処理の詳細を理解する

Git操作の各ステップを詳細に把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | Git clone/fetch/checkout処理 |

**主要処理フロー**:
- **171-197行目**: 新規cloneの場合
  - source.tmpディレクトリを作成
  - git clone --recursive実行
  - ブランチ指定、shallow_clone対応
  - アトミックリネーム
- **198-249行目**: 既存更新の場合
  - .gitディレクトリの存在確認
  - git remote set-url origin
  - git fetch origin
  - 更新必要性の判定
- **251-290行目**: チェックアウト処理
  - rev指定時：git checkout {rev}
  - ブランチ同期処理
  - パッチ/スクリプトがある場合のgit reset --hard
- **292-312行目**: サブモジュール処理
  - git submodule sync --recursive
  - git submodule update --init --recursive

#### Step 4: ヘルパー関数を理解する

Git情報取得のためのヘルパー関数を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fs.rs | `src/cook/fs.rs` | get_git_head_rev(), get_git_fetch_rev(), get_git_remote_tracking() |

**主要処理フロー**:
- **291-308行目**: get_git_head_rev() - HEADのコミットハッシュとdetached状態を取得
- **336-367行目**: get_git_fetch_rev() - FETCH_HEADからリモートブランチのコミットを取得
- **371-460行目**: get_git_remote_tracking() - ローカルブランチのリモートトラッキング情報を取得

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    ├─ BuildKind判定
    │   ├─ None → fetch_apply_source_info()のみ
    │   └─ Remote → fetch_remote()
    │
    └─ SourceRecipe::Git処理
           │
           ├─ 新規clone
           │   ├─ create_dir_clean(source.tmp)
           │   ├─ Command::new("git").arg("clone")
           │   │   ├─ translate_mirror(git)
           │   │   ├─ --recursive
           │   │   ├─ --branch（指定時）
           │   │   └─ --filter=tree:0（shallow時）
           │   ├─ run_command()
           │   └─ rename(source.tmp, source)
           │
           ├─ 既存更新
           │   ├─ Command::new("git").arg("remote set-url")
           │   ├─ Command::new("git").arg("fetch origin")
           │   ├─ get_git_head_rev()
           │   ├─ get_git_remote_tracking()
           │   └─ get_git_fetch_rev()
           │
           ├─ checkout
           │   ├─ Command::new("git").arg("checkout").arg(rev)
           │   └─ Command::new("bash").arg(GIT_RESET_BRANCH)
           │
           ├─ submodule
           │   ├─ Command::new("git").arg("submodule sync")
           │   └─ Command::new("git").arg("submodule update")
           │
           └─ fetch_apply_patches()
               ├─ patch適用
               └─ script実行
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ source.git ────────▶ translate_mirror() ─────────────▶ Mirror URL
  └─ source.branch                  │
  └─ source.rev                     ▼
  └─ source.shallow_clone    git clone/fetch ─────────────▶ source/
  └─ source.patches                 │                          │
  └─ source.script                  ▼                          │
                            git checkout ──────────────────▶ 特定revision
                                    │
                                    ▼
                            git submodule update ──────────▶ submodules/
                                    │
                                    ▼
                            fetch_apply_patches() ─────────▶ パッチ適用済みソース
                                    │
                                    ▼
                            fetch_apply_source_info() ─────▶ source_info.toml
                                    │
                                    └─ source_identifier: コミットハッシュ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | ソース取得のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | SourceRecipe::Git定義 |
| fs.rs | `src/cook/fs.rs` | ソース | Git情報取得ヘルパー関数 |
| script.rs | `src/cook/script.rs` | ソース | GIT_RESET_BRANCHスクリプト定義 |
| config.rs | `src/config.rs` | ソース | translate_mirror()関数、ミラー設定 |
| pty.rs | `src/cook/pty.rs` | ソース | コマンド出力のロギング |
| recipe.toml | 各レシピディレクトリ | 設定 | Gitソース設定 |
| cookbook.toml | プロジェクトルート | 設定 | ミラー設定 |
