# 機能設計書 23-リポジトリ公開

## 概要

本ドキュメントは、Redox OS Cookbookシステムにおけるリポジトリ公開機能の設計を記述したものである。本機能は、ビルド済みパッケージをリポジトリディレクトリにコピーし、パッケージマネージャからアクセス可能な状態にする。

### 本機能の処理概要

リポジトリ公開機能は、ビルド完了後のパッケージファイル（.pkgar）とメタデータファイル（.toml）をリポジトリディレクトリに公開し、ユーザーがパッケージをインストールできる状態にする機能である。

**業務上の目的・背景**：Redox OSのパッケージ配布において、ビルドされたパッケージを中央リポジトリに集約し、パッケージマネージャから一元的にアクセスできるようにする必要がある。本機能により、ビルド成果物を適切な場所に配置し、repo.tomlを更新して配布可能な状態にする。

**機能の利用シーン**：`repo cook`コマンドの実行完了後に自動的に呼び出される。また、`repo_builder`コマンドとして単独でも実行可能。複数パッケージをまとめてリポジトリに公開する際に使用される。

**主要な処理内容**：
1. 対象パッケージのリスト取得と依存関係の再帰的解決
2. 各パッケージのステージファイル（.pkgar, .toml）をリポジトリディレクトリにコピー
3. 更新判定による効率的なファイルコピー
4. AppStreamメタデータの検出と記録
5. ビルド失敗パッケージのoutdated_packages記録
6. repo.tomlの更新

**関連システム・外部連携**：pkgar形式のパッケージファイル、パッケージメタデータ（TOML形式）、AppStreamメタデータと連携。repo.toml生成（機能No.24）と密接に関連。

**権限による制御**：特に権限制御はないが、リポジトリディレクトリへの書き込み権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 10 | repo_builderコマンド画面 | 主機能 | ビルド済みパッケージのリポジトリへのコピー |
| 4 | repo cookコマンド画面 | API連携 | cookコマンド完了後に自動実行 |

## 機能種別

ファイル操作 / パッケージ配布

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| repo_dir | PathBuf | Yes | リポジトリディレクトリ | 書き込み可能であること |
| recipe_list | Vec<String> | Yes | 公開対象レシピ名リスト | 少なくとも1つ以上 |
| appstream | bool | No | AppStream生成フラグ | COOKBOOK_APPSTREAM環境変数 |

### 入力データソース

- コマンドライン引数（repo_dir, recipe_list）
- 環境変数（COOKBOOK_APPSTREAM）
- 各レシピのtarget/{target}/stage.toml、stage.pkgar

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| {package}.pkgar | File | リポジトリにコピーされたパッケージファイル |
| {package}.toml | File | リポジトリにコピーされたメタデータ |
| repo.toml | File | リポジトリのパッケージ一覧 |

### 出力先

- パッケージファイル：`{repo_dir}/{target}/{package}.pkgar`
- メタデータファイル：`{repo_dir}/{target}/{package}.toml`
- リポジトリ一覧：`{repo_dir}/{target}/repo.toml`

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数の解析
   └─ repo_dir、recipe_listを取得
2. パッケージリストの構築
   └─ Package::new_recursive_nonstopで依存関係を含めて取得
3. 各レシピのパッケージ公開
   └─ stage.toml/stage.pkgarをrepoディレクトリにコピー
4. AppStreamメタデータの検出
   └─ usr/share/metainfoディレクトリの存在確認
5. AppStream生成（オプション）
   └─ appstreamcliでカタログ生成
6. ビルド失敗パッケージの記録
   └─ outdated_packagesとしてマーク
7. repo.tomlの更新
   └─ パッケージ一覧の生成
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数解析]
    B --> C{ホストパッケージ除外}
    C --> D[パッケージリスト構築]
    D --> E{パッケージあり?}
    E -->|No| F[終了]
    E -->|Yes| G[各レシピをループ]
    G --> H[レシピパス検索]
    H --> I{見つかった?}
    I -->|No| J[エラー出力]
    I -->|Yes| K[パッケージリスト取得]
    J --> G
    K --> L[各パッケージをループ]
    L --> M{stage.toml存在?}
    M -->|No| N[スキップ]
    M -->|Yes| O{更新必要?}
    N --> L
    O -->|Yes| P[ファイルコピー]
    O -->|No| Q[AppStream検出]
    P --> Q
    Q --> R{次のパッケージ?}
    R -->|Yes| L
    R -->|No| S{次のレシピ?}
    S -->|Yes| G
    S -->|No| T{AppStream有効?}
    T -->|Yes| U[AppStream生成]
    T -->|No| V[outdated記録]
    U --> V
    V --> W[repo.toml更新]
    W --> F
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | ホストパッケージ除外 | is_host()がtrueのパッケージはリポジトリに公開しない | パッケージリスト構築時 |
| BR-23-02 | 更新判定 | stage.tomlがrepo/{package}.tomlより新しい場合のみコピー | ファイルコピー時 |
| BR-23-03 | クロスターゲット除外 | COOKBOOK_CROSS_TARGETが設定されている場合は公開をスキップ | 処理開始時 |
| BR-23-04 | 依存関係含有 | パッケージ依存関係を再帰的に解決して公開対象に含める | パッケージリスト構築時 |

### 計算ロジック

**更新判定（is_newer関数）**：
```rust
fn is_newer(src: &Path, dst: &Path) -> bool {
    match (fs::metadata(src), fs::metadata(dst)) {
        (Ok(src_meta), Ok(dst_meta)) => {
            src_time > dst_time
        }
        (Ok(_), Err(_)) => true,  // dstが存在しない
        _ => false,
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はファイルシステム操作のみであり、データベース操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | RecipeNotFound | レシピパスが見つからない | レシピ名の確認 |
| - | RecipeUnreadable | レシピ読み込み失敗 | recipe.tomlの確認 |
| - | StageMissing | stage.tomlが存在しない | ビルド完了確認 |
| - | CopyError | ファイルコピー失敗 | ディスク容量・権限確認 |

### リトライ仕様

個別のパッケージでエラーが発生しても、エラーを出力して次のパッケージの処理を継続する（nonstop動作）。

## トランザクション仕様

ファイルシステム操作のため、トランザクション制御は行われない。各パッケージは独立して処理される。

## パフォーマンス要件

- ファイルコピー：ファイルサイズに依存
- 更新判定により不要なコピーを回避して効率化

## セキュリティ考慮事項

- リポジトリディレクトリへの書き込み権限が必要
- 公開されたパッケージは署名済みであることが前提

## 備考

- repo_builderは独立した実行可能ファイルとして提供される
- repo cookコマンドから自動的に呼び出される
- AppStream生成はオプション機能（COOKBOOK_APPSTREAM=true）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

repo_builderの設定構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | repo_builder.rs | `src/bin/repo_builder.rs` | CliConfig構造体（28-33行目） |

**読解のコツ**: CliConfigはrepo_dir、appstreamフラグ、recipe_listを保持する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | repo_builder.rs | `src/bin/repo_builder.rs` | main関数（49-53行目）、publish_packages関数（56行目） |

**主要処理フロー**:
- **49-50行目**: init_ident()でソース識別子を初期化
- **51行目**: CliConfig::parse_args()で引数解析
- **52行目**: publish_packages()を呼び出し

#### Step 3: パッケージリスト構築を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | repo_builder.rs | `src/bin/repo_builder.rs` | publish_packages関数（56-89行目） |

**主要処理フロー**:
- **57-59行目**: リポジトリパスの構築と作成
- **63-68行目**: ホストパッケージを除外したリスト作成
- **70-72行目**: ゼロパッケージの場合は早期終了
- **75-77行目**: クロスターゲットの場合はスキップ
- **84行目**: Package::new_recursive_nonstopで依存関係を含めて取得

#### Step 4: ファイルコピー処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | repo_builder.rs | `src/bin/repo_builder.rs` | パッケージ公開ループ（96-132行目） |

**主要処理フロー**:
- **96-104行目**: レシピ検索と読み込み
- **108-114行目**: パッケージリストのループ、ステージパス取得
- **115-118行目**: stage.toml存在チェック
- **120-126行目**: is_newerによる更新判定とファイルコピー
- **128-130行目**: AppStreamメタデータの検出

#### Step 5: 更新判定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | repo_builder.rs | `src/bin/repo_builder.rs` | is_newer関数（16-26行目） |

**主要処理フロー**:
- **17-19行目**: 両ファイルのメタデータ取得
- **20行目**: 更新時刻の比較
- **21行目**: srcのみ存在する場合はtrue

### プログラム呼び出し階層図

```
main() [src/bin/repo_builder.rs:49]
    │
    ├─ init_ident() [src/cook/ident.rs]
    │      └─ ソース識別子初期化
    │
    ├─ CliConfig::parse_args() [src/bin/repo_builder.rs:36]
    │      └─ コマンドライン引数解析
    │
    └─ publish_packages() [src/bin/repo_builder.rs:56]
           │
           ├─ Package::new_recursive_nonstop() [外部ライブラリ]
           │      └─ 依存関係を含むパッケージリスト取得
           │
           ├─ recipes::find() [外部ライブラリ]
           │      └─ レシピパス検索
           │
           ├─ CookRecipe::from_path() [src/recipe.rs]
           │      └─ レシピ読み込み
           │
           ├─ package_stage_paths() [src/cook/package.rs]
           │      └─ ステージパス取得
           │
           ├─ is_newer() [src/bin/repo_builder.rs:16]
           │      └─ 更新判定
           │
           ├─ fs::copy() [標準ライブラリ]
           │      └─ ファイルコピー
           │
           ├─ AppStream生成処理 [134-174行目]
           │      └─ appstreamcliコマンド実行
           │
           ├─ outdated_packages記録 [177-215行目]
           │      └─ ビルド失敗パッケージのマーク
           │
           └─ repo.toml更新 [219-274行目]
                  └─ パッケージ一覧の生成
```

### データフロー図

```
[入力]                      [処理]                      [出力]

コマンドライン引数 ─────────┐
  repo_dir                 │
  recipe_list              │
                           ├──▶ parse_args ──▶ CliConfig
COOKBOOK_APPSTREAM ────────┘

recipe_list ───────────────▶ new_recursive_nonstop
                                   │
                                   ▼
                           パッケージリスト（依存含む）
                                   │
                                   ▼
stage.pkgar ───────────────▶ is_newer ───┬──▶ {repo}/{pkg}.pkgar
stage.toml ────────────────▶ is_newer ───┴──▶ {repo}/{pkg}.toml
                                   │
                                   ▼
                           AppStream検出
                                   │
                                   ▼
                           outdated_packages記録
                                   │
                                   ▼
                           repo.toml更新 ──▶ repo.toml
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| repo_builder.rs | `src/bin/repo_builder.rs` | ソース | リポジトリ公開のメイン処理 |
| repo.rs | `src/bin/repo.rs` | ソース | CLIエントリーポイント、publish_packages呼び出し |
| recipe.rs | `src/recipe.rs` | ソース | CookRecipe構造体、レシピ読み込み |
| package.rs | `src/cook/package.rs` | ソース | package_stage_paths関数 |
| ident.rs | `src/cook/ident.rs` | ソース | ソース識別子管理 |
| fetch.rs | `src/cook/fetch.rs` | ソース | fetch_get_source_info関数 |
