# 機能設計書 24-repo.toml生成

## 概要

本ドキュメントは、Redox OS Cookbookシステムにおけるrepo.toml生成機能の設計を記述したものである。本機能は、リポジトリ内の全パッケージ一覧とバージョン情報をrepo.tomlファイルに出力する。

### 本機能の処理概要

repo.toml生成機能は、リポジトリディレクトリ内のすべてのパッケージメタデータを集約し、パッケージマネージャがパッケージ一覧を取得するためのインデックスファイルを生成する機能である。

**業務上の目的・背景**：パッケージマネージャがリポジトリからパッケージをインストールする際、どのパッケージが利用可能かを知る必要がある。repo.tomlファイルは、リポジトリ内の全パッケージ名とそのバージョン（blake3ハッシュ）を一覧化し、効率的なパッケージ検索とインストールを可能にする。

**機能の利用シーン**：リポジトリ公開処理（機能No.23）の最終段階で自動的に実行される。パッケージのビルド・公開が完了するたびにrepo.tomlが更新され、最新のパッケージ情報が反映される。

**主要な処理内容**：
1. 既存のrepo.tomlがあれば読み込んで既存データを保持
2. リポジトリディレクトリ内の全.tomlファイルをスキャン
3. 各パッケージのblake3ハッシュまたはバージョンを抽出
4. outdated_packagesリストを更新（ビルド失敗パッケージ）
5. 統合されたパッケージ一覧をrepo.tomlに出力

**関連システム・外部連携**：パッケージメタデータファイル（.toml）、Repository構造体（外部pkgライブラリ）と連携。

**権限による制御**：特に権限制御はないが、リポジトリディレクトリへの書き込み権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 10 | repo_builderコマンド画面 | 主機能 | リポジトリ内パッケージ一覧の生成 |

## 機能種別

データ集約処理 / ファイル生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| repo_path | PathBuf | Yes | リポジトリディレクトリ | 読み取り可能であること |
| packages | BTreeMap<String, String> | Yes | 現在処理中のパッケージ情報 | - |
| outdated_packages | BTreeMap<String, SourceIdentifier> | Yes | ビルド失敗パッケージ情報 | - |

### 入力データソース

- リポジトリディレクトリ内の.tomlファイル
- 既存のrepo.toml（存在する場合）
- ビルド処理中に収集されたパッケージ情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| repo.toml | File | パッケージ一覧ファイル |

### 出力先

- `{repo_dir}/{target}/repo.toml`

**出力形式（TOML）**：
```toml
[packages]
packageA = "blake3hash1"
packageB = "blake3hash2"

[outdated_packages]
packageC.source_identifier = "..."
packageC.commit_identifier = "..."
packageC.time_identifier = "..."
```

## 処理フロー

### 処理シーケンス

```
1. 既存repo.tomlの読み込み（存在する場合）
   └─ Repository構造体にパース
2. 既存パッケージ情報のマージ
   └─ packages、outdated_packagesを保持
3. リポジトリディレクトリのスキャン
   └─ 全.tomlファイルを列挙
4. 各メタデータファイルの解析
   └─ blake3またはversionを抽出
5. パッケージ情報の更新
   └─ BTreeMapに格納
6. repo.tomlの生成
   └─ Repository構造体をTOMLシリアライズして書き込み
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{既存repo.toml存在?}
    B -->|Yes| C[repo.toml読み込み]
    B -->|No| D[空のpackagesマップ初期化]
    C --> E[既存packagesをマージ]
    D --> E
    E --> F[outdated_packagesをマージ]
    F --> G[リポジトリディレクトリスキャン]
    G --> H[各.tomlファイルをループ]
    H --> I{repo.tomlをスキップ}
    I -->|Yes| H
    I -->|No| J[TOMLパース]
    J --> K[blake3/version抽出]
    K --> L[packagesマップに追加]
    L --> M{次のファイル?}
    M -->|Yes| H
    M -->|No| N[Repositoryシリアライズ]
    N --> O[repo.toml書き込み]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-01 | blake3優先 | パッケージバージョンはblake3ハッシュを優先、なければversionを使用 | メタデータ解析時 |
| BR-24-02 | 既存データ保持 | 今回ビルドしなかったパッケージの情報は既存repo.tomlから引き継ぐ | マージ処理時 |
| BR-24-03 | outdated継承 | 今回ビルド成功したパッケージはoutdated_packagesから除外 | マージ処理時 |
| BR-24-04 | repo.toml除外 | ディレクトリスキャン時にrepo.toml自体は処理対象外 | スキャン時 |

### 計算ロジック

**バージョン抽出ロジック**：
```rust
let version_str = parsed
    .get("blake3")
    .unwrap_or_else(|| parsed.get("version").unwrap_or_else(|| &empty_ver))
    .as_str()
    .unwrap_or("");
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はファイルシステム操作のみであり、データベース操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ParseError | 既存repo.tomlのパースに失敗 | ファイル形式の確認 |
| - | ReadError | .tomlファイルの読み込みに失敗 | ファイル権限確認 |
| - | WriteError | repo.toml書き込みに失敗 | ディスク容量・権限確認 |

### リトライ仕様

特にリトライ処理は実装されていない。

## トランザクション仕様

ファイルシステム操作のため、トランザクション制御は行われない。repo.tomlは直接上書きされる。

## パフォーマンス要件

- ディレクトリスキャン：パッケージ数に依存
- 通常は数秒以内で完了

## セキュリティ考慮事項

- repo.tomlはパッケージのハッシュ値を含むため、改ざんされると偽パッケージのインストールにつながる可能性
- リポジトリサーバーでの適切なアクセス制御が必要

## 備考

- BTreeMapを使用しているため、出力されるパッケージ一覧はアルファベット順にソートされる
- outdated_packagesはビルドに失敗したパッケージを追跡し、次回ビルド時の参考情報として使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

repo.tomlのデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | 外部ライブラリ | pkg::package::Repository | packages、outdated_packagesフィールド |
| 1-2 | 外部ライブラリ | pkg::package::SourceIdentifier | ソース識別情報構造体 |

**読解のコツ**: Repositoryはpackages（パッケージ名→バージョン/ハッシュ）とoutdated_packages（パッケージ名→SourceIdentifier）のマップを持つ。

#### Step 2: 既存repo.toml読み込みを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | repo_builder.rs | `src/bin/repo_builder.rs` | repo.toml読み込み部分（219-241行目） |

**主要処理フロー**:
- **220行目**: repo.tomlパス構築
- **221行目**: 存在確認
- **222-226行目**: ファイル読み込みとパース
- **227-229行目**: 既存packagesをマージ
- **230-240行目**: outdated_packagesをマージ（ビルド成功分を除外）

#### Step 3: ディレクトリスキャンを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | repo_builder.rs | `src/bin/repo_builder.rs` | ディレクトリスキャン部分（243-266行目） |

**主要処理フロー**:
- **243行目**: fs::read_dir()でディレクトリエントリ取得
- **247-249行目**: .toml以外をスキップ
- **251-253行目**: repo.tomlをスキップ
- **255-256行目**: ファイル読み込みとTOMLパース
- **258-264行目**: blake3/version抽出
- **265行目**: packagesマップに追加

#### Step 4: シリアライズと書き込みを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | repo_builder.rs | `src/bin/repo_builder.rs` | シリアライズと書き込み（268-274行目） |

**主要処理フロー**:
- **268-271行目**: Repository構造体をTOMLシリアライズ
- **272-273行目**: repo.tomlに書き込み

### プログラム呼び出し階層図

```
publish_packages() [src/bin/repo_builder.rs:56]
    │
    ... (パッケージ公開処理)
    │
    ├─ repo.toml読み込み [219-241行目]
    │      │
    │      ├─ File::open()
    │      │      └─ 既存repo.toml読み込み
    │      │
    │      └─ toml::from_str::<Repository>()
    │             └─ TOMLパース
    │
    ├─ ディレクトリスキャン [243-266行目]
    │      │
    │      ├─ fs::read_dir()
    │      │      └─ エントリ列挙
    │      │
    │      ├─ fs::read_to_string()
    │      │      └─ メタデータ読み込み
    │      │
    │      └─ toml::from_str::<Value>()
    │             └─ TOMLパース
    │
    └─ repo.toml書き込み [268-274行目]
           │
           ├─ toml::to_string()
           │      └─ Repositoryシリアライズ
           │
           └─ File::create() + write_all()
                  └─ ファイル書き込み
```

### データフロー図

```
[入力]                      [処理]                      [出力]

既存repo.toml ──────────────┐
  packages: {...}           │
  outdated_packages: {...}  ├──▶ 既存データ読み込み
                           │         │
                           │         ▼
*.toml (メタデータ) ────────┼──▶ ディレクトリスキャン
  blake3: "..."            │         │
  version: "..."           │         ▼
                           │    データマージ
                           │         │
                           │         ├─ packages更新
                           │         └─ outdated_packages更新
                           │         │
outdated_packages ─────────┘         ▼
  (処理中に収集)                Repository構造体
                                     │
                                     ▼
                              TOMLシリアライズ
                                     │
                                     ▼
                              repo.toml ──▶ {repo}/{target}/repo.toml
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| repo_builder.rs | `src/bin/repo_builder.rs` | ソース | repo.toml生成のメイン処理 |
| lib.rs | `src/lib.rs` | ソース | 定数定義（WALK_DEPTH等） |
| 外部ライブラリ | pkg::package::Repository | 外部 | Repository構造体定義 |
| 外部ライブラリ | pkg::package::SourceIdentifier | 外部 | SourceIdentifier構造体定義 |
