# 機能設計書 25-古いパッケージ追跡

## 概要

本ドキュメントは、Redox OS Cookbookシステムにおける古いパッケージ追跡機能の設計を記述したものである。本機能は、ビルドに失敗したパッケージをoutdated_packagesとして記録し、リポジトリの健全性管理に活用する。

### 本機能の処理概要

古いパッケージ追跡機能は、ビルドプロセスで失敗したパッケージを追跡し、repo.tomlのoutdated_packagesセクションに記録する機能である。これにより、どのパッケージが最新のソースでビルドできなかったかを把握できる。

**業務上の目的・背景**：大規模なパッケージリポジトリでは、一部のパッケージがビルドに失敗することがある。このような失敗を追跡することで、メンテナーはどのパッケージに対応が必要かを把握でき、リポジトリ全体の品質管理が可能となる。また、パッケージマネージャは古いパッケージを警告付きで提供するか、インストールを拒否するかを判断できる。

**機能の利用シーン**：リポジトリ公開処理（機能No.23）において、Package::new_recursive_nonstopの結果からエラー情報を抽出する際に使用される。ビルドが成功したパッケージはoutdated_packagesから除外され、失敗したパッケージのみが記録される。

**主要な処理内容**：
1. パッケージリスト構築時のエラー情報収集
2. エラーが発生したパッケージのソース識別情報取得
3. outdated_packagesへの記録
4. ビルド成功時のoutdated_packagesからの除外
5. repo.tomlへの永続化

**関連システム・外部連携**：repo.toml生成（機能No.24）、ソース識別子管理（機能No.50）と連携。

**権限による制御**：特に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 10 | repo_builderコマンド画面 | 補助機能 | ビルド失敗パッケージのoutdated_packages記録 |

## 機能種別

状態管理 / エラー追跡

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| recipe_map | HashMap<PackageName, Result<Package, Error>> | Yes | パッケージ処理結果マップ | - |
| recipe_list | Vec<Package> | Yes | ビルド成功パッケージリスト | - |

### 入力データソース

- Package::new_recursive_nonstopの戻り値（成功リストとエラーマップ）
- 各レシピのソース識別情報（source_info.toml）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| outdated_packages | BTreeMap<String, SourceIdentifier> | 失敗パッケージとその識別情報 |

### 出力先

- repo.tomlのoutdated_packagesセクション

**出力形式例**：
```toml
[outdated_packages]
[outdated_packages.failed_package]
source_identifier = "abc123..."
commit_identifier = "def456..."
time_identifier = "2024-01-01T00:00:00Z"
```

## 処理フロー

### 処理シーケンス

```
1. パッケージリスト構築
   └─ Package::new_recursive_nonstopでエラー情報も収集
2. エラーパッケージのフィルタリング
   └─ recipe_mapからErr()のエントリを抽出
3. ソース識別情報の取得
   └─ fetch_get_source_infoまたはフォールバック値
4. outdated_packagesマップへの追加
5. 既存outdated_packagesとのマージ
   └─ 今回ビルド成功分は除外
6. repo.tomlへの永続化
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[new_recursive_nonstop実行]
    B --> C[recipe_listとrecipe_map取得]
    C --> D[recipe_mapからErr抽出]
    D --> E[各エラーパッケージをループ]
    E --> F[レシピパス検索]
    F --> G{見つかった?}
    G -->|No| H[スキップ]
    G -->|Yes| I[fetch_get_source_info呼び出し]
    H --> E
    I --> J{成功?}
    J -->|Yes| K[SourceIdentifier取得]
    J -->|No| L[フォールバック値設定]
    K --> M[outdated_packagesに追加]
    L --> M
    M --> N{次のパッケージ?}
    N -->|Yes| E
    N -->|No| O[既存outdated_packagesとマージ]
    O --> P[ビルド成功分を除外]
    P --> Q[repo.tomlに永続化]
    Q --> R[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-25-01 | エラー抽出 | recipe_mapからErr()のエントリのみを処理 | outdated_packages構築時 |
| BR-25-02 | 成功時除外 | 今回ビルドが成功したパッケージはoutdated_packagesから除外 | マージ処理時 |
| BR-25-03 | フォールバック識別子 | ソース情報取得失敗時は"missing_source"を使用 | ソース情報取得失敗時 |
| BR-25-04 | 永続化継承 | 今回処理しなかったパッケージのoutdated情報は既存repo.tomlから継承 | マージ処理時 |

### 計算ロジック

**フォールバック識別子の設定**：
```rust
let ident = get_ident();
outdated_packages.insert(
    recipe.name().to_string(),
    SourceIdentifier {
        source_identifier: "missing_source".to_string(),
        commit_identifier: ident.commit.clone(),
        time_identifier: ident.time.clone(),
    },
);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はファイルシステム操作のみであり、データベース操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | RecipeNotFound | レシピパスが見つからない | スキップして継続 |
| - | SourceInfoError | ソース識別情報取得失敗 | フォールバック値を使用 |

### リトライ仕様

特にリトライ処理は実装されていない。エラーが発生してもフォールバック値で処理を継続。

## トランザクション仕様

ファイルシステム操作のため、トランザクション制御は行われない。

## パフォーマンス要件

- 失敗パッケージ数に依存
- 通常は数秒以内で完了

## セキュリティ考慮事項

- outdated_packagesの情報はリポジトリの状態を示すのみで、セキュリティリスクは低い
- ただし、意図的に古いバージョンを使わせるような改ざんには注意

## 備考

- outdated_packagesはビルドシステムの健全性モニタリングに使用可能
- CI/CDパイプラインでのビルド失敗追跡に活用できる
- source_identifierが"missing_source"の場合、ソース取得自体が失敗していることを示す

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

SourceIdentifier構造体を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | 外部ライブラリ | pkg::package::SourceIdentifier | source/commit/time_identifierフィールド |

**読解のコツ**: SourceIdentifierはソースコードの識別情報（コミットハッシュ、タイムスタンプ等）を保持する。

#### Step 2: エラー収集を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | repo_builder.rs | `src/bin/repo_builder.rs` | new_recursive_nonstop呼び出し（84行目） |

**主要処理フロー**:
- **84行目**: Package::new_recursive_nonstopで(recipe_list, recipe_map)を取得
- recipe_listは成功パッケージ、recipe_mapはパッケージ名→Result<Package, Error>のマップ

#### Step 3: outdated_packages構築を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | repo_builder.rs | `src/bin/repo_builder.rs` | outdated_packages記録（177-215行目） |

**主要処理フロー**:
- **177-180行目**: recipe_mapからErr()のエントリをフィルタ
- **181-184行目**: エラーメッセージ出力
- **186-193行目**: レシピパス検索とCookRecipe読み込み
- **195-214行目**: fetch_get_source_info呼び出しとフォールバック処理

#### Step 4: マージと除外を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | repo_builder.rs | `src/bin/repo_builder.rs` | 既存outdated_packagesとのマージ（230-240行目） |

**主要処理フロー**:
- **230-234行目**: 今回ビルドしたパッケージのセット作成
- **235-239行目**: 今回ビルド成功していないパッケージのみ保持

### プログラム呼び出し階層図

```
publish_packages() [src/bin/repo_builder.rs:56]
    │
    ├─ Package::new_recursive_nonstop() [外部ライブラリ]
    │      └─ (recipe_list, recipe_map)を返却
    │
    ├─ エラーパッケージ処理 [177-215行目]
    │      │
    │      ├─ recipe_map.filter(Err)
    │      │      └─ エラーエントリ抽出
    │      │
    │      ├─ recipes::find()
    │      │      └─ レシピパス検索
    │      │
    │      ├─ CookRecipe::from_path()
    │      │      └─ レシピ読み込み
    │      │
    │      ├─ fetch_get_source_info()
    │      │      └─ ソース識別情報取得
    │      │
    │      └─ get_ident()
    │             └─ フォールバック識別子取得
    │
    └─ マージ処理 [230-240行目]
           └─ built_packagesでフィルタリング
```

### データフロー図

```
[入力]                      [処理]                      [出力]

new_recursive_nonstop ──────┐
  recipe_list (成功)        │
  recipe_map (全結果)       ├──▶ Errフィルタリング
                           │         │
                           │         ▼
                           │    エラーパッケージリスト
                           │         │
source_info.toml ──────────┼──▶ fetch_get_source_info
                           │         │
get_ident() ───────────────┼──▶ フォールバック識別子
                           │         │
                           │         ▼
                           │    SourceIdentifier
                           │         │
                           │         ▼
既存repo.toml ─────────────┼──▶ outdated_packagesマージ
  outdated_packages        │         │
                           │         ▼
recipe_list ───────────────┘    ビルド成功分除外
                                     │
                                     ▼
                              outdated_packages ──▶ repo.toml
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| repo_builder.rs | `src/bin/repo_builder.rs` | ソース | outdated_packages処理のメイン |
| fetch.rs | `src/cook/fetch.rs` | ソース | fetch_get_source_info関数 |
| ident.rs | `src/cook/ident.rs` | ソース | get_ident関数（フォールバック識別子） |
| 外部ライブラリ | pkg::package::SourceIdentifier | 外部 | SourceIdentifier構造体 |
