# 機能設計書 27-fetchコマンド

## 概要

本ドキュメントは、Redox OS Cookbookシステムにおけるfetchコマンドの設計を記述したものである。本機能は、レシピのソースコードをダウンロードする。

### 本機能の処理概要

fetchコマンドは、レシピに定義されたソース（Git、Tar、Path、SameAs）をダウンロードまたはコピーし、ビルド準備を行う機能である。

**業務上の目的・背景**：パッケージビルドの前段階として、ソースコードの取得が必要である。fetchコマンドにより、Git リポジトリのクローン、tarballのダウンロード・展開、パッチの適用などが自動化される。オフラインモードでの動作もサポートしており、事前にダウンロードしたソースを使用したビルドが可能。

**機能の利用シーン**：`repo fetch <recipe>` コマンドで明示的に実行するか、`repo cook <recipe>` コマンドの前処理として自動的に実行される。複数のレシピを指定して一括でfetchすることも可能。

**主要な処理内容**：
1. レシピのソースセクションの解析
2. ソースタイプに応じたダウンロード処理（Git clone、wget、ファイルコピー）
3. パッチの適用（patches指定がある場合）
4. ソース前処理スクリプトの実行（script指定がある場合）
5. Cargoプロジェクトの場合、cargo fetchの実行
6. ソース識別情報の記録

**関連システム・外部連携**：git、wget、tar、patch等の外部コマンドと連携。cookbook.tomlのmirrors設定によるミラーサーバー利用に対応。

**権限による制御**：特に権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | repo fetchコマンド画面 | 主機能 | レシピソースコードのダウンロード |
| 4 | repo cookコマンド画面 | API連携 | cookコマンドの前処理として実行 |

## 機能種別

ソース取得 / ファイル操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| recipe | CookRecipe | Yes | 対象レシピ | 有効なレシピ構造体 |
| logger | PtyOut | No | ログ出力先 | - |
| offline | bool | No | オフラインモード | COOKBOOK_OFFLINE環境変数 |

### 入力データソース

- recipe.tomlの[source]セクション
- cookbook.tomlのmirrors設定

**ソースタイプ別の入力**：
```toml
# Gitソース
[source]
git = "https://github.com/..."
branch = "main"
rev = "abc123"
patches = ["fix.patch"]

# Tarballソース
[source]
tar = "https://example.com/package-1.0.tar.gz"
blake3 = "hash..."
patches = ["fix.patch"]

# ローカルパス
[source]
path = "../local-source"

# 他レシピと共有
[source]
same_as = "../other-recipe"
```

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_dir | PathBuf | ソースディレクトリのパス |
| source_info.toml | File | ソース識別情報 |

### 出力先

- ソースディレクトリ：`{recipe_dir}/source/`
- ソース識別情報：`{recipe_dir}/target/{target}/source_info.toml`

## 処理フロー

### 処理シーケンス

```
1. ソースタイプの判定
   └─ Git / Tar / Path / SameAs / None
2. ソースタイプ別の処理
   └─ Git: clone/fetch/checkout
   └─ Tar: wget/extract
   └─ Path: copy
   └─ SameAs: symlink
   └─ None: ディレクトリ作成のみ
3. パッチ適用（patches指定時）
4. 前処理スクリプト実行（script指定時）
5. Cargo fetch（Cargoテンプレート時）
6. ソース識別情報の記録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{BuildKind?}
    B -->|None| C[空のソース識別情報記録]
    B -->|Remote| D[リモートパッケージ取得]
    B -->|Other| E{SourceRecipe?}
    C --> Z[終了]
    D --> Z
    E -->|Git| F[Git処理]
    E -->|Tar| G[Tar処理]
    E -->|Path| H[Path処理]
    E -->|SameAs| I[SameAs処理]
    E -->|None| J[ローカルソース]
    F --> K[パッチ適用]
    G --> K
    H --> K
    I --> L[ソース識別情報取得]
    J --> M[ソース識別情報記録]
    K --> N{Cargoテンプレート?}
    L --> Z
    N -->|Yes| O[cargo fetch]
    N -->|No| M
    O --> M
    M --> Z
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-27-01 | blake3検証 | Tarballはblake3ハッシュで検証する | Tarソース時 |
| BR-27-02 | shallow_clone | rev指定時はtreelessクローンを使用 | Git shallow_clone=true時 |
| BR-27-03 | ミラー置換 | cookbook.tomlのmirrors設定でURLを置換 | URLダウンロード時 |
| BR-27-04 | オフラインチェック | オフライン時は既存ソースの存在を確認 | COOKBOOK_OFFLINE=true時 |
| BR-27-05 | パッチ更新検出 | パッチファイルがソースより新しい場合は再適用 | Tarソース時 |

### 計算ロジック

**blake3ハッシュ検証**：
```rust
let source_tar_blake3 = get_blake3(&source_tar, true && logger.is_none())?;
if source_tar_blake3 != *blake3 {
    return Err("The downloaded tar blake3 is not equal to blake3 in recipe.toml");
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はファイルシステム操作のみであり、データベース操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Blake3Mismatch | blake3ハッシュ不一致 | recipe.tomlのハッシュ値確認 |
| - | GitCloneFailed | git clone失敗 | ネットワーク・URL確認 |
| - | TarExtractFailed | tar展開失敗 | アーカイブ形式確認 |
| - | PatchFailed | patch適用失敗 | パッチファイル確認 |
| - | OfflineSourceMissing | オフライン時にソースなし | 事前にfetch実行 |

### リトライ仕様

特にリトライ処理は実装されていない。エラー発生時はエラーメッセージを返却。

## トランザクション仕様

ファイルシステム操作のため、トランザクション制御は行われない。ただし、一時ディレクトリ（source.tmp）を使用してアトミックな移動を行う。

## パフォーマンス要件

- ネットワーク速度に依存
- 大規模リポジトリのクローンは数分かかる場合あり
- shallow_clone使用で高速化可能

## セキュリティ考慮事項

- blake3ハッシュによるTarball検証で改ざんを検出
- rev指定によるGitリビジョン固定で再現性確保
- ミラーサーバー設定の信頼性に注意

## 備考

- cargo fetchはCargoテンプレートの場合のみ実行される
- same_asはシンボリックリンクで実装される
- source.tmpは処理完了後にsourceにリネームされる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ソースレシピの定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe列挙型（18-67行目） |

**読解のコツ**: SourceRecipeはSameAs、Path、Git、Tarの4種類。それぞれ異なるフィールドを持つ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | repo.rs | `src/bin/repo.rs` | handle_fetch関数（553-566行目） |
| 2-2 | fetch.rs | `src/cook/fetch.rs` | fetch関数（117-406行目） |

**主要処理フロー**:
- **553-566行目（repo.rs）**: オフライン判定とfetch/fetch_offline呼び出し
- **117-131行目（fetch.rs）**: BuildKind::None、Remoteの特殊処理
- **133-393行目**: SourceRecipeタイプ別の処理

#### Step 3: Git処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | Gitソース処理（161-317行目） |

**主要処理フロー**:
- **172-198行目**: 初回クローン処理
- **199-249行目**: 既存リポジトリの更新判定
- **251-313行目**: checkout、submodule更新、パッチ適用

#### Step 4: Tar処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fetch.rs | `src/cook/fetch.rs` | Tarソース処理（318-380行目） |

**主要処理フロー**:
- **326-359行目**: ダウンロードとblake3検証
- **360-368行目**: パッチ更新検出
- **369-378行目**: 展開とパッチ適用

#### Step 5: ソース識別情報記録を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | fetch.rs | `src/cook/fetch.rs` | fetch_apply_source_info関数（648-670行目） |

**主要処理フロー**:
- **652-657行目**: SourceIdentifier構造体の構築
- **667-668行目**: source_info.tomlへの書き込み

### プログラム呼び出し階層図

```
repo_inner() [src/bin/repo.rs:286]
    │
    └─ handle_fetch() [src/bin/repo.rs:553]
           │
           ├─ fetch_offline() [src/cook/fetch.rs:37]
           │      └─ オフラインモード処理
           │
           └─ fetch() [src/cook/fetch.rs:117]
                  │
                  ├─ Git処理 [161-317行目]
                  │      ├─ git clone
                  │      ├─ git fetch
                  │      ├─ git checkout
                  │      ├─ git submodule update
                  │      └─ fetch_apply_patches()
                  │
                  ├─ Tar処理 [318-380行目]
                  │      ├─ download_wget()
                  │      ├─ get_blake3()
                  │      ├─ fetch_extract_tar()
                  │      └─ fetch_apply_patches()
                  │
                  ├─ Path処理 [141-159行目]
                  │      └─ copy_dir_all()
                  │
                  ├─ SameAs処理 [134-139行目]
                  │      ├─ fetch_resolve_canon()
                  │      └─ fetch_make_symlink()
                  │
                  ├─ fetch_cargo() [478-501行目]
                  │      └─ cargo fetch
                  │
                  └─ fetch_apply_source_info() [648-659行目]
                         └─ source_info.toml書き込み
```

### データフロー図

```
[入力]                      [処理]                      [出力]

recipe.toml ────────────────┐
  [source]                  │
    git/tar/path/same_as    ├──▶ ソースタイプ判定
                           │         │
                           │         ▼
                           │    タイプ別処理
                           │         │
Git URL ───────────────────┼──▶ git clone/fetch ──▶ source/
                           │         │
Tar URL ───────────────────┼──▶ wget/extract ──▶ source/
                           │         │
Local Path ────────────────┼──▶ copy_dir_all ──▶ source/
                           │         │
Same As ───────────────────┼──▶ symlink ──▶ source/
                           │         │
patches/*.patch ───────────┼──▶ patch適用
                           │         │
cookbook.toml ─────────────┘    ミラー置換
  mirrors                        │
                                 ▼
                           fetch_apply_source_info
                                 │
                                 ▼
                           source_info.toml ──▶ target/{target}/
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| repo.rs | `src/bin/repo.rs` | ソース | CLIエントリーポイント、handle_fetch関数 |
| fetch.rs | `src/cook/fetch.rs` | ソース | fetchメイン処理 |
| recipe.rs | `src/recipe.rs` | ソース | SourceRecipe列挙型定義 |
| fs.rs | `src/cook/fs.rs` | ソース | download_wget、run_command等 |
| config.rs | `src/config.rs` | ソース | translate_mirror関数 |
