# 機能設計書 3-ソース取得（Tar）

## 概要

本ドキュメントは、Redox OSビルドシステムにおけるTarアーカイブからのソースコード取得機能の設計を記載する。TarアーカイブのダウンロードとBLAKE3ハッシュによる整合性検証について詳述する。

### 本機能の処理概要

ソース取得（Tar）機能は、recipe.tomlで指定されたURLからTarアーカイブをダウンロードし、BLAKE3ハッシュで整合性を検証した上で展開する機能である。GNU系ライブラリなど外部プロジェクトのソースコード取得に広く使用される。

**業務上の目的・背景**：多くのオープンソースプロジェクトは、リリース時にTar形式のソースアーカイブを提供している。これらを効率的にダウンロードし、改ざんや破損がないことを暗号学的ハッシュで検証することで、セキュアで再現可能なビルドを実現する。

**機能の利用シーン**：
- GNU系ライブラリ（glibc, bash, coreutils等）のソース取得
- バージョン固定された外部ライブラリの取得
- ネットワーク帯域が限られる環境でのキャッシュ活用
- オフラインビルド環境での事前ダウンロード済みTarの使用

**主要な処理内容**：
1. Tarアーカイブのダウンロード（wgetコマンド使用）
2. BLAKE3ハッシュによる整合性検証
3. Tarアーカイブの展開（stripディレクトリ対応）
4. パッチファイルの適用（存在する場合）
5. 準備スクリプトの実行（存在する場合）

**関連システム・外部連携**：
- wgetコマンド：HTTPダウンロード
- tarコマンド：アーカイブ展開
- BLAKE3ライブラリ：ハッシュ計算
- ミラーサーバー：cookbook.tomlで設定されたミラーURL変換

**権限による制御**：特になし（ファイルシステムの書き込み権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | repo fetchコマンド画面 | API連携 | Tarアーカイブのダウンロードと展開 |

## 機能種別

データ連携 / 外部コマンド実行 / バリデーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| tar | String | Yes | TarアーカイブのダウンロードURL | 有効なURL形式 |
| blake3 | Option<String> | No（推奨） | 期待するBLAKE3ハッシュ値 | 64文字の16進数文字列 |
| patches | Vec<String> | No | 適用するパッチファイルのリスト | ファイルが存在すること |
| script | Option<String> | No | ソース準備スクリプト | - |

### 入力データソース

- recipe.toml: sourceセクションのTar設定
- cookbook.toml: ミラー設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_dir | PathBuf | ソースコードが配置されたディレクトリパス |
| source_identifier | String | Tarファイルのblake3ハッシュ（または"no_tar_blake3_hash_info"） |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/source.tar`（キャッシュ）
- ファイルシステム：`{レシピディレクトリ}/source/`（展開後）

## 処理フロー

### 処理シーケンス

```
1. Tarアーカイブの存在確認とダウンロード
   ├─ source.tarファイルの存在チェック
   ├─ 存在しない場合：wgetでダウンロード
   │   └─ URLをミラーサーバーに変換
   └─ source.tar.tmpに一時保存後リネーム
2. BLAKE3ハッシュ検証
   ├─ source.tarのBLAKE3ハッシュを計算
   ├─ recipe.tomlのblake3値と比較
   └─ 不一致の場合：エラー（新規DL時）または再ダウンロード
3. 展開要否判定
   ├─ sourceディレクトリの存在確認
   ├─ Tarファイルまたはパッチが更新されている場合
   └─ sourceディレクトリを削除して再展開
4. Tarアーカイブの展開
   ├─ source.tmpディレクトリを作成
   ├─ tar --extract --strip-components=1
   └─ source.tmpをsourceにアトミックリネーム
5. パッチ・スクリプト適用
   ├─ patchesの適用（存在する場合）
   └─ scriptの実行（存在する場合）
6. ソース情報の記録
   └─ source_info.tomlに識別子を保存
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{source.tarが存在するか}
    B -->|No| C[wgetでダウンロード]
    B -->|Yes| D[BLAKE3ハッシュ計算]
    C --> D
    D --> E{blake3が指定されているか}
    E -->|Yes| F{ハッシュが一致するか}
    E -->|No| G[警告: blake3の設定を推奨]
    F -->|Yes| H{sourceディレクトリが存在するか}
    F -->|No| I{新規ダウンロードか}
    I -->|Yes| J[エラー: ハッシュ不一致]
    I -->|No| K[source.tarを削除して再ダウンロード]
    K --> C
    G --> H
    H -->|Yes| L{更新が必要か}
    H -->|No| M[source.tmpを作成]
    L -->|Yes| N[sourceディレクトリを削除]
    L -->|No| O[展開をスキップ]
    N --> M
    M --> P[tar --extract実行]
    P --> Q[source.tmpをsourceにリネーム]
    Q --> R{パッチが存在するか}
    O --> R
    R -->|Yes| S[パッチを適用]
    R -->|No| T{スクリプトが存在するか}
    S --> T
    T -->|Yes| U[スクリプトを実行]
    T -->|No| V[source_info.tomlを保存]
    U --> V
    V --> W[終了]
    J --> W
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ミラーURL変換 | cookbook.tomlで設定されたミラーがあればURLを変換 | 常時 |
| BR-02 | BLAKE3必須推奨 | blake3が未指定の場合は警告を出力し、計算値を表示 | blake3未指定時 |
| BR-03 | ハッシュ不一致時の処理 | 新規DL時はエラー、既存Tar時は再ダウンロード | ハッシュ不一致時 |
| BR-04 | strip-components | 展開時にトップレベルディレクトリを除去 | 常時 |
| BR-05 | パッチ更新検出 | パッチファイルがsourceより新しい場合は再展開 | patches存在時 |
| BR-06 | オフラインモード | source.tarが存在し、blake3検証OKなら展開のみ実行 | COOKBOOK_OFFLINE時 |

### 計算ロジック

**BLAKE3ハッシュ計算**：
```
BLAKE3暗号学的ハッシュ関数を使用
出力: 256ビット（64文字の16進数表現）
進捗表示: 大きなファイルの場合はプログレスバー付き
```

**バージョン推測ロジック**：
```
正規表現パターン: \d+\.\d+\.\d+
TarソースのURLからセマンティックバージョン形式（X.Y.Z）を抽出
例: "http://ftp.gnu.org/gnu/bash/bash-5.2.tar.gz" → "5.2"（該当なしの場合はNone）
```

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムとシェルコマンドのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ダウンロードエラー | wgetコマンド失敗 | ネットワーク接続確認、URL確認 |
| - | ハッシュ不一致 | 計算値とrecipe.tomlのblake3が異なる | 正しいハッシュ値を設定 |
| - | 展開エラー | tarコマンド失敗 | アーカイブ形式の確認 |
| - | オフラインエラー | オフラインモードでsource.tarが存在しない | 事前にsource.tarをダウンロード |

### リトライ仕様

- 既存のsource.tarのハッシュが不一致の場合、自動的に削除して再ダウンロード
- TUIモードではユーザーにリトライ/スキップ/終了を選択させる

## トランザクション仕様

- source.tar.tmpへの一時ダウンロード後にsource.tarへアトミックリネーム
- source.tmpへの一時展開後にsourceへアトミックリネーム
- 失敗時は一時ファイルを削除

## パフォーマンス要件

- BLAKE3ハッシュ計算時のプログレスバー表示（大きなファイル向け）
- ダウンロード時間はネットワーク帯域とファイルサイズに依存

## セキュリティ考慮事項

- BLAKE3ハッシュによる整合性検証：改ざん・破損の検出
- ミラーURL変換：信頼できるミラーサーバーのみ設定すること
- HTTPS優先：HTTPはHTTPSに自動アップグレード

## 備考

- BLAKE3はSHA-256より高速かつセキュア
- --strip-components=1はトップレベルディレクトリを除去（例: package-1.0/src → src）
- オフラインモードではblake3指定がない場合、信頼を前提に処理を中断

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Tarソース設定を表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe::Tar variantの定義 |

**読解のコツ**:
- `#[serde(untagged)]`によりTOMLのキー名から自動的にTarタイプと判定される
- tarフィールドが存在すればSourceRecipe::Tarとしてデシリアライズされる

**主要処理フロー**:
- **55-66行目**: SourceRecipe::Tar variant定義
  - tar: String - ダウンロードURL
  - blake3: Option<String> - 期待するBLAKE3ハッシュ
  - patches: Vec<String> - パッチファイルリスト
  - script: Option<String> - 準備スクリプト

#### Step 2: エントリーポイントを理解する

ソース取得処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch()関数のTar処理部分 |

**主要処理フロー**:
- **318-380行目**: Tar処理ブロック
  - **324-359行目**: ダウンロードとハッシュ検証ループ
  - **360-378行目**: 展開処理

#### Step 3: ダウンロードとハッシュ検証を理解する

ダウンロードとBLAKE3ハッシュ検証の詳細を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | get_blake3()関数 |
| 3-2 | blake3.rs | `src/blake3.rs` | blake3_progress(), blake3_silent() |
| 3-3 | fs.rs | `src/cook/fs.rs` | download_wget()関数 |

**主要処理フロー**:
- **21-35行目** (fetch.rs): get_blake3() - BLAKE3ハッシュ計算のラッパー
- **6-23行目** (blake3.rs): blake3_progress() - プログレスバー付きハッシュ計算
- **25-31行目** (blake3.rs): blake3_silent() - サイレントハッシュ計算
- **278-288行目** (fs.rs): download_wget() - wgetによるダウンロード

#### Step 4: Tar展開処理を理解する

Tarアーカイブの展開処理を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fetch.rs | `src/cook/fetch.rs` | fetch_extract_tar()関数 |

**主要処理フロー**:
- **454-476行目**: fetch_extract_tar()関数
  - tarコマンドの構築
  - --extract, --no-same-owner, --strip-components=1
  - Redox OS上での動作差異対応

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    └─ SourceRecipe::Tar処理
           │
           ├─ ダウンロードループ（while）
           │   │
           │   ├─ download_wget(tar_url, source_tar, logger)
           │   │   └─ Command::new("wget")
           │   │       ├─ translate_mirror(url)
           │   │       ├─ --continue
           │   │       └─ -O source.tar.tmp
           │   │
           │   └─ get_blake3(source_tar, show_progress)
           │       └─ blake3_progress() / blake3_silent()
           │           └─ Hasher::new().update_reader()
           │
           ├─ 展開要否判定
           │   └─ fetch_is_patches_newer()
           │
           ├─ fetch_extract_tar(source_tar, source_dir_tmp, logger)
           │   └─ Command::new("tar")
           │       ├─ --extract
           │       ├─ --no-same-owner
           │       ├─ --strip-components=1
           │       └─ --directory source.tmp
           │
           └─ fetch_apply_patches(patches, script, source_dir, logger)
               ├─ Command::new("patch")
               └─ Command::new("bash")
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ source.tar ─────────▶ translate_mirror() ─────────────▶ Mirror URL
  └─ source.blake3                  │
  └─ source.patches                 ▼
  └─ source.script           download_wget() ──────────────▶ source.tar
                                    │
                                    ▼
                            get_blake3() ──────────────────▶ 計算されたハッシュ
                                    │
                                    ▼
                            blake3比較 ─────────────────────▶ 検証結果（OK/NG）
                                    │
                                    ▼
                            fetch_extract_tar() ───────────▶ source/
                                    │
                                    ▼
                            fetch_apply_patches() ─────────▶ パッチ適用済みソース
                                    │
                                    ▼
                            fetch_apply_source_info() ─────▶ source_info.toml
                                    │
                                    └─ source_identifier: blake3ハッシュ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | Tar取得のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | SourceRecipe::Tar定義 |
| blake3.rs | `src/blake3.rs` | ソース | BLAKE3ハッシュ計算 |
| fs.rs | `src/cook/fs.rs` | ソース | download_wget()関数 |
| progress_bar.rs | `src/progress_bar.rs` | ソース | プログレスバー表示 |
| config.rs | `src/config.rs` | ソース | translate_mirror()関数 |
| recipe.toml | 各レシピディレクトリ | 設定 | Tarソース設定 |
| cookbook.toml | プロジェクトルート | 設定 | ミラー設定 |
