# 機能設計書 32-treeコマンド

## 概要

本ドキュメントは、cookbookシステムにおける`tree`コマンド機能の設計を記述する。`tree`コマンドは、パッケージの依存関係をツリー形式で視覚的に表示し、イメージサイズの推定を行う機能を提供する。

### 本機能の処理概要

`tree`コマンドは、指定されたパッケージとその依存関係をツリー構造で表示し、各パッケージのサイズと合計サイズを計算して表示する機能である。

**業務上の目的・背景**：Redox OSの開発において、パッケージの依存関係を把握することはシステム構成の理解とデバッグに不可欠である。また、最終的なシステムイメージのサイズを事前に見積もることで、ディスク容量の計画やパッケージの取捨選択に役立てることができる。本機能は、複雑な依存関係を可視化し、イメージサイズの推定値を提供することで、開発者の意思決定を支援する。

**機能の利用シーン**：
- パッケージの依存関係構造を確認したい場合
- システムイメージのサイズを事前に見積もりたい場合
- 特定のパッケージが引き込む依存パッケージを調査したい場合
- 重複する依存関係を特定したい場合

**主要な処理内容**：
1. 対象パッケージの依存関係を再帰的に解決
2. 依存関係ツリーの深さ優先走査
3. 各パッケージのpkgarファイルサイズを取得
4. ツリー構造と各パッケージサイズを標準出力に表示
5. 合計サイズとパッケージ総数を計算・表示

**関連システム・外部連携**：
- ファイルシステム：pkgarアーカイブファイルのメタデータ取得
- stage.toml：パッケージの依存関係情報の取得

**権限による制御**：本機能は情報表示のみであり、特別な権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | repo treeコマンド画面 | 主機能 | パッケージ依存関係ツリーの表示 |
| 12 | ヘルプ画面 | 補助機能 | treeコマンドの説明表示 |

## 機能種別

情報表示（依存関係ツリーとサイズ情報の可視化）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| recipe_names | Vec<PackageName> | Yes | 表示対象のルートパッケージ名リスト | 有効なパッケージ名形式 |
| --with-package-deps | bool | No | パッケージ依存関係を含めるか | - |
| --cookbook | PathBuf | No | cookbookディレクトリ | 有効なディレクトリパス |
| --repo | PathBuf | No | リポジトリディレクトリ | 有効なディレクトリパス |

### 入力データソース

- コマンドライン引数
- レシピファイル（recipe.toml）
- ステージ情報ファイル（stage.toml）
- pkgarアーカイブファイルのメタデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ツリー表示 | String | 依存関係のツリー構造表示 |
| パッケージサイズ | String | 各パッケージのサイズ（人間可読形式） |
| 合計サイズ | String | 推定イメージサイズ（人間可読形式） |
| パッケージ数 | usize | 総パッケージ数 |

### 出力先

標準出力

### 出力形式例

```
├── kernel [2.50 MiB]
│   └── relibc [1.20 MiB]
├── init [512.00 KiB]
│   ├── relibc
│   └── redox_syscall [128.00 KiB]
└── drivers [3.00 MiB]

Estimated image size: 7.32 MiB of 5 packages
```

## 処理フロー

### 処理シーケンス

```
1. コマンドライン引数解析
   └─ 対象パッケージリストを取得

2. 依存関係の解決
   └─ with-package-deps指定時は依存関係を再帰的に収集

3. recipe_mapの構築
   └─ パッケージ名からCookRecipeへのマッピング作成

4. ルートパッケージごとにツリー表示
   ├─ display_tree_entry呼び出し
   ├─ 再帰的に依存関係を走査
   ├─ 各パッケージのサイズ取得と累積
   └─ ツリー形式で標準出力に出力

5. サマリー出力
   └─ 合計サイズとパッケージ数を表示
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[parse_args]
    B --> C[recipe_map構築]
    C --> D{ルートパッケージあり?}
    D -->|No| E[サマリー出力]
    D -->|Yes| F[display_tree_entry]
    F --> G{訪問済み?}
    G -->|Yes| H[重複表示]
    G -->|No| I{pkgar存在?}
    I -->|Yes| J[サイズ取得・表示]
    I -->|No| K[not built表示]
    J --> L[依存関係取得]
    K --> L
    H --> M[次のルートへ]
    L --> N{依存パッケージあり?}
    N -->|Yes| O[子の再帰処理]
    N -->|No| M
    O --> N
    M --> D
    E --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-32-1 | 重複検出 | 既に表示したパッケージは重複として扱い、サイズ集計から除外 | 常時 |
| BR-32-2 | 未ビルド表示 | pkgarファイルが存在しないパッケージは"(not built)"と表示 | pkgarファイル不在時 |
| BR-32-3 | サイズ形式 | サイズはB/KiB/MiB/GiB/TiBで表示 | 常時 |
| BR-32-4 | ツリー記号 | 最後の要素は"└──"、それ以外は"├──"を使用 | 常時 |

### 計算ロジック

**サイズフォーマット計算（format_size関数）**：
```rust
if bytes == 0 { return "0 B" }
const UNITS: [&str; 5] = ["B", "KiB", "MiB", "GiB", "TiB"];
let i = (bytes as f64).log(1024.0).floor() as usize;
let size = bytes as f64 / 1024.0_f64.powi(i as i32);
format!("{:.2} {}", size, UNITS[i])
```

## データベース操作仕様

本機能はデータベースを使用しない。

### 操作別データベース影響一覧

該当なし（ファイルシステムからの読み取りのみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Missing | 依存関係情報が取得できない | 依存パッケージのビルドを実行 |
| - | パース失敗 | stage.tomlの解析エラー | stage.tomlの形式を確認 |

### リトライ仕様

本機能は情報表示のみのため、リトライ処理なし。

## トランザクション仕様

読み取り専用のため、トランザクション制御なし。

## パフォーマンス要件

- 大量のパッケージでも高速にツリーを構築・表示できること
- 既に訪問したパッケージはスキップし、無限ループを防止

## セキュリティ考慮事項

- 情報表示のみであり、特別なセキュリティ考慮は不要
- ファイルシステムへの読み取りアクセスのみ

## 備考

- 依存関係の重複は最初の出現箇所でのみサイズを計上し、以降は空文字列で表示
- stage.tomlが存在しない場合はrecipe.tomlのpackage.dependenciesを使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tree.rs | `src/cook/tree.rs` | WalkTreeEntry列挙型（12-17行目）：ノードの状態定義 |
| 1-2 | recipe.rs | `src/recipe.rs` | CookRecipe構造体（196-204行目）：パッケージ情報 |

**読解のコツ**: `WalkTreeEntry`の4つの状態（Built/NotBuilt/Deduped/Missing）がツリー表示の分岐条件となる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | repo.rs | `src/bin/repo.rs` | handle_tree関数への分岐（205-206行目） |
| 2-2 | repo.rs | `src/bin/repo.rs` | handle_tree関数（735-766行目） |

**主要処理フロー**:
1. **735-741行目**: recipe_mapの構築とルートパッケージの抽出
2. **742-751行目**: 各ルートに対するdisplay_tree_entry呼び出し
3. **753-765行目**: サマリー出力

#### Step 3: ツリー表示処理の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tree.rs | `src/cook/tree.rs` | display_tree_entry関数（19-36行目） |
| 3-2 | tree.rs | `src/cook/tree.rs` | walk_tree_entry関数（38-107行目）：再帰的走査 |
| 3-3 | tree.rs | `src/cook/tree.rs` | display_pkg_fn関数（109-124行目）：表示処理 |

**主要処理フロー**:
- **47-54行目**: recipe_mapからパッケージを取得、なければMissing
- **56-63行目**: pkgarファイル存在確認と状態決定
- **72-74行目**: サイズの累積計算
- **77-105行目**: 依存関係の再帰処理

#### Step 4: サイズフォーマットを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | tree.rs | `src/cook/tree.rs` | format_size関数（166-174行目） |

### プログラム呼び出し階層図

```
main (repo.rs:174)
    │
    └─ main_inner (repo.rs:182)
           │
           ├─ parse_args (repo.rs:373)
           │
           └─ handle_tree (repo.rs:735)
                  │
                  └─ display_tree_entry (tree.rs:19)
                         │
                         └─ walk_tree_entry (tree.rs:38)
                                │
                                ├─ display_pkg_fn (tree.rs:109)
                                │      └─ format_size (tree.rs:166)
                                │
                                └─ walk_tree_entry [再帰]
```

### データフロー図

```
[入力]                    [処理]                         [出力]

コマンドライン引数 ───▶ parse_args ───▶ CliConfig, recipes
                                          │
                                          ▼
recipe.toml ───────────▶ recipe_map構築 ───▶ HashMap<PackageName, CookRecipe>
                                          │
                                          ▼
pkgar/stage.toml ──────▶ walk_tree_entry ───▶ ツリー表示 ───▶ 標準出力
                                │
                                ▼
                         format_size ───▶ サマリー出力 ───▶ 標準出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| repo.rs | `src/bin/repo.rs` | ソース | treeコマンドのエントリーポイント |
| tree.rs | `src/cook/tree.rs` | ソース | ツリー走査・表示・サイズ計算のコアロジック |
| recipe.rs | `src/recipe.rs` | ソース | CookRecipe構造体と依存関係情報 |
| stage.toml | `recipes/*/target/*/stage.toml` | 設定 | パッケージ依存関係情報 |
