# 機能設計書 4-ソース取得（パス参照）

## 概要

本ドキュメントは、Redox OSビルドシステムにおけるローカルファイルシステムからのソースコード取得機能の設計を記載する。ローカルパスからソースをコピーする処理について詳述する。

### 本機能の処理概要

ソース取得（パス参照）機能は、recipe.tomlで指定されたローカルファイルシステムのパスからソースコードをコピーする機能である。ネットワークアクセスを必要とせず、ローカルで管理されているソースコードを使用する場合に利用される。

**業務上の目的・背景**：Redox OSのコアコンポーネントやカーネル関連のソースコードは、redoxリポジトリ内にサブディレクトリとして配置されている場合がある。また、開発中のパッケージをローカルソースからビルドしたい場合にも使用される。外部からのダウンロードを必要とせず、ローカルファイルシステムからソースを取得することで、オフライン環境でのビルドや高速なビルドサイクルを実現する。

**機能の利用シーン**：
- Redox OSコアコンポーネント（kernel, drivers等）のビルド
- 開発中のパッケージのローカルソースからのビルド
- 外部ネットワークアクセスが制限された環境でのビルド
- 高速なビルドサイクルが必要な開発シナリオ

**主要な処理内容**：
1. 指定されたローカルパスの存在確認
2. 更新検出（ソースパスとsourceディレクトリの更新日時比較）
3. ソースディレクトリへの再帰的コピー

**関連システム・外部連携**：
- ファイルシステム：ローカルパスからのファイルコピー

**権限による制御**：特になし（ファイルシステムの読み取り・書き込み権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | repo fetchコマンド画面 | API連携 | ローカルファイルシステムからのソースコピー |

## 機能種別

データ連携 / ファイル操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | String | Yes | コピー元のローカルパス | パスが存在すること |

### 入力データソース

- recipe.toml: sourceセクションのPath設定
- ローカルファイルシステム: 指定されたパスのファイル/ディレクトリ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_dir | PathBuf | ソースコードが配置されたディレクトリパス |
| source_identifier | String | 固定値 "local_source" |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/source/`

## 処理フロー

### 処理シーケンス

```
1. 更新検出
   ├─ sourceディレクトリが存在しない場合：コピー実行
   └─ 存在する場合：指定パスとsourceの更新日時を比較
       └─ 指定パスが新しい場合：コピー実行
2. ソースコピー
   ├─ copy_dir_all()で再帰的にコピー
   └─ ディレクトリ構造を保持してコピー
3. ソース情報の記録
   └─ source_info.tomlに"local_source"を保存
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{sourceディレクトリが存在するか}
    B -->|No| C[copy_dir_all実行]
    B -->|Yes| D{指定パスがsourceより新しいか}
    D -->|Yes| E[DEBUG: 更新メッセージ出力]
    D -->|No| F[コピーをスキップ]
    E --> C
    C --> G[source_info.tomlを保存]
    F --> G
    G --> H[source_dirを返却]
    H --> I[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 増分更新 | 指定パスがsourceより新しい場合のみコピー | sourceディレクトリが存在する場合 |
| BR-02 | 完全コピー | ディレクトリ構造を保持して再帰的にコピー | コピー実行時 |
| BR-03 | 識別子固定 | source_identifierは常に"local_source" | 常時 |

### 計算ロジック

**更新検出ロジック**：
```
modified_dir(path) > modified_dir(source_dir)
の場合にコピーを実行

modified_dir(): ディレクトリ内の全ファイルの最新更新日時を取得
```

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | パスエラー | 指定されたパスが存在しない | パスの設定を確認 |
| - | コピーエラー | ファイルコピーに失敗 | 書き込み権限を確認 |

### リトライ仕様

リトライは行わない（ファイルコピーエラーは即時返却）

## トランザクション仕様

特になし（コピー処理は非アトミック）

## パフォーマンス要件

- ローカルファイルシステム内でのコピーのため高速
- 大きなソースツリーでは初回コピーに時間がかかる

## セキュリティ考慮事項

- パストラバーサル攻撃の防止：相対パスの検証

## 備考

- このソースタイプはネットワークアクセスを必要としない
- オフラインモードでも動作する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、パスソース設定を表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe::Path variantの定義 |

**読解のコツ**:
- `#[serde(untagged)]`によりTOMLのキー名から自動的にPathタイプと判定される
- pathフィールドが存在すればSourceRecipe::Pathとしてデシリアライズされる

**主要処理フロー**:
- **29-33行目**: SourceRecipe::Path variant定義
  - path: String - コピー元のローカルパス

#### Step 2: エントリーポイントを理解する

ソース取得処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch()関数のPath処理部分 |

**主要処理フロー**:
- **141-159行目**: Path処理ブロック
  - sourceディレクトリの存在確認
  - 更新日時の比較
  - copy_dir_all()の呼び出し

#### Step 3: ファイルコピー処理を理解する

再帰的なディレクトリコピーの実装を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fs.rs | `src/cook/fs.rs` | copy_dir_all()関数、modified_dir()関数 |

**主要処理フロー**:
- **52-64行目** (fs.rs): copy_dir_all() - 再帰的なディレクトリコピー
- **163-172行目** (fs.rs): modified_dir() - ディレクトリの更新日時取得

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    └─ SourceRecipe::Path処理
           │
           ├─ 更新検出
           │   ├─ source_dir.is_dir()
           │   ├─ modified_dir(path)
           │   └─ modified_dir(source_dir)
           │
           └─ copy_dir_all(path, source_dir)
               │
               ├─ fs::create_dir_all(dst)
               │
               └─ for entry in fs::read_dir(src)
                   ├─ is_dir → 再帰呼び出し
                   └─ is_file → fs::copy()
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ source.path ────────▶ 存在確認 ─────────────────────▶ パス検証OK
                                  │
                                  ▼
                          更新日時比較 ───────────────────▶ 更新要否判定
                                  │
                                  ▼
                          copy_dir_all() ─────────────────▶ source/
                                  │
                                  ▼
                          fetch_apply_source_info() ──────▶ source_info.toml
                                  │
                                  └─ source_identifier: "local_source"
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | Path取得のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | SourceRecipe::Path定義 |
| fs.rs | `src/cook/fs.rs` | ソース | copy_dir_all(), modified_dir()関数 |
| recipe.toml | 各レシピディレクトリ | 設定 | Pathソース設定 |
