# 機能設計書 40-ミラーURL変換

## 概要

本ドキュメントは、cookbookシステムにおける`ミラーURL変換`機能の設計を記述する。この機能は、cookbook.tomlの[mirrors]設定に基づいてソースURLをミラーサイトURLに自動変換する機能を提供する。

### 本機能の処理概要

ミラーURL変換機能は、git clone/fetchやtarダウンロード時に使用されるURLを、cookbook.tomlの[mirrors]セクションで定義されたルールに従ってミラーサイトのURLに変換する機能である。

**業務上の目的・背景**：外部のgitリポジトリやtarアーカイブをダウンロードする際、ネットワーク環境によっては外部サイトへのアクセスが制限されていたり、低速であったりする場合がある。組織内にミラーサイトを設置することで、ダウンロード速度の向上やネットワークトラフィックの削減が可能になる。また、外部サイトの一時的な障害時にもビルドを継続できる。

**機能の利用シーン**：
- 社内ネットワークからのアクセス制限がある場合
- ミラーサイトを使用してダウンロード速度を向上させたい場合
- 外部サイトの障害時にミラーサイトにフォールバックしたい場合
- 地理的に近いミラーサイトを使用したい場合

**主要な処理内容**：
1. cookbook.tomlの[mirrors]セクションの読み込み
2. 対象URLとミラールールの照合（前方一致）
3. マッチしたルールに基づくURL置換
4. 変換後URLの返却

**関連システム・外部連携**：
- gitコマンド：クローン/フェッチ時のURL
- wgetコマンド：tarダウンロード時のURL

**権限による制御**：本機能に特別な権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ソース取得時のURL変換（内部処理） |

## 機能種別

設定管理（URL変換）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| url | &str | Yes | 変換対象のURL | 有効なURL文字列 |

### 入力データソース

- 変換対象URL（gitリポジトリURLまたはtarダウンロードURL）
- cookbook.tomlの[mirrors]セクション

### 設定ファイル形式

```toml
[mirrors]
# キー: 元のURLプレフィックス
# 値: 置換後のURLプレフィックス
"https://github.com/" = "https://my-mirror.com/github/"
"https://gitlab.redox-os.org/" = "https://my-mirror.com/redox/"
```

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 変換後URL | String | ミラーURL変換後の文字列 |

### 出力先

関数の戻り値として返却

### 変換例

| 元URL | ミラー設定 | 変換後URL |
|-------|-----------|----------|
| `https://github.com/user/repo.git` | `"https://github.com/" = "https://mirror.local/"` | `https://mirror.local/user/repo.git` |
| `https://gitlab.redox-os.org/redox-os/ion.git` | `"https://gitlab.redox-os.org/" = "https://internal.com/redox/"` | `https://internal.com/redox/redox-os/ion.git` |

## 処理フロー

### 処理シーケンス

```
1. translate_mirror関数の呼び出し
   └─ 変換対象URLを受け取る

2. ミラー設定の取得
   └─ get_config().mirrors から HashMap を取得

3. 各ミラールールのチェック
   ├─ URLがルールのキー（プレフィックス）で始まるかチェック
   └─ マッチした場合、プレフィックスを値で置換

4. 結果の返却
   ├─ マッチあり：変換後URL
   └─ マッチなし：元のURL
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[translate_mirror呼び出し]
    B --> C[ミラー設定取得]
    C --> D{mirrors設定あり?}
    D -->|No| E[元URLを返却]
    D -->|Yes| F[ミラールールループ]
    F --> G{URLがプレフィックスで始まる?}
    G -->|Yes| H[プレフィックスを置換]
    H --> I[変換後URLを返却]
    G -->|No| J{次のルールあり?}
    J -->|Yes| F
    J -->|No| E
    E --> K[終了]
    I --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-1 | 前方一致 | URLがキーで始まる場合のみ置換 | 常時 |
| BR-40-2 | 最初のマッチ | 複数ルールがマッチする場合、最初にマッチしたものを使用 | 複数マッチ時 |
| BR-40-3 | 未設定時スキップ | mirrors設定がない場合は変換なし | 設定なし時 |
| BR-40-4 | 完全置換 | マッチしたプレフィックス部分のみを置換 | マッチ時 |

### 計算ロジック

**URL変換ロジック（translate_mirror関数）**：
```rust
pub fn translate_mirror(url: &str) -> String {
    for (key, value) in get_config().mirrors.iter() {
        if url.starts_with(key.as_str()) {
            return url.replacen(key.as_str(), value.as_str(), 1);
        }
    }
    url.to_string()
}
```

## データベース操作仕様

本機能はデータベースを使用しない。

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

本機能は明示的なエラーを発生させない（マッチしない場合は元URLを返す）

### リトライ仕様

リトライなし（単純な文字列変換）

## トランザクション仕様

トランザクション制御なし

## パフォーマンス要件

- URL変換は即座に完了すること
- ミラー設定の参照は効率的に行われること（HashMapを使用）

## セキュリティ考慮事項

- ミラーサイトのURLが信頼できるものであることを確認する必要がある
- 不正なミラー設定により意図しないサイトからダウンロードされる可能性がある

## 備考

- HashMapの順序は保証されないため、複数のルールがマッチする可能性がある場合は注意が必要
- git URLの.git拡張子は変換に影響しない
- http/httpsの両方のURLに対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | config.rs | `src/config.rs` | MirrorsConfig型（40行目）：HashMap<String, String> |
| 1-2 | config.rs | `src/config.rs` | CookbookConfig.mirrorsフィールド（67行目） |

**読解のコツ**: `MirrorsConfig`はキーが元URLプレフィックス、値が置換後プレフィックスのHashMap。

#### Step 2: 変換関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | config.rs | `src/config.rs` | translate_mirror関数（150-157行目） |

**主要処理フロー**:
1. **151行目**: `get_config().mirrors.iter()` でミラー設定を取得
2. **152行目**: `url.starts_with(key.as_str())` で前方一致チェック
3. **153行目**: `url.replacen(key.as_str(), value.as_str(), 1)` で置換
4. **156行目**: マッチしなければ元URLを返す

#### Step 3: 使用箇所を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | git clone時のURL変換（182行目） |
| 3-2 | fs.rs | `src/cook/fs.rs` | wget時のURL変換（282行目） |

**使用箇所**:
- **fetch.rs:182行目**: `command.arg(translate_mirror(&git));`
- **fs.rs:282行目**: `command.arg(translate_mirror(url));`

### プログラム呼び出し階層図

```
fetch (fetch.rs:117)
    │
    ├─ Git clone処理
    │      └─ Command::new("git").arg(translate_mirror(&git))
    │                                    │
    │                                    └─ translate_mirror (config.rs:150)
    │                                           │
    │                                           ├─ get_config().mirrors取得
    │                                           │
    │                                           └─ starts_with + replacen
    │
    └─ Tarダウンロード処理
           └─ download_wget(url, ...)
                  └─ wget(translate_mirror(url))
```

### データフロー図

```
[入力]                       [処理]                           [出力]

元URL ─────────────────▶ translate_mirror
"https://github.com/        │
 user/repo.git"             │
                            ▼
cookbook.toml ────────▶ get_config().mirrors
[mirrors]                   │
"https://github.com/" =     │
"https://mirror.local/"     │
                            ▼
                    starts_with チェック
                            │
                            ▼
                    replacen で置換
                            │
                            ▼
                    変換後URL ───▶ git/wget コマンド
                    "https://mirror.local/
                     user/repo.git"
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| config.rs | `src/config.rs` | ソース | translate_mirror関数とMirrorsConfig型 |
| fetch.rs | `src/cook/fetch.rs` | ソース | git clone時のURL変換呼び出し |
| fs.rs | `src/cook/fs.rs` | ソース | wget時のURL変換呼び出し |
| cookbook.toml | プロジェクトルート | 設定 | [mirrors]セクションの定義 |
