# 機能設計書 41-環境変数オーバーライド

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）における環境変数オーバーライド機能の設計を記載する。この機能は、COOKBOOK_*形式の環境変数によってビルド設定を動的に上書きする機能を提供する。

### 本機能の処理概要

環境変数オーバーライド機能は、cookbook.tomlファイルで定義された設定値を環境変数によって動的に変更する機能である。これにより、CI/CD環境やローカル開発環境など、異なる実行環境に応じた柔軟なビルド設定が可能となる。

**業務上の目的・背景**：ビルドシステムの設定は、開発環境・CI環境・本番ビルド環境など、実行環境によって異なる要件が発生することが多い。例えば、CI環境ではTUIを無効化し、ローカル開発ではログ出力を有効化するといった要件がある。この機能により、cookbook.tomlを変更することなく、環境変数だけでビルド動作をカスタマイズできる。

**機能の利用シーン**：
- CI/CDパイプラインでのビルド設定（TUI無効化、ノンストップモード有効化）
- 開発者のローカル環境でのデバッグ用設定（verbose有効化）
- オフラインビルド環境での設定（offline有効化）
- 並列ビルドのジョブ数カスタマイズ

**主要な処理内容**：
1. 環境変数からCOOKBOOK_*形式の設定値を取得する
2. 環境変数の値をRustの型（bool, usize等）にパースする
3. cookbook.tomlで未設定の項目に対して環境変数値を適用する
4. パース失敗時はデフォルト値を使用する

**関連システム・外部連携**：特になし（OSの環境変数機構を利用）

**権限による制御**：特になし（環境変数の設定はOS側の権限に依存）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | repo cookコマンド画面 | API連携 | COOKBOOK_*環境変数での設定上書き |

## 機能種別

設定管理 / 環境変数処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| COOKBOOK_MAKE_JOBS | String (環境変数) | No | 並列ビルドのジョブ数 | usizeにパース可能な整数文字列 |
| COOKBOOK_LOGS | String (環境変数) | No | ログ出力有効/無効 | true/false |
| COOKBOOK_OFFLINE | String (環境変数) | No | オフラインモード | true/false |
| COOKBOOK_VERBOSE | String (環境変数) | No | 詳細ログ出力 | true/false |
| COOKBOOK_NONSTOP | String (環境変数) | No | エラースキップ継続 | true/false |
| COOKBOOK_CLEAN_BUILD | String (環境変数) | No | ビルド前クリーン | true/false |
| COOKBOOK_CLEAN_TARGET | String (環境変数) | No | ビルド後クリーン | true/false |
| CI | String (環境変数) | No | CI環境判定用 | 空でない値でCI環境と判定 |

### 入力データソース

- OSの環境変数（std::env::var関数で取得）
- cookbook.tomlファイル（未設定項目の判定用）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CookConfig.offline | bool | オフラインモードの設定値 |
| CookConfig.jobs | usize | 並列ジョブ数 |
| CookConfig.tui | bool | TUI有効/無効 |
| CookConfig.logs | bool | ログ出力有効/無効 |
| CookConfig.nonstop | bool | ノンストップモード |
| CookConfig.verbose | bool | 詳細ログ出力 |
| CookConfig.clean_build | bool | ビルド前クリーン |
| CookConfig.clean_target | bool | ビルド後クリーン |

### 出力先

CookbookConfig構造体のcookフィールド（メモリ上の静的変数CONFIG）

## 処理フロー

### 処理シーケンス

```
1. init_config()関数の呼び出し
   └─ プログラム起動時に一度だけ実行される
2. cookbook.tomlファイルの読み込み（存在する場合）
   └─ 存在しない場合はデフォルト値を使用
3. TUI設定の環境変数チェック
   └─ cook_opt.tuiがNoneの場合、CI環境変数で判定
4. ジョブ数の環境変数チェック
   └─ cook_opt.jobsがNoneの場合、COOKBOOK_MAKE_JOBSを確認
   └─ 未設定時はCPU並列度を使用
5. ログ設定の環境変数チェック
   └─ COOKBOOK_LOGS環境変数を確認
6. オフライン設定の環境変数チェック
   └─ COOKBOOK_OFFLINE環境変数を確認
7. 詳細ログ設定の環境変数チェック
   └─ COOKBOOK_VERBOSE環境変数を確認
8. ノンストップ設定の環境変数チェック
   └─ COOKBOOK_NONSTOP環境変数を確認
9. クリーン設定の環境変数チェック
   └─ COOKBOOK_CLEAN_BUILD、COOKBOOK_CLEAN_TARGET環境変数を確認
10. CookConfigの生成と静的変数への設定
    └─ CONFIG.set()で設定を保存
```

### フローチャート

```mermaid
flowchart TD
    A[init_config開始] --> B{cookbook.toml存在?}
    B -->|Yes| C[tomlファイルをパース]
    B -->|No| D[デフォルト設定を使用]
    C --> E{cook_opt.tuiがNone?}
    D --> E
    E -->|Yes| F[CI環境変数でTUI判定]
    E -->|No| G{cook_opt.jobsがNone?}
    F --> G
    G -->|Yes| H[extract_env COOKBOOK_MAKE_JOBS]
    G -->|No| I{cook_opt.logsがNone?}
    H --> I
    I -->|Yes| J[extract_env COOKBOOK_LOGS]
    I -->|No| K{他の設定項目も同様に処理}
    J --> K
    K --> L[CookConfig生成]
    L --> M[CONFIG.set]
    M --> N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-41-01 | 設定優先度 | cookbook.toml設定 > デフォルト値、環境変数はtoml未設定時のみ適用 | 常時 |
| BR-41-02 | CI環境判定 | CI環境変数が設定されている場合はTUIを無効化 | cook_opt.tuiがNoneの場合 |
| BR-41-03 | パース失敗時 | 環境変数のパースに失敗した場合はデフォルト値を使用 | 型変換エラー発生時 |
| BR-41-04 | ジョブ数デフォルト | ジョブ数未指定時はCPU並列度を使用 | COOKBOOK_MAKE_JOBS未設定時 |

### 計算ロジック

```rust
// extract_env関数の計算ロジック
fn extract_env<T: FromStr>(key: &str, default: T) -> T {
    if let Ok(e) = env::var(&key) {
        str::parse(&e).unwrap_or(default)  // パース失敗時はデフォルト値
    } else {
        default  // 環境変数未設定時はデフォルト値
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-41-01 | パースエラー | 環境変数の値が型に変換できない | デフォルト値を使用して処理継続 |
| E-41-02 | 初期化重複 | CONFIG.set()が2回呼ばれた | panic（設計上発生しない） |

### リトライ仕様

リトライは不要（パース失敗時はデフォルト値で処理継続）

## トランザクション仕様

該当なし

## パフォーマンス要件

- 環境変数の読み込みは起動時に一度だけ実行
- OnceLock使用により、以降の設定参照はロックフリーで高速

## セキュリティ考慮事項

- 環境変数による設定上書きは、シェルアクセス権を持つユーザーのみが可能
- 機密情報は環境変数として扱わない設計

## 備考

- extract_env関数はジェネリックで実装されており、FromStrトレイトを実装した任意の型に対応
- bool型の場合、"true"/"false"文字列がパース対象

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、設定を保持するデータ構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | config.rs | `src/config.rs` | CookConfigOpt構造体（5-27行目）- Optionでラップされた設定値 |
| 1-2 | config.rs | `src/config.rs` | CookConfig構造体（29-39行目）- 実際に使用される設定値 |
| 1-3 | config.rs | `src/config.rs` | CookbookConfig構造体（56-64行目）- 全体の設定コンテナ |

**読解のコツ**: CookConfigOptの各フィールドがOption型であることに注目。これにより、cookbook.tomlで設定された値と未設定の値を区別できる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | config.rs | `src/config.rs` | init_config関数（68-121行目）- 設定初期化のメイン処理 |

**主要処理フロー**:
1. **69-78行目**: cookbook.tomlの読み込みとパース
2. **80-82行目**: TUI設定（CI環境変数による自動判定）
3. **83-90行目**: ジョブ数の環境変数適用
4. **91-108行目**: 各設定項目の環境変数適用
5. **118行目**: CookConfigOpt → CookConfig変換
6. **120行目**: CONFIG静的変数への設定

#### Step 3: 環境変数抽出ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | config.rs | `src/config.rs` | extract_env関数（123-129行目）- 環境変数からの値抽出 |

**主要処理フロー**:
- **124行目**: env::varで環境変数取得を試行
- **125行目**: str::parseで型変換、失敗時はデフォルト値
- **127行目**: 環境変数未設定時はデフォルト値を返却

#### Step 4: 設定取得インターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | config.rs | `src/config.rs` | get_config関数（131-133行目）- 設定値の取得 |
| 4-2 | config.rs | `src/config.rs` | CONFIG静的変数（66行目）- OnceLockによるスレッドセーフな設定保持 |

### プログラム呼び出し階層図

```
init_config()
    │
    ├─ fs::read_to_string("cookbook.toml")
    │      └─ toml::from_str() → CookbookConfig
    │
    ├─ env::var("CI")
    │      └─ TUI自動判定
    │
    ├─ extract_env("COOKBOOK_MAKE_JOBS", default)
    │      ├─ env::var()
    │      └─ str::parse()
    │
    ├─ extract_env("COOKBOOK_LOGS", default)
    ├─ extract_env("COOKBOOK_OFFLINE", default)
    ├─ extract_env("COOKBOOK_VERBOSE", default)
    ├─ extract_env("COOKBOOK_NONSTOP", default)
    ├─ extract_env("COOKBOOK_CLEAN_BUILD", default)
    ├─ extract_env("COOKBOOK_CLEAN_TARGET", default)
    │
    └─ CONFIG.set(config)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

cookbook.toml ─────────▶ toml::from_str() ─────▶ CookConfigOpt
                                │
                                ▼
COOKBOOK_MAKE_JOBS ────▶ extract_env() ────────▶ jobs: usize
COOKBOOK_LOGS ─────────▶ extract_env() ────────▶ logs: bool
COOKBOOK_OFFLINE ──────▶ extract_env() ────────▶ offline: bool     ─▶ CookConfig
COOKBOOK_VERBOSE ──────▶ extract_env() ────────▶ verbose: bool
COOKBOOK_NONSTOP ──────▶ extract_env() ────────▶ nonstop: bool
CI ────────────────────▶ env::var() ───────────▶ tui: bool
                                │
                                ▼
                        CONFIG (OnceLock<CookbookConfig>)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| config.rs | `src/config.rs` | ソース | 設定管理の主要実装 |
| lib.rs | `src/lib.rs` | ソース | configモジュールのエクスポート |
| repo.rs | `src/bin/repo.rs` | ソース | init_config()の呼び出し元 |
| cookbook.toml | `./cookbook.toml` | 設定 | 設定ファイル（オプション） |
