# 機能設計書 42-blake3ハッシュ計算

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるblake3ハッシュ計算機能の設計を記載する。この機能は、ファイルの整合性検証のためにblake3暗号学的ハッシュ値を計算する機能を提供する。

### 本機能の処理概要

blake3ハッシュ計算機能は、ダウンロードしたソースアーカイブ（tarファイル）やパッケージファイルの整合性を検証するために、blake3アルゴリズムによるハッシュ値を計算する。大きなファイルの処理時には進捗バーを表示するオプションも提供する。

**業務上の目的・背景**：パッケージビルドシステムでは、外部からダウンロードしたソースコードが改竄されていないことを確認する必要がある。また、ビルド成果物の一意性を保証するためにもハッシュ値による検証が重要である。blake3はSHA-256より高速でありながら同等以上のセキュリティを提供するため採用されている。

**機能の利用シーン**：
- tarアーカイブダウンロード後の整合性検証
- recipe.tomlのblake3フィールドとの照合
- パッケージ作成時のメタデータ生成
- ビルド成果物の識別

**主要な処理内容**：
1. ファイルをオープンしてメタデータ（サイズ）を取得する
2. blake3::Hasherを使用してストリーミングでハッシュ値を計算する
3. 進捗バー表示が有効な場合はProgressBarReadラッパーでバイト数をトラッキングする
4. 計算結果を16進文字列形式で返却する

**関連システム・外部連携**：
- blake3クレート（暗号学的ハッシュライブラリ）
- pbrクレート（進捗バー表示）

**権限による制御**：特になし（ファイル読み取り権限のみ必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連せず、内部ユーティリティとして使用 |

## 機能種別

ユーティリティ / ハッシュ計算

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | impl AsRef<Path> | Yes | ハッシュ計算対象ファイルのパス | ファイルが存在すること |

### 入力データソース

- ローカルファイルシステム上のファイル（source.tar、*.pkgar等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ハッシュ値 | String | 64文字の16進数文字列（256bit） |

### 出力先

- 関数の戻り値としてResult<String, String>で返却
- 進捗バー表示時は標準出力（stderr）にも出力

## 処理フロー

### 処理シーケンス

```
1. blake3_progress / blake3_silent関数呼び出し
   └─ 用途に応じて選択
2. ファイルメタデータ取得（進捗バー用）
   └─ fs::metadata()でファイルサイズを取得
3. ファイルオープン
   └─ fs::File::open()
4. Hasherの初期化
   └─ blake3::Hasher::new()
5. ストリーミング読み込みとハッシュ更新
   └─ update_reader()で効率的に処理
6. ハッシュ値の確定
   └─ finalize()
7. 16進文字列への変換
   └─ to_hex()
8. 進捗バーの終了処理（該当時）
   └─ finish_println()
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{進捗表示必要?}
    B -->|Yes| C[blake3_progress]
    B -->|No| D[blake3_silent]
    C --> E[ファイルサイズ取得]
    E --> F[ProgressBar初期化]
    F --> G[ファイルオープン]
    D --> G
    G --> H[Hasher初期化]
    H --> I{進捗バーあり?}
    I -->|Yes| J[ProgressBarReadでラップ]
    I -->|No| K[直接ファイルを使用]
    J --> L[update_reader]
    K --> L
    L --> M[finalize]
    M --> N[to_hex変換]
    N --> O{進捗バーあり?}
    O -->|Yes| P[finish_println]
    O -->|No| Q[終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-42-01 | ハッシュ長 | blake3は256bit（64文字の16進表記）のハッシュを生成 | 常時 |
| BR-42-02 | 進捗バー更新間隔 | 最大1秒に1回の更新頻度 | blake3_progress使用時 |
| BR-42-03 | バイト単位表示 | 進捗バーはバイト単位で表示 | blake3_progress使用時 |

### 計算ロジック

```rust
// ハッシュ計算の基本ロジック
let hash = Hasher::new()
    .update_reader(&mut reader)?  // ストリーミング読み込み
    .finalize();                  // ハッシュ確定
let hex_string = format!("{}", hash.to_hex());  // 16進文字列化
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-42-01 | IOエラー | ファイルが存在しない | エラーメッセージを返却 |
| E-42-02 | IOエラー | ファイル読み取り権限がない | エラーメッセージを返却 |
| E-42-03 | IOエラー | 読み込み中にエラー発生 | エラーメッセージを返却 |

### リトライ仕様

リトライは行わない。エラー発生時は呼び出し元でハンドリング。

## トランザクション仕様

該当なし

## パフォーマンス要件

- blake3はSIMD命令を活用し、SHA-256の約10倍の速度を実現
- ストリーミング処理によりメモリ使用量を最小化
- 進捗バー更新は最大1秒/回に制限し、I/O性能への影響を軽減

## セキュリティ考慮事項

- blake3は暗号学的ハッシュ関数であり、衝突耐性・原像攻撃耐性を持つ
- ダウンロードファイルの改竄検知に使用

## 備考

- blake3クレートはRust製の高速ハッシュライブラリ
- update_readerはReadトレイト実装型に対して効率的にハッシュを計算
- hash_to_hex関数は[u8; 32]配列から16進文字列への変換ユーティリティ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | progress_bar.rs | `src/progress_bar.rs` | ProgressBarRead構造体（5-8行目）- 進捗バーとリーダーのラッパー |

**読解のコツ**: ProgressBarReadはReadトレイトを実装しており、読み込みバイト数を進捗バーに通知する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | blake3.rs | `src/blake3.rs` | blake3_progress関数（6-23行目）- 進捗バー付きハッシュ計算 |
| 2-2 | blake3.rs | `src/blake3.rs` | blake3_silent関数（25-31行目）- サイレントハッシュ計算 |

**主要処理フロー（blake3_progress）**:
1. **7行目**: ファイルメタデータ取得でサイズを取得
2. **9行目**: ファイルオープン
3. **11-14行目**: ProgressBar初期化と設定
4. **16行目**: ProgressBarReadでラップ
5. **17行目**: Hasher::new().update_reader()でハッシュ計算
6. **18行目**: finalize().to_hex()で16進文字列化
7. **20行目**: 進捗バー終了処理

**主要処理フロー（blake3_silent）**:
1. **26行目**: ファイルオープン
2. **28行目**: Hasher::new().update_reader()でハッシュ計算
3. **29行目**: finalize().to_hex()で16進文字列化

#### Step 3: 進捗バー実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | progress_bar.rs | `src/progress_bar.rs` | ProgressBarRead::new（10-14行目）- コンストラクタ |
| 3-2 | progress_bar.rs | `src/progress_bar.rs` | Read実装（16-22行目）- 読み込みバイト数のトラッキング |

**主要処理フロー**:
- **18行目**: 内部リーダーからread()
- **19行目**: pb.add()で進捗バーを更新

#### Step 4: ユーティリティ関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | blake3.rs | `src/blake3.rs` | hash_to_hex関数（33-35行目）- バイト配列から16進文字列変換 |

### プログラム呼び出し階層図

```
fetch.rs / package.rs
    │
    ├─ blake3_progress(path)
    │      ├─ fs::metadata() → ファイルサイズ
    │      ├─ fs::File::open()
    │      ├─ ProgressBar::new()
    │      │      └─ message(), set_max_refresh_rate(), set_units()
    │      ├─ ProgressBarRead::new()
    │      ├─ Hasher::new().update_reader()
    │      ├─ finalize().to_hex()
    │      └─ finish_println()
    │
    └─ blake3_silent(path)
           ├─ fs::File::open()
           ├─ Hasher::new().update_reader()
           └─ finalize().to_hex()
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ファイルパス ─────────────▶ fs::File::open() ─────▶ File
                                  │
                                  ▼
File ─────────────────────▶ ProgressBarRead ─────▶ バイト数カウント
                                  │
                                  ▼ (read chunks)
                          blake3::Hasher
                                  │
                                  ▼ (update_reader)
                          内部状態更新
                                  │
                                  ▼ (finalize)
                          [u8; 32] ハッシュ値
                                  │
                                  ▼ (to_hex)
                          String (64文字の16進数)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| blake3.rs | `src/blake3.rs` | ソース | blake3ハッシュ計算の主要実装 |
| progress_bar.rs | `src/progress_bar.rs` | ソース | 進捗バー表示のラッパー実装 |
| fetch.rs | `src/cook/fetch.rs` | ソース | ハッシュ計算の呼び出し元（tar検証） |
| lib.rs | `src/lib.rs` | ソース | blake3モジュールのエクスポート |
| Cargo.toml | `Cargo.toml` | 設定 | blake3クレート依存定義 |
