# 機能設計書 45-Gitリモートトラッキング取得

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるGitリモートトラッキング取得機能の設計を記載する。この機能は、ローカルブランチが追跡するリモートブランチの情報を取得する機能を提供する。

### 本機能の処理概要

Gitリモートトラッキング取得機能は、現在チェックアウトしているローカルブランチが追跡しているリモートブランチの情報（ローカルブランチ名、リモートブランチ名、リモート名、リモートURL）を.git/configファイルから取得する。これにより、ソースの更新元を特定し、適切なフェッチ操作を行うことができる。

**業務上の目的・背景**：パッケージビルドシステムでは、ソースコードのフェッチ時に正しいリモートリポジトリからデータを取得する必要がある。ローカルブランチのトラッキング情報を取得することで、recipe.tomlで指定されたGit URLと現在のリポジトリ状態を比較し、リビルドの必要性を判定できる。

**機能の利用シーン**：
- can_skip_rebuild判定（リモートURLとブランチの一致確認）
- get_git_fetch_revの引数（remote_url, remote_branch）の取得
- リポジトリ状態の検証
- 開発者のローカル変更検出

**主要な処理内容**：
1. .git/HEADファイルから現在のブランチ名を取得する
2. .git/configファイルを読み込み、ブランチセクションを検索する
3. ブランチのremote設定とmerge設定を抽出する
4. リモートセクションからURLを取得する

**関連システム・外部連携**：
- Gitリポジトリ（.git/HEAD、.git/config）

**権限による制御**：特になし（ファイル読み取り権限のみ必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面に直接関連せず、内部ユーティリティとして使用 |

## 機能種別

ユーティリティ / Git操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dir | &PathBuf | Yes | Gitリポジトリのルートディレクトリパス | .git/HEAD、.git/configファイルが存在すること |

### 入力データソース

- ローカルファイルシステム上のGitリポジトリ
  - .git/HEAD
  - .git/config

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ローカルブランチ名 | String | 現在のブランチ名（例: "main"） |
| リモートブランチ名 | String | 追跡先のブランチ名（例: "main"） |
| リモート名 | String | リモートの名前（例: "origin"） |
| リモートURL | String | リモートリポジトリのURL |

### 出力先

- 関数の戻り値としてResult<(String, String, String, String), String>で返却

## 処理フロー

### 処理シーケンス

```
1. get_git_remote_tracking関数呼び出し
   └─ ディレクトリパスを受け取る
2. .git/HEADファイルの読み込み
   └─ fs::read_to_string()
3. detached HEAD判定
   └─ "ref: "で始まらない場合はdetached（空のトラッキング情報を返却）
4. ローカルブランチ名の抽出
   └─ get_git_branch_name()で最後の"/"以降を取得
5. .git/configファイルの読み込み
   └─ fs::read_to_string()
6. [branch "{branch}"]セクションの検索
   └─ remote = とmerge = を抽出
7. [remote "{remote}"]セクションの検索
   └─ url = を抽出
8. 結果の返却
   └─ (local_branch, remote_branch, remote_name, remote_url)
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[.git/HEAD読み込み]
    B --> C{detached HEAD?}
    C -->|Yes| D[SHA, 空, 空, 空 を返却]
    C -->|No| E[ブランチ名抽出]
    E --> F[.git/config読み込み]
    F --> G[branch セクション検索]
    G --> H{remote, merge 取得成功?}
    H -->|No| I[エラー: トラッキングなし]
    H -->|Yes| J[remote セクション検索]
    J --> K{url 取得成功?}
    K -->|No| L[エラー: URL取得失敗]
    K -->|Yes| M[4つ組を返却]
    D --> N[終了]
    I --> N
    L --> N
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-01 | detached処理 | detached HEADの場合はSHAと空文字列を返却 | "ref: "で始まらない場合 |
| BR-45-02 | ブランチ名抽出 | refs/heads/main → main のように最後の要素を取得 | 常時 |
| BR-45-03 | .git接尾辞除去 | URLから.gitサフィックスを除去 | URL取得時 |
| BR-45-04 | セクションパース | INI形式で[section]と key = value を解析 | .git/config読み込み時 |

### 計算ロジック

```rust
// トラッキング情報取得のロジック（概要）
// 1. HEADからブランチ名を取得
let local_branch_path = head_content["ref: ".len()..].trim_end();
let local_branch_name = get_git_branch_name(local_branch_path)?;

// 2. [branch "{branch}"]セクションからremote, mergeを取得
let branch_section = format!("[branch \"{}\"]", local_branch_name);
// config内を解析してremote_name, remote_branchを取得

// 3. [remote "{remote}"]セクションからurlを取得
let remote_section = format!("[remote \"{}\"]", remote_name_str);
// config内を解析してremote_urlを取得

// 4. .gitサフィックスを除去
let url = chop_dot_git(url);
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-45-01 | ファイル不在 | .git/HEADまたは.git/configファイルが存在しない | エラーメッセージを返却 |
| E-45-02 | 読み取りエラー | ファイル読み取りに失敗 | エラーメッセージを返却 |
| E-45-03 | トラッキングなし | ブランチがリモートを追跡していない | エラーメッセージを返却 |
| E-45-04 | URL取得失敗 | リモートのURLが見つからない | エラーメッセージを返却 |

### リトライ仕様

リトライは行わない。トラッキング未設定の場合はエラーを返却。

## トランザクション仕様

該当なし

## パフォーマンス要件

- .git/configは通常小さいファイルのため、線形スキャンで問題なし
- 2回のファイル読み込み（HEAD、config）で完了

## セキュリティ考慮事項

- ローカルファイルシステムのみにアクセス
- .git/configの整合性に依存

## 備考

- .git/configのフォーマット例：
  ```ini
  [branch "main"]
      remote = origin
      merge = refs/heads/main
  [remote "origin"]
      url = https://github.com/user/repo.git
      fetch = +refs/heads/*:refs/remotes/origin/*
  ```
- chop_dot_git関数でURLから.gitサフィックスを除去

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: .git/configのフォーマットを理解する

Git設定ファイルのINI形式の理解が前提となる。

| セクション | キー | 説明 | 例 |
|-----------|------|------|-----|
| [branch "name"] | remote | リモート名 | origin |
| [branch "name"] | merge | マージ先参照 | refs/heads/main |
| [remote "name"] | url | リモートURL | https://... |

**読解のコツ**: Gitのconfig形式はINIファイル形式に類似しており、セクションとキー・バリューで構成される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fs.rs | `src/cook/fs.rs` | get_git_remote_tracking関数（371-460行目）- メイン処理 |

**主要処理フロー**:
1. **372-373行目**: .git/HEAD、.git/configパスの構築
2. **375-376行目**: HEADファイルの読み込み
3. **378-381行目**: detached HEADの判定と早期リターン
4. **383-384行目**: ローカルブランチ名の抽出
5. **386-387行目**: .git/configファイルの読み込み
6. **389-415行目**: [branch "..."]セクションのパース
7. **421-445行目**: [remote "..."]セクションのパース
8. **454-459行目**: 結果の返却

#### Step 3: ヘルパー関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fs.rs | `src/cook/fs.rs` | get_git_branch_name関数（469-476行目）- ブランチ名抽出 |
| 3-2 | fs.rs | `src/cook/fs.rs` | chop_dot_git関数（462-467行目）- .gitサフィックス除去 |

**主要処理フロー（get_git_branch_name）**:
- **471-474行目**: パスを"/"で分割し最後の要素を取得

**主要処理フロー（chop_dot_git）**:
- **463-465行目**: ".git"で終わる場合はそれを除去

#### Step 4: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fetch.rs | `src/cook/fetch.rs` | fetch関数内のcan_skip_rebuild判定（228-247行目） |

**主要処理フロー**:
- **228-229行目**: get_git_remote_trackingの呼び出し
- **231-238行目**: recipe.tomlのブランチ・URLとの比較
- **240行目**: get_git_fetch_revへのremote_url, remote_branch引数の受け渡し

### プログラム呼び出し階層図

```
fetch.rs
    │
    └─ fetch()
           │
           └─ get_git_remote_tracking(dir)
                  │
                  ├─ fs::read_to_string(".git/HEAD")
                  │      └─ get_git_branch_name() → ブランチ名
                  │
                  ├─ fs::read_to_string(".git/config")
                  │      │
                  │      ├─ [branch "{branch}"]セクション解析
                  │      │      └─ remote, merge 取得
                  │      │
                  │      └─ [remote "{remote}"]セクション解析
                  │             └─ url 取得
                  │
                  └─ chop_dot_git(url)
                         └─ .gitサフィックス除去
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ディレクトリパス ────────▶ .git/HEAD読み込み
                                  │
                                  ▼
                          detached判定
                                  │
              ┌───────────────────┴───────────────────┐
              ▼                                       ▼
      [detached]                              [ブランチ参照]
              │                                       │
              ▼                                       ▼
    (SHA, "", "", "")                        ブランチ名抽出
                                                      │
                                                      ▼
                                              .git/config読み込み
                                                      │
                                    ┌─────────────────┴─────────────────┐
                                    ▼                                   ▼
                            [branch]セクション                    [remote]セクション
                                    │                                   │
                                    ▼                                   ▼
                            remote, merge                            url
                                    │                                   │
                                    └─────────────────┬─────────────────┘
                                                      ▼
                                    (local_branch, remote_branch,
                                     remote_name, remote_url)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fs.rs | `src/cook/fs.rs` | ソース | get_git_remote_tracking, get_git_branch_name, chop_dot_gitの実装 |
| fetch.rs | `src/cook/fetch.rs` | ソース | can_skip_rebuild判定での呼び出し元 |
| .git/HEAD | ソースリポジトリ | Git内部 | 現在のHEAD参照 |
| .git/config | ソースリポジトリ | Git内部 | リポジトリ設定（ブランチ・リモート情報） |
