# 機能設計書 46-PTY管理

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるPTY（擬似端末）管理機能の設計を記載する。この機能は、子プロセス（ビルドコマンド等）の出力をキャプチャするための擬似端末を作成・管理する機能を提供する。

### 本機能の処理概要

PTY管理機能は、Unixの擬似端末（pty）機構を利用して、ビルドプロセスや外部コマンドの出力をキャプチャする。これにより、TUIモードでのビルドログのリアルタイム表示や、ログファイルへの保存が可能となる。また、子プロセスに対して端末環境を提供し、色付き出力やインタラクティブな動作を可能にする。

**業務上の目的・背景**：パッケージビルドシステムでは、並列ビルド時に複数のビルドプロセスの出力を管理する必要がある。TUIモードでは各プロセスの出力を分離して表示し、非TUIモードでもログファイルへの保存を行う。PTYを使用することで、子プロセスが端末を前提とした動作（色付き出力、進捗表示等）を正しく行える。

**機能の利用シーン**：
- TUIモードでの並列ビルド時の出力キャプチャ
- ビルドログのファイル保存
- 子プロセスへの端末環境提供
- 色付き出力のサポート

**主要な処理内容**：
1. openpty()でマスター/スレーブのPTYペアを作成する
2. スレーブPTYを子プロセスのstdout/stderrに接続する
3. マスターPTYからの出力を読み取りスレッドでキャプチャする
4. ログパイプを通じてログファイルにも出力を記録する

**関連システム・外部連携**：
- Unix PTY API（libc::openpty、ioctl等）
- filedescriptorクレート

**権限による制御**：特になし（PTY作成はユーザー権限で可能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | TUIメイン画面 | 補助機能 | ビルドログのリアルタイム表示 |
| 4 | repo cookコマンド画面 | 補助機能 | ビルドログ出力 |

## 機能種別

ユーティリティ / プロセス管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| PtySize.rows | u16 | No | 端末の行数（デフォルト24） | 0より大きい値 |
| PtySize.cols | u16 | No | 端末の列数（デフォルト80） | 0より大きい値 |

### 入力データソース

- なし（PTYは新規作成）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| pty_reader | Box<dyn Read + Send> | マスターPTYからの読み取りストリーム |
| log_reader | PipeReader | ログパイプの読み取り側 |
| slave_pty | UnixSlavePty | スレーブPTY（子プロセスに接続） |
| log_writer | PipeWriter | ログパイプの書き込み側 |

### 出力先

- setup_pty()の戻り値としてタプルで返却
- 子プロセスのstdout/stderrに接続

## 処理フロー

### 処理シーケンス

```
1. setup_pty関数呼び出し
   └─ PTYセットアップの開始
2. UnixPtySystem::openpty()
   └─ マスター/スレーブPTYペアの作成
3. pty_reader取得
   └─ master.try_clone_reader()
4. ログパイプ作成
   └─ std::io::pipe()
5. 戻り値の構築
   └─ (pty_reader, log_reader, (slave, log_writer))
6. [子プロセス起動時] spawn_to_pipe()
   └─ スレーブPTYをstdout/stderrに接続
7. [読み取り時] pty_readerからの読み取り
   └─ 出力のキャプチャ
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[setup_pty呼び出し]
    B --> C[openpty でPTYペア作成]
    C --> D[マスターPTY readerクローン]
    D --> E[ログパイプ作成]
    E --> F[戻り値返却]
    F --> G[spawn_to_pipe呼び出し]
    G --> H[子プロセス起動]
    H --> I[スレーブPTYに接続]
    I --> J[マスターPTYから読み取り]
    J --> K{EOF?}
    K -->|No| L[ログ/TUIに出力]
    L --> J
    K -->|Yes| M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | デフォルトサイズ | 端末サイズは24行x80列をデフォルト | PtySizeが指定されない場合 |
| BR-46-02 | CLOEXEC設定 | PTYファイルディスクリプタはexec時にクローズ | 常時 |
| BR-46-03 | EIOハンドリング | スレーブクローズ時のEIOはEOFとして扱う | 読み取り時 |
| BR-46-04 | シグナル初期化 | 子プロセスではシグナルをデフォルトにリセット | spawn時 |

### 計算ロジック

```rust
// PTYセットアップの基本ロジック
pub fn setup_pty() -> (Box<dyn Read + Send>, PipeReader, (UnixSlavePty, PipeWriter)) {
    let pty_system = UnixPtySystem::default();
    let pair = pty_system.openpty(PtySize {
        rows: 24,
        cols: 80,
        ..Default::default()
    }).expect("Unable to open pty");

    let pty_reader = pair.master.try_clone_reader().expect("Unable to clone pty reader");
    let (log_reader, log_writer) = std::io::pipe().expect("Failed to create log pipe");

    (pty_reader, log_reader, (pair.slave, log_writer))
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-46-01 | PTY作成失敗 | openptyシステムコール失敗 | panic（致命的エラー） |
| E-46-02 | リーダークローン失敗 | try_clone_reader失敗 | panic（致命的エラー） |
| E-46-03 | パイプ作成失敗 | std::io::pipe失敗 | panic（致命的エラー） |
| E-46-04 | EIO | スレーブPTYクローズ | EOFとして処理（正常終了） |

### リトライ仕様

リトライは行わない。PTY作成失敗は致命的エラーとして扱う。

## トランザクション仕様

該当なし

## パフォーマンス要件

- PTY作成は低レイテンシ（システムコール1回）
- 読み取りはノンブロッキング可能
- flush_ptyでは100msのスリープを挿入（バッファフラッシュ待ち）

## セキュリティ考慮事項

- PTYファイルディスクリプタはCLOEXECを設定し、子プロセスへの意図しない継承を防止
- setsid()で新セッションを作成し、プロセスグループを分離
- シグナルハンドラをデフォルトにリセットして安全な状態で子プロセスを起動

## 備考

- portable-ptyクレートをベースにしたカスタム実装
- libgit2等のライブラリは使用せず、libc直接呼び出し
- マクロlog_to_pty!でPTY有無に応じたログ出力を切り替え

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | pty.rs | `src/cook/pty.rs` | PtySize構造体（86-98行目）- 端末サイズ定義 |
| 1-2 | pty.rs | `src/cook/pty.rs` | PtyPair構造体（155-160行目）- マスター/スレーブペア |
| 1-3 | pty.rs | `src/cook/pty.rs` | UnixMasterPty/UnixSlavePty構造体（301-311行目）- PTYエンドポイント |

**読解のコツ**: PTYはマスター（親プロセス側）とスレーブ（子プロセス側）のペアで構成される。

#### Step 2: セットアップ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | pty.rs | `src/cook/pty.rs` | setup_pty関数（34-57行目）- PTYセットアップのエントリーポイント |
| 2-2 | pty.rs | `src/cook/pty.rs` | openpty関数（111-153行目）- libc::openptyラッパー |

**主要処理フロー（setup_pty）**:
1. **39-46行目**: UnixPtySystem::openptyでPTYペア作成
2. **49-52行目**: マスターPTYからリーダーをクローン
3. **54-55行目**: ログパイプの作成
4. **56行目**: 戻り値の構築

**主要処理フロー（openpty）**:
1. **115-120行目**: winsize構造体の構築
2. **122-132行目**: libc::openptyシステムコール
3. **138-143行目**: UnixMasterPty/UnixSlavePtyの構築
4. **149-150行目**: CLOEXEC設定

#### Step 3: 子プロセス起動を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | pty.rs | `src/cook/pty.rs` | spawn_to_pipe関数（70-75行目）- コマンド起動ラッパー |
| 3-2 | pty.rs | `src/cook/pty.rs` | PtyFd::spawn_command（249-294行目）- 実際の起動処理 |

**主要処理フロー（spawn_command）**:
1. **252-254行目**: stdout/stderrをPTYに接続
2. **255-278行目**: pre_execフック（シグナルリセット、setsid）
3. **282行目**: cmd.spawn()で子プロセス起動
4. **290-291行目**: stdout/stderrのtake（参照解放）

#### Step 4: 読み取り・フラッシュ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | pty.rs | `src/cook/pty.rs` | PtyFd::read（185-198行目）- EIOハンドリング付き読み取り |
| 4-2 | pty.rs | `src/cook/pty.rs` | flush_pty関数（59-68行目）- バッファフラッシュ |

**主要処理フロー（read）**:
- **187-193行目**: EIOをEOFとして扱う特殊処理

#### Step 5: ログマクロを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | pty.rs | `src/cook/pty.rs` | log_to_pty!マクロ（18-30行目）- 条件分岐ログ出力 |

**主要処理フロー**:
- PTYが存在する場合はPTYに出力
- 存在しない場合はeprintln!で標準エラー出力

### プログラム呼び出し階層図

```
TUI / repo cook
    │
    └─ setup_pty()
           │
           ├─ UnixPtySystem::openpty(PtySize)
           │      │
           │      ├─ libc::openpty()
           │      │      └─ マスター/スレーブFD作成
           │      │
           │      └─ cloexec() x 2
           │
           ├─ master.try_clone_reader()
           │
           └─ std::io::pipe()
                  └─ (log_reader, log_writer)

spawn_to_pipe(&mut Command, &PtyOut)
    │
    └─ slave.spawn_command(command)
           │
           ├─ command.stdout(pty.as_stdio())
           ├─ command.stderr(pty.as_stdio())
           │
           ├─ pre_exec()
           │      ├─ シグナルリセット
           │      └─ setsid()
           │
           └─ command.spawn()
```

### データフロー図

```
[子プロセス]              [PTY]                    [親プロセス]

stdout ─────────────────▶ スレーブPTY
stderr ─────────────────▶      │
                               │
                               ▼
                         マスターPTY ─────────────▶ pty_reader
                                                        │
                                                        ▼
                                                   TUI/ログ表示

log_to_pty!マクロ ────────────────────────────────▶ log_writer
                                                        │
                                                        ▼
                                                   log_reader
                                                        │
                                                        ▼
                                                   ログファイル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| pty.rs | `src/cook/pty.rs` | ソース | PTY管理の主要実装 |
| fs.rs | `src/cook/fs.rs` | ソース | run_command等でのPTY使用 |
| cook.rs | `src/cook.rs` | ソース | TUIモードでのPTYセットアップ呼び出し元 |
