# 機能設計書 47-プログレスバー

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるプログレスバー機能の設計を記載する。この機能は、大きなファイルの処理時に進捗状況を視覚的に表示する機能を提供する。

### 本機能の処理概要

プログレスバー機能は、ファイルのダウンロードやハッシュ計算など、時間のかかる処理の進捗状況をターミナルに表示する。pbrクレートをベースにしたProgressBarと、Readトレイトを実装したラッパー構造体ProgressBarReadを提供し、ストリーム処理と進捗表示を統合する。

**業務上の目的・背景**：大きなソースアーカイブのダウンロードやblake3ハッシュ計算では、処理完了までに時間がかかることがある。ユーザーに処理の進捗状況を表示することで、処理が正常に進行していることを示し、ユーザーエクスペリエンスを向上させる。

**機能の利用シーン**：
- source.tar（ソースアーカイブ）のblake3ハッシュ計算時
- 大きなファイルのダウンロード時（将来的な拡張）
- パッケージアーカイブの処理時

**主要な処理内容**：
1. ProgressBarを初期化し、総バイト数を設定する
2. ProgressBarReadでReadストリームをラップする
3. read()呼び出し毎に読み取りバイト数を進捗バーに通知する
4. 処理完了時にfinish_println()で進捗バーを終了する

**関連システム・外部連携**：
- pbrクレート（進捗バー表示ライブラリ）

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ターミナル出力（CLIモードで使用） |

## 機能種別

ユーティリティ / UI表示

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| total_bytes | u64 | Yes | 処理対象の総バイト数 | 0以上 |
| reader | R: Read | Yes | 読み取り元のストリーム | Readトレイト実装 |

### 入力データソース

- ファイルストリーム（fs::File等）
- ネットワークストリーム（Response等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 進捗表示 | 標準エラー出力 | ターミナルへの進捗バー表示 |
| 読み取りデータ | &[u8] | 元のストリームからの読み取りデータ |

### 出力先

- 標準エラー出力（進捗バー表示）
- 呼び出し元へのデータ転送

## 処理フロー

### 処理シーケンス

```
1. ProgressBar::new(total_bytes)
   └─ 進捗バーの初期化
2. message(), set_max_refresh_rate(), set_units()
   └─ 進捗バーの設定
3. ProgressBarRead::new(&mut pb, &mut reader)
   └─ リーダーのラッピング
4. [ループ] pbr.read(buf)
   └─ データ読み取りと進捗更新
5. pb.finish_println("")
   └─ 進捗バーの終了
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[ProgressBar::new]
    B --> C[進捗バー設定]
    C --> D[ProgressBarRead::new]
    D --> E{read呼び出し}
    E --> F[内部リーダーから読み取り]
    F --> G[pb.add count]
    G --> H{EOF?}
    H -->|No| E
    H -->|Yes| I[finish_println]
    I --> J[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-47-01 | 更新頻度制限 | 最大1秒に1回の更新頻度 | blake3計算時 |
| BR-47-02 | バイト単位表示 | 進捗はバイト単位で表示 | 常時 |
| BR-47-03 | 透過的読み取り | read()は元のストリームのデータをそのまま返す | 常時 |
| BR-47-04 | 累積カウント | add()は累積的にカウントを増加 | read()呼び出し毎 |

### 計算ロジック

```rust
// ProgressBarReadの読み取りロジック
impl<'p, 'r, P: Write, R: Read> Read for ProgressBarRead<'p, 'r, P, R> {
    fn read(&mut self, buf: &mut [u8]) -> Result<usize> {
        let count = self.r.read(buf)?;  // 元のリーダーから読み取り
        self.pb.add(count as u64);       // 進捗バーにバイト数を加算
        Ok(count)                         // 読み取りバイト数を返却
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-47-01 | IOエラー | 元のリーダーの読み取り失敗 | エラーをそのまま上位に伝播 |

### リトライ仕様

リトライは行わない。元のリーダーのエラーをそのまま返却。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 進捗バー更新は最大1秒/回に制限
- read()の追加オーバーヘッドは最小限（add()呼び出しのみ）
- 大きなファイル（数GB）でも安定して動作

## セキュリティ考慮事項

- 特になし（表示のみ）

## 備考

- pbrクレートのProgressBarを再エクスポート
- ProgressBarReadはジェネリックでReadトレイト実装型をラップ
- blake3_progress()で主に使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | progress_bar.rs | `src/progress_bar.rs` | ProgressBarRead構造体（5-8行目）- リーダーラッパー |

**読解のコツ**: ProgressBarReadは2つの参照（進捗バーとリーダー）を保持するラッパー構造体。ライフタイムパラメータ`'p`、`'r`で借用を管理。

```rust
pub struct ProgressBarRead<'p, 'r, P: Write + 'p, R: Read + 'r> {
    pb: &'p mut ProgressBar<P>,  // 進捗バーへの参照
    r: &'r mut R,                 // 元のリーダーへの参照
}
```

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | progress_bar.rs | `src/progress_bar.rs` | ProgressBarRead::new（10-14行目）- コンストラクタ |

**主要処理フロー**:
- **11行目**: 進捗バーとリーダーを受け取る
- **12行目**: ProgressBarRead構造体を構築して返却

#### Step 3: Readトレイト実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | progress_bar.rs | `src/progress_bar.rs` | Read for ProgressBarRead（16-22行目）- トレイト実装 |

**主要処理フロー**:
- **17行目**: read()メソッドの実装
- **18行目**: 内部リーダーからの読み取り
- **19行目**: 読み取りバイト数を進捗バーに加算
- **20行目**: 読み取りバイト数を返却

#### Step 4: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | blake3.rs | `src/blake3.rs` | blake3_progress関数（6-23行目）- 主要な呼び出し元 |

**主要処理フロー**:
- **7行目**: ファイルサイズ取得
- **11-14行目**: ProgressBar初期化と設定
- **16行目**: ProgressBarReadでファイルをラップ
- **17行目**: update_readerでハッシュ計算
- **20行目**: finish_println()で進捗バー終了

### プログラム呼び出し階層図

```
blake3_progress(path)
    │
    ├─ fs::metadata(path) → len
    │
    ├─ ProgressBar::new(len)
    │      ├─ message("blake3: ")
    │      ├─ set_max_refresh_rate(1秒)
    │      └─ set_units(Bytes)
    │
    ├─ fs::File::open(path)
    │
    └─ ProgressBarRead::new(&mut pb, &mut file)
           │
           └─ [Hasher::update_reader(&mut pbr)]
                  │
                  └─ [ループ] pbr.read(buf)
                         │
                         ├─ file.read(buf) → count
                         │
                         └─ pb.add(count)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ファイルサイズ ───────────▶ ProgressBar::new(len)
                                  │
                                  ▼
                           進捗バー初期化
                                  │
ファイル ────────────────▶ ProgressBarRead::new()
                                  │
                                  ▼
                           [read()ループ]
                                  │
              ┌───────────────────┴───────────────────┐
              ▼                                       ▼
    内部リーダーread()                          pb.add(count)
              │                                       │
              ▼                                       ▼
    データバッファ ─────────────▶ 呼び出し元    stderr進捗表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| progress_bar.rs | `src/progress_bar.rs` | ソース | ProgressBarReadの実装 |
| blake3.rs | `src/blake3.rs` | ソース | 主要な呼び出し元（blake3_progress） |
| lib.rs | `src/lib.rs` | ソース | progress_barモジュールの宣言 |
| Cargo.toml | `Cargo.toml` | 設定 | pbrクレート依存定義 |
