# 機能設計書 48-依存関係ツリー表示

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）における依存関係ツリー表示機能の設計を記載する。この機能は、パッケージの依存関係をツリー形式で視覚的に表示する機能を提供する。

### 本機能の処理概要

依存関係ツリー表示機能は、指定されたパッケージとその依存パッケージの関係を再帰的に走査し、ツリー構造として標準出力に表示する。各パッケージのビルド状態やファイルサイズも表示し、依存関係全体のディスク使用量の見積もりを提供する。

**業務上の目的・背景**：パッケージビルドシステムでは、パッケージ間の依存関係が複雑になることがある。開発者やビルド担当者が依存関係を視覚的に把握することで、ビルド順序の理解、循環依存の検出、ディスク使用量の見積もりが容易になる。

**機能の利用シーン**：
- `repo tree`コマンドでの依存関係確認
- イメージサイズの事前見積もり
- 依存パッケージのビルド状態確認
- デバッグ時の依存関係追跡

**主要な処理内容**：
1. ルートパッケージから依存関係を再帰的に走査する
2. 各パッケージのビルド状態（Built/NotBuilt/Missing）を判定する
3. ツリー形式で整形して標準出力に表示する
4. 重複パッケージは再展開せず、合計サイズを累積する

**関連システム・外部連携**：
- レシピ解析機能（CookRecipe）
- パッケージメタデータ（stage.toml）

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | repo treeコマンド画面 | 主機能 | パッケージ依存関係ツリーの表示 |

## 機能種別

ユーティリティ / 表示機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| package_name | &PackageName | Yes | ルートパッケージ名 | 存在するパッケージ |
| recipe_map | &HashMap<&PackageName, &CookRecipe> | Yes | パッケージとレシピのマップ | - |
| prefix | &str | No | ツリー描画用プレフィックス | - |
| is_last | bool | No | 兄弟ノードの最後かどうか | - |

### 入力データソース

- レシピマップ（CookRecipe）
- stage.toml（パッケージメタデータ）
- pkgarファイル（ビルド成果物）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ツリー表示 | 標準出力 | ツリー形式の依存関係表示 |
| 合計サイズ | u64 | 依存関係全体のファイルサイズ |

### 出力先

- 標準出力（println!）
- total_sizeの累積（参照渡し）

## 処理フロー

### 処理シーケンス

```
1. display_tree_entry / walk_tree_entry関数呼び出し
   └─ ルートパッケージから開始
2. パッケージのビルド状態判定
   └─ pkgarファイルの存在確認
3. 重複チェック
   └─ visitedセットで管理
4. ツリー行の表示
   └─ プレフィックス + パッケージ名 + サイズ
5. 依存パッケージの取得
   └─ stage.tomlまたはrecipe.tomlから
6. 子パッケージの再帰的処理
   └─ walk_tree_entryを再帰呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{レシピマップに存在?}
    B -->|No| C[Missing表示]
    B -->|Yes| D[pkgarファイル確認]
    D --> E{ビルド済み?}
    E -->|Yes| F{重複チェック}
    E -->|No| G[NotBuilt表示]
    F -->|重複| H[Deduped処理]
    F -->|未訪問| I[Built表示]
    I --> J[visitedに追加]
    J --> K[サイズ累積]
    K --> L[依存パッケージ取得]
    L --> M{依存あり?}
    M -->|Yes| N[子パッケージを再帰処理]
    M -->|No| O[終了]
    N --> O
    C --> O
    G --> O
    H --> O
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-48-01 | 重複スキップ | 既に訪問したパッケージは再展開しない | visited.containsがtrue |
| BR-48-02 | サイズ累積 | Builtパッケージのサイズのみ累積 | WalkTreeEntry::Built時 |
| BR-48-03 | 自動依存優先 | stage.tomlの依存情報をrecipe.tomlより優先 | stage.toml存在時 |
| BR-48-04 | ツリー記号 | 最後の子は"└── "、それ以外は"├── " | 常時 |

### 計算ロジック

```rust
// ツリー表示のロジック（概要）
pub fn walk_tree_entry(...) -> anyhow::Result<()> {
    // ビルド状態判定
    let entry = match (std::fs::metadata(&pkg_path), deduped) {
        (_, true) => WalkTreeEntry::Deduped,
        (Ok(meta), _) => WalkTreeEntry::Built(&pkg_path, meta.len()),
        (Err(_), _) => WalkTreeEntry::NotBuilt,
    };

    // 表示
    op(package_name, prefix, is_last, &entry)?;

    // サイズ累積
    if let WalkTreeEntry::Built(_p, pkg_size) = &entry {
        *total_size += pkg_size;
    }

    // 子パッケージの再帰処理
    for (i, dep_name) in sorted_deps.iter().enumerate() {
        walk_tree_entry(dep_name, recipe_map, &new_prefix, i == deps_count - 1, visited, total_size, op)?;
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-48-01 | レシピ不在 | パッケージがrecipe_mapに存在しない | Missingとして表示 |
| E-48-02 | パースエラー | stage.tomlのパースに失敗 | anyhow::Errorを返却 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 重複パッケージのスキップにより、大きな依存ツリーでも高速に処理
- HashSetによるO(1)の重複チェック

## セキュリティ考慮事項

- ローカルファイルシステムのみにアクセス

## 備考

- WalkTreeEntryは4状態: Built, NotBuilt, Deduped, Missing
- display_pkg_fnが実際の表示処理を担当
- format_size関数でサイズを人間可読形式に変換

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tree.rs | `src/cook/tree.rs` | WalkTreeEntry列挙型（12-17行目）- パッケージ状態 |

**読解のコツ**: WalkTreeEntryは4つの状態を表す。Builtはパスとサイズを、Deduped/NotBuilt/Missingは状態のみを保持。

```rust
pub enum WalkTreeEntry<'a> {
    Built(&'a PathBuf, u64),  // ビルド済み（パス、サイズ）
    NotBuilt,                  // 未ビルド
    Deduped,                   // 重複（展開スキップ）
    Missing,                   // レシピ不在
}
```

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tree.rs | `src/cook/tree.rs` | display_tree_entry関数（19-36行目）- 表示用エントリーポイント |
| 2-2 | tree.rs | `src/cook/tree.rs` | walk_tree_entry関数（38-107行目）- 汎用ツリー走査 |

**主要処理フロー（walk_tree_entry）**:
1. **47-54行目**: レシピ存在確認、Missingの場合は早期リターン
2. **56行目**: stage_paths()でパッケージパスを取得
3. **58-63行目**: ビルド状態の判定
4. **65行目**: op()で表示処理を実行
5. **67-69行目**: 重複の場合は再帰せず終了
6. **71-74行目**: visitedへの追加とサイズ累積
7. **77-86行目**: 依存パッケージの取得
8. **91-104行目**: 子パッケージの再帰処理

#### Step 3: 表示処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tree.rs | `src/cook/tree.rs` | display_pkg_fn関数（109-124行目）- パッケージ行の表示 |

**主要処理フロー**:
- **115-120行目**: 状態に応じたサイズ文字列の生成
- **121行目**: ツリー記号の決定（└── または ├──）
- **122行目**: println!で1行表示

#### Step 4: ファイルツリー表示を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | tree.rs | `src/cook/tree.rs` | walk_file_tree関数（126-164行目）- ディレクトリツリー表示 |

**主要処理フロー**:
- **131行目**: read_dir()でエントリ一覧取得
- **144-147行目**: ディレクトリの再帰処理
- **148-159行目**: ファイルのサイズ付き表示

### プログラム呼び出し階層図

```
repo tree コマンド
    │
    └─ display_tree_entry(package_name, recipe_map, ...)
           │
           └─ walk_tree_entry(package_name, recipe_map, ..., display_pkg_fn)
                  │
                  ├─ recipe_map.get(package_name)
                  │
                  ├─ cook_recipe.stage_paths() → (_, pkg_path, pkg_toml)
                  │
                  ├─ fs::metadata(pkg_path) → ビルド状態判定
                  │
                  ├─ display_pkg_fn(package_name, prefix, is_last, entry)
                  │      └─ println!(ツリー行)
                  │
                  ├─ visited.insert(package_name)
                  │
                  ├─ toml::from_str(pkg_toml) → 依存関係取得
                  │
                  └─ [再帰] walk_tree_entry(dep_name, ...)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

パッケージ名 ────────────▶ recipe_map.get()
                                  │
                                  ▼
                          stage_paths()
                                  │
                                  ▼
                          fs::metadata(pkg_path)
                                  │
              ┌───────────────────┴───────────────────┐
              ▼                                       ▼
      [Built]                                   [NotBuilt/Missing]
              │                                       │
              ▼                                       ▼
    display_pkg_fn() ─────────────────────────────▶ stdout
              │
              ▼
    total_size += pkg_size
              │
              ▼
    依存パッケージ取得
              │
              ▼
    [再帰処理]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tree.rs | `src/cook/tree.rs` | ソース | ツリー表示の主要実装 |
| repo.rs | `src/bin/repo.rs` | ソース | treeコマンドの実装（呼び出し元） |
| recipe.rs | `src/recipe.rs` | ソース | CookRecipe構造体の定義 |
| stage.toml | 各パッケージのtarget/ | 設定 | パッケージメタデータ |
