# 機能設計書 49-ファイルサイズフォーマット

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるファイルサイズフォーマット機能の設計を記載する。この機能は、バイト数を人間が読みやすい形式（KiB/MiB/GiB等）に変換する機能を提供する。

### 本機能の処理概要

ファイルサイズフォーマット機能は、バイト単位のファイルサイズを1024進数（バイナリプレフィックス）で適切な単位に変換し、人間が読みやすい文字列形式で返却する。依存関係ツリー表示やパッケージサイズ表示で使用される。

**業務上の目的・背景**：パッケージビルドシステムでは、ファイルサイズやパッケージサイズをユーザーに表示する場面が多い。バイト数をそのまま表示すると桁数が多くなり読みにくいため、適切な単位（KiB, MiB, GiB等）に変換して表示することで、ユーザーエクスペリエンスを向上させる。

**機能の利用シーン**：
- `repo tree`コマンドでのパッケージサイズ表示
- ファイルツリー表示でのファイルサイズ表示
- ダウンロードサイズの表示
- ビルド成果物のサイズ表示

**主要な処理内容**：
1. 入力バイト数が0の場合は"0 B"を返却する
2. 1024を底とした対数を計算し、適切な単位インデックスを決定する
3. バイト数を適切な単位で割り、小数点以下2桁でフォーマットする
4. 単位文字列を付与して返却する

**関連システム・外部連携**：なし

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | repo treeコマンド画面 | 補助機能 | バイト数の人間可読形式変換 |

## 機能種別

ユーティリティ / フォーマット

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| bytes | u64 | Yes | フォーマット対象のバイト数 | 0以上 |

### 入力データソース

- ファイルメタデータ（fs::metadata().len()）
- 計算結果のサイズ値

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| フォーマット済みサイズ | String | 単位付きのサイズ文字列（例: "1.50 MiB"） |

### 出力先

- 関数の戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. format_size関数呼び出し
   └─ バイト数を受け取る
2. ゼロチェック
   └─ 0の場合は"0 B"を返却
3. 単位インデックスの計算
   └─ log(1024)で適切な単位を決定
4. サイズ値の計算
   └─ bytes / 1024^i
5. 文字列フォーマット
   └─ "{:.2} {単位}"形式で返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{bytes == 0?}
    B -->|Yes| C["0 B"を返却]
    B -->|No| D[log1024 bytes を計算]
    D --> E[単位インデックス i を決定]
    E --> F[size = bytes / 1024^i]
    F --> G["{:.2} {UNITS[i]}"でフォーマット]
    G --> H[結果を返却]
    C --> I[終了]
    H --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-49-01 | バイナリプレフィックス | 1024進数（KiB, MiB等）を使用 | 常時 |
| BR-49-02 | 小数点精度 | 小数点以下2桁で表示 | 常時 |
| BR-49-03 | ゼロ処理 | 0バイトは"0 B"として表示 | bytes == 0 |
| BR-49-04 | 単位上限 | TiB（テビバイト）まで対応 | 常時 |

### 計算ロジック

```rust
pub fn format_size(bytes: u64) -> String {
    if bytes == 0 {
        return "0 B".to_string();
    }

    const UNITS: [&str; 5] = ["B", "KiB", "MiB", "GiB", "TiB"];

    // 単位インデックスの計算: log(bytes) / log(1024)
    let i = (bytes as f64).log(1024.0).floor() as usize;

    // サイズ値の計算: bytes / 1024^i
    let size = bytes as f64 / 1024.0_f64.powi(i as i32);

    format!("{:.2} {}", size, UNITS[i])
}
```

**計算例**：
- 1024 bytes → i=1, size=1.00 → "1.00 KiB"
- 1572864 bytes → i=2, size=1.50 → "1.50 MiB"
- 1073741824 bytes → i=3, size=1.00 → "1.00 GiB"

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

本機能はエラーを発生させない。全ての入力値に対して有効な出力を返却する。

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 単純な数学計算のみで、非常に高速
- メモリ割り当ては戻り値のString分のみ

## セキュリティ考慮事項

- 特になし（純粋な計算関数）

## 備考

- バイナリプレフィックス（KiB, MiB等）を使用し、SI接頭辞（KB, MB等）は使用しない
- TiBを超えるサイズは配列インデックス範囲外となるが、通常は発生しない想定

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tree.rs | `src/cook/tree.rs` | format_size関数（166-174行目）- サイズフォーマットの実装 |

**主要処理フロー**:
1. **167-169行目**: ゼロチェックと早期リターン
2. **170行目**: 単位配列の定義（B, KiB, MiB, GiB, TiB）
3. **171行目**: 単位インデックスの計算（対数計算）
4. **172行目**: サイズ値の計算（割り算）
5. **173行目**: フォーマット文字列の生成

**読解のコツ**:
- `(bytes as f64).log(1024.0).floor()` で1024を底とした対数を計算し、適切な単位を決定
- `1024.0_f64.powi(i as i32)` で1024のi乗を計算

#### Step 2: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tree.rs | `src/cook/tree.rs` | display_pkg_fn（116行目）- パッケージサイズ表示 |
| 2-2 | tree.rs | `src/cook/tree.rs` | walk_file_tree（157行目）- ファイルサイズ表示 |

**主要処理フロー（display_pkg_fn）**:
- **116行目**: Built状態のパッケージサイズをformat_sizeでフォーマット

**主要処理フロー（walk_file_tree）**:
- **157行目**: ファイルサイズをformat_sizeでフォーマット

### プログラム呼び出し階層図

```
display_pkg_fn(package_name, ..., WalkTreeEntry::Built(_, size))
    │
    └─ format_size(size)
           │
           ├─ [bytes == 0] → "0 B"
           │
           └─ [bytes > 0]
                  ├─ i = log(bytes) / log(1024)
                  ├─ size = bytes / 1024^i
                  └─ "{:.2} {UNITS[i]}"

walk_file_tree(dir, prefix, buffer)
    │
    └─ [ファイルの場合]
           │
           └─ format_size(metadata.len())
```

### データフロー図

```
[入力]                      [処理]                           [出力]

bytes: u64 ─────────────▶ ゼロチェック
                                │
              ┌─────────────────┴─────────────────┐
              ▼                                   ▼
        [bytes == 0]                        [bytes > 0]
              │                                   │
              ▼                                   ▼
        "0 B" ───────▶               log(bytes) / log(1024)
                                              │
                                              ▼
                                        単位インデックス i
                                              │
                                              ▼
                                      bytes / 1024^i
                                              │
                                              ▼
                                      format!("{:.2} {}", size, UNITS[i])
                                              │
                                              ▼
                                      String（例: "1.50 MiB"）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tree.rs | `src/cook/tree.rs` | ソース | format_size関数の実装 |
