# 機能設計書 5-ソース取得（同一ソース参照）

## 概要

本ドキュメントは、Redox OSビルドシステムにおける別パッケージのソースディレクトリをシンボリックリンクで参照する機能の設計を記載する。same_as機能による効率的なソース共有について詳述する。

### 本機能の処理概要

ソース取得（同一ソース参照）機能は、別のパッケージのソースディレクトリをシンボリックリンクで参照することで、ディスク容量を節約し、ソースの一貫性を保証する機能である。単一のソースリポジトリから複数のパッケージをビルドする場合に使用される。

**業務上の目的・背景**：多くのプロジェクトでは、単一のソースリポジトリ内に複数のサブプロジェクトが含まれており、それぞれが独立したパッケージとしてビルドされる。例えば、GCC（GNU Compiler Collection）では、gcc、g++、gfortranなどが同じソースツリーから個別のパッケージとしてビルドされる。このような場合に、ソースを重複してダウンロード・展開することを避け、効率的なビルド環境を実現する。

**機能の利用シーン**：
- 単一ソースリポジトリから複数パッケージをビルドする際（例：GCC, LLVM）
- 共通ライブラリを複数パッケージで共有する際
- ディスク容量を節約したい場合
- ソースの一貫性を保証したい場合

**主要な処理内容**：
1. 参照先パッケージのレシピ解析と存在確認
2. 参照先パッケージのソース取得（再帰的fetch）
3. シンボリックリンクの作成
4. ソース情報の引き継ぎ

**関連システム・外部連携**：
- ファイルシステム：シンボリックリンク作成
- 参照先パッケージのfetch処理

**権限による制御**：特になし（シンボリックリンク作成権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | repo fetchコマンド画面 | API連携 | 別パッケージソースへのシンボリックリンク作成 |

## 機能種別

データ連携 / ファイル操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| same_as | String | Yes | 参照先パッケージへの相対パス | パスが存在すること、無限ループでないこと |

### 入力データソース

- recipe.toml: sourceセクションのSameAs設定
- 参照先パッケージのrecipe.toml

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| source_dir | PathBuf | シンボリックリンクが配置されたパス |
| source_identifier | String | 参照先パッケージのsource_identifier |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/source` → `{参照先}/source`へのシンボリックリンク

## 処理フロー

### 処理シーケンス

```
1. 参照先の解決
   ├─ 相対パスから絶対パスを構築
   └─ 無限ループ検出（パス深度50以上でエラー）
2. 参照先レシピの読み込み
   └─ CookRecipe::from_path()で参照先のレシピを解析
3. 参照先のソース取得（再帰的）
   └─ fetch()を再帰的に呼び出し
4. シンボリックリンク作成
   ├─ sourceディレクトリが既にディレクトリの場合：エラー
   └─ symlink({same_as}/source, source)
5. ソース情報の引き継ぎ
   └─ 参照先のsource_identifierを取得して使用
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[相対パスを解決]
    B --> C{パス深度が50未満か}
    C -->|No| D[エラー: 無限ループ検出]
    C -->|Yes| E{参照先パスが存在するか}
    E -->|No| F[エラー: パスが存在しない]
    E -->|Yes| G[参照先レシピを読み込み]
    G --> H[参照先のfetch実行（再帰）]
    H --> I{sourceがシンボリックリンクか}
    I -->|Yes| J[既存リンクを使用]
    I -->|No| K{sourceがディレクトリか}
    K -->|Yes| L[エラー: ディレクトリが存在]
    K -->|No| M[シンボリックリンク作成]
    M --> N[参照先のsource_identifierを取得]
    J --> N
    N --> O[source_info.tomlを保存]
    O --> P[終了]
    D --> P
    F --> P
    L --> P
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 無限ループ検出 | パス内の'/'が50を超える場合はエラー | 常時 |
| BR-02 | ディレクトリ競合エラー | sourceがディレクトリの場合はエラー（手動削除を要求） | sourceが既に存在する場合 |
| BR-03 | 再帰的fetch | 参照先が未fetchの場合は先にfetchを実行 | 常時 |
| BR-04 | source_identifier継承 | 参照先のsource_identifierをそのまま使用 | 常時 |

### 計算ロジック

**無限ループ検出ロジック**：
```rust
if canon_dir.to_str().unwrap().chars().filter(|c| *c == '/').count() > 50 {
    return Err("Infinite loop detected");
}
```

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 無限ループ | same_as参照が循環している | 参照関係を確認 |
| - | パス不在 | 参照先パスが存在しない | 参照先パスを確認 |
| - | ディレクトリ競合 | sourceがシンボリックリンクでなくディレクトリ | sourceディレクトリを手動削除 |
| - | レシピ読込エラー | 参照先のrecipe.tomlが無効 | 参照先レシピを確認 |

### リトライ仕様

リトライは行わない（エラーは即時返却）

## トランザクション仕様

特になし（シンボリックリンク作成はアトミック操作）

## パフォーマンス要件

- シンボリックリンク作成は即座に完了
- 参照先の初回fetchには時間がかかる可能性あり

## セキュリティ考慮事項

- シンボリックリンクによるパストラバーサル：相対パスのみ許可
- 無限ループ検出によるDoS防止

## 備考

- このソースタイプはディスク容量を大幅に節約できる
- 参照先パッケージのソースが更新されると、参照元も自動的に更新される
- ビルド時に--filter=tree:0（shallow clone）と組み合わせることで効率化

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、SameAsソース設定を表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe::SameAs variantの定義 |

**読解のコツ**:
- `#[serde(untagged)]`によりTOMLのキー名から自動的にSameAsタイプと判定される
- same_asフィールドが存在すればSourceRecipe::SameAsとしてデシリアライズされる

**主要処理フロー**:
- **21-28行目**: SourceRecipe::SameAs variant定義
  - same_as: String - 参照先パッケージへの相対パス

#### Step 2: エントリーポイントを理解する

ソース取得処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch()関数のSameAs処理部分 |

**主要処理フロー**:
- **134-140行目**: SameAs処理ブロック
  - fetch_resolve_canon(): 参照先パスの解決
  - fetch(): 再帰的なソース取得
  - fetch_make_symlink(): シンボリックリンク作成
  - fetch_get_source_info(): ソース情報の取得

#### Step 3: ヘルパー関数を理解する

シンボリックリンク作成とパス解決の詳細を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | fetch_resolve_canon(), fetch_make_symlink() |

**主要処理フロー**:
- **431-452行目**: fetch_resolve_canon() - 参照先パスの解決と無限ループ検出
- **408-429行目**: fetch_make_symlink() - シンボリックリンク作成

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    └─ SourceRecipe::SameAs処理
           │
           ├─ fetch_resolve_canon(recipe_dir, same_as, is_host)
           │   ├─ canon_dir = recipe_dir.join(same_as)
           │   ├─ 無限ループ検出（'/' > 50）
           │   └─ CookRecipe::from_path(canon_dir, true, is_host)
           │
           ├─ fetch(参照先recipe, logger)  ← 再帰呼び出し
           │
           ├─ fetch_make_symlink(source_dir, same_as)
           │   ├─ source_dir.is_symlink() → 既存リンク使用
           │   ├─ source_dir.is_dir() → エラー
           │   └─ std::os::unix::fs::symlink()
           │
           └─ fetch_get_source_info(参照先recipe)
               └─ source_identifierを取得
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ source.same_as ─────▶ fetch_resolve_canon() ─────────▶ 参照先CookRecipe
                                    │
                                    ▼
                            fetch(参照先) ─────────────────▶ 参照先source/
                                    │
                                    ▼
                            fetch_make_symlink() ──────────▶ source → ../same_as/source
                                    │
                                    ▼
                            fetch_get_source_info() ───────▶ 参照先source_identifier
                                    │
                                    ▼
                            fetch_apply_source_info() ─────▶ source_info.toml
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | SameAs取得のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | SourceRecipe::SameAs定義 |
| fs.rs | `src/cook/fs.rs` | ソース | symlink()関数 |
| recipe.toml | 各レシピディレクトリ | 設定 | SameAsソース設定 |
