# 機能設計書 50-ソース識別子管理

## 概要

本ドキュメントは、cookbook（Redox OSのパッケージビルドシステム）におけるソース識別子管理機能の設計を記載する。この機能は、ビルドのコミット識別子とタイムスタンプを管理し、ビルドの再現性とトレーサビリティを確保する機能を提供する。

### 本機能の処理概要

ソース識別子管理機能は、ビルド実行時のcookbookリポジトリのコミットハッシュと現在時刻を記録し、ビルド成果物のメタデータに埋め込む。これにより、どのバージョンのcookbookでいつビルドされたかを追跡できる。

**業務上の目的・背景**：パッケージビルドの再現性を確保するため、ビルド時のcookbookのバージョン（コミットハッシュ）と実行時刻を記録する必要がある。これにより、問題が発生した場合のデバッグや、特定バージョンでのリビルドが可能となる。

**機能の利用シーン**：
- ビルド開始時の識別子初期化
- source_info.tomlへの識別子書き込み
- パッケージメタデータ（stage.toml）への埋め込み
- リモートパッケージ取得時の識別子復元

**主要な処理内容**：
1. プログラム起動時にIdentifierConfigを初期化する
2. 現在のディレクトリのGitコミットハッシュを取得する
3. 現在時刻をISO 8601形式で取得する
4. 静的変数に保存し、以降は参照のみで使用する

**関連システム・外部連携**：
- Gitリポジトリ（get_git_head_rev）
- dateコマンド（現在時刻取得）

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 10 | repo_builderコマンド画面 | API連携 | ビルドのコミット識別子とタイムスタンプ管理 |

## 機能種別

設定管理 / 識別子管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| なし | - | - | init_ident()は引数なし | - |

### 入力データソース

- 現在のディレクトリ（$PWD）のGitリポジトリ
- dateコマンドの出力

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| commit | String | cookbookリポジトリのコミットハッシュ |
| time | String | ISO 8601形式の現在時刻（例: "2026-01-22T15:30:00Z"） |

### 出力先

- IDENTIFIER_CONFIG静的変数
- source_info.toml（fetch時に書き込み）
- stage.toml（パッケージメタデータ）

## 処理フロー

### 処理シーケンス

```
1. init_ident関数呼び出し
   └─ プログラム起動時に一度だけ実行
2. IdentifierConfig::new()
   └─ 識別子の生成
3. get_git_head_rev()
   └─ 現在ディレクトリのコミットハッシュ取得
4. dateコマンド実行
   └─ date -u "+%Y-%m-%dT%H:%M:%SZ"
5. IDENTIFIER_CONFIG.set()
   └─ 静的変数に保存
6. [以降] get_ident()
   └─ 識別子の参照
```

### フローチャート

```mermaid
flowchart TD
    A[開始: init_ident] --> B[IdentifierConfig::new]
    B --> C[get_git_head_rev PWD]
    C --> D{成功?}
    D -->|Yes| E[コミットハッシュを保存]
    D -->|No| F[空文字列を保存]
    E --> G[date -u +format 実行]
    F --> G
    G --> H[ISO 8601形式の時刻を保存]
    H --> I[IDENTIFIER_CONFIG.set]
    I --> J[終了]

    K[get_ident呼び出し] --> L[IDENTIFIER_CONFIG.get]
    L --> M[&IdentifierConfigを返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-50-01 | 一度だけ初期化 | init_ident()は2回呼び出すとpanic | 常時 |
| BR-50-02 | Git失敗時 | コミット取得失敗時は空文字列を使用 | get_git_head_revがエラーの場合 |
| BR-50-03 | UTC時刻 | 時刻はUTCで記録（-u オプション） | 常時 |
| BR-50-04 | ISO 8601形式 | 時刻は"YYYY-MM-DDTHH:MM:SSZ"形式 | 常時 |

### 計算ロジック

```rust
impl IdentifierConfig {
    fn new() -> Self {
        // cookbookリポジトリのコミットハッシュを取得
        let (commit, _) = crate::cook::fs::get_git_head_rev(
            &std::env::current_dir().expect("unable to get $PWD"),
        )
        .unwrap_or(("".into(), false));

        // 現在時刻をISO 8601形式で取得
        let time = String::from_utf8_lossy(
            &Command::new("date")
                .arg("-u")
                .arg("+%Y-%m-%dT%H:%M:%SZ")
                .stdout(Stdio::piped())
                .output()
                .expect("Failed to get current ISO-formatted time")
                .stdout
                .trim_ascii(),
        )
        .into();

        IdentifierConfig { commit, time }
    }
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-50-01 | 初期化重複 | init_ident()が2回呼ばれた | panic（設計上発生しない） |
| E-50-02 | 未初期化アクセス | init前にget_identが呼ばれた | panic（設計上発生しない） |
| E-50-03 | $PWD取得失敗 | 現在ディレクトリが取得できない | panic |
| E-50-04 | date実行失敗 | dateコマンドが存在しない/失敗 | panic |

### リトライ仕様

リトライは行わない。エラーは致命的として扱う。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 初期化は起動時に一度だけ
- get_ident()はロックフリーで高速（OnceLock使用）

## セキュリティ考慮事項

- コミットハッシュとタイムスタンプは公開情報
- 機密情報は含まない

## 備考

- OnceLockにより、スレッドセーフに初期化と参照が可能
- chronoクレートを使用せず、軽量にdateコマンドで時刻を取得
- source_info.tomlに書き込まれる識別子はpkgクレートのSourceIdentifier構造体で管理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ident.rs | `src/cook/ident.rs` | IdentifierConfig構造体（7-10行目）- 識別子データ |

**読解のコツ**: IdentifierConfigはcommitとtimeの2フィールドを持つシンプルな構造体。

```rust
#[derive(Debug, Default)]
pub struct IdentifierConfig {
    pub commit: String,  // cookbookのコミットハッシュ
    pub time: String,    // ISO 8601形式の時刻
}
```

#### Step 2: 初期化処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ident.rs | `src/cook/ident.rs` | IdentifierConfig::new（13-31行目）- 識別子生成 |
| 2-2 | ident.rs | `src/cook/ident.rs` | init_ident関数（42-46行目）- 初期化エントリーポイント |

**主要処理フロー（new）**:
1. **14-17行目**: get_git_head_revでコミットハッシュ取得
2. **19-28行目**: dateコマンドでISO 8601形式の時刻取得
3. **30行目**: IdentifierConfig構築

**主要処理フロー（init_ident）**:
1. **43-45行目**: IdentifierConfig::new()を呼び出しOnceLockに設定

#### Step 3: 参照インターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ident.rs | `src/cook/ident.rs` | get_ident関数（36-40行目）- 識別子取得 |
| 3-2 | ident.rs | `src/cook/ident.rs` | IDENTIFIER_CONFIG静的変数（34行目）- OnceLockによる保持 |

**主要処理フロー（get_ident）**:
- **37-39行目**: IDENTIFIER_CONFIG.get()で参照を取得

#### Step 4: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | fetch.rs | `src/cook/fetch.rs` | fetch_apply_source_info（648-660行目）- source_info.toml書き込み |

**主要処理フロー**:
- **652行目**: get_ident()で識別子を取得
- **653-657行目**: SourceIdentifier構造体を構築
- **659行目**: serialize_and_writeでTOML書き込み

### プログラム呼び出し階層図

```
main() / repo.rs
    │
    └─ init_ident()
           │
           └─ IdentifierConfig::new()
                  │
                  ├─ get_git_head_rev($PWD)
                  │      └─ .git/HEAD読み込み
                  │
                  └─ Command::new("date")
                         └─ -u "+%Y-%m-%dT%H:%M:%SZ"

fetch_apply_source_info(recipe, source_identifier)
    │
    └─ get_ident()
           │
           └─ IDENTIFIER_CONFIG.get()
                  │
                  └─ &IdentifierConfig { commit, time }
```

### データフロー図

```
[入力]                      [処理]                           [出力]

$PWD ─────────────────────▶ get_git_head_rev()
                                  │
                                  ▼
                           コミットハッシュ
                                  │
                                  ▼
dateコマンド ─────────────▶ Command::new("date")
                                  │
                                  ▼
                           ISO 8601時刻
                                  │
              ┌───────────────────┴───────────────────┐
              ▼                                       ▼
    IdentifierConfig                          IDENTIFIER_CONFIG
    { commit, time }                          (OnceLock)
                                                      │
                                                      ▼
                                              get_ident() → &IdentifierConfig
                                                      │
                                                      ▼
                                              source_info.toml
                                              stage.toml
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ident.rs | `src/cook/ident.rs` | ソース | IdentifierConfig、init_ident、get_identの実装 |
| fetch.rs | `src/cook/fetch.rs` | ソース | fetch_apply_source_infoでの識別子使用 |
| fs.rs | `src/cook/fs.rs` | ソース | get_git_head_rev関数（コミットハッシュ取得） |
| source_info.toml | 各パッケージのtarget/ | 設定 | 識別子の保存先 |
