# 機能設計書 6-リモートパッケージ取得

## 概要

本ドキュメントは、Redox OSビルドシステムにおける事前ビルド済みパッケージをリモートリポジトリからダウンロードする機能の設計を記載する。build.kind=remoteによるバイナリパッケージの取得について詳述する。

### 本機能の処理概要

リモートパッケージ取得機能は、ソースからビルドする代わりに、事前にビルドされたパッケージ（pkgar形式）をリモートサーバーからダウンロードする機能である。ビルド時間の大幅な短縮と、ビルド環境の簡素化を実現する。

**業務上の目的・背景**：Redox OSのフルビルドには多くの時間とリソースが必要である。特にGCCやLLVMなどの大規模なパッケージは、ビルドに数時間かかることがある。これらのパッケージを事前にビルドしてリモートサーバーに配置し、必要に応じてダウンロードすることで、開発者のビルド時間を大幅に短縮し、開発効率を向上させる。

**機能の利用シーン**：
- ビルド時間を短縮したい場合
- ビルド環境が限られている場合（低スペックマシン）
- 同一バージョンのパッケージを複数環境で共有する場合
- CI/CD環境での高速なビルドパイプライン

**主要な処理内容**：
1. リモートサーバーからの公開鍵ダウンロード
2. パッケージファイル（.pkgar）のダウンロード
3. メタデータファイル（.toml）のダウンロード
4. 署名検証
5. ソース情報の記録

**関連システム・外部連携**：
- Redox OSパッケージサーバー（https://static.redox-os.org/pkg）
- wgetコマンド：HTTPダウンロード
- pkgarライブラリ：パッケージ検証

**権限による制御**：特になし（ネットワークアクセスとファイルシステムの書き込み権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | repo fetchコマンド画面 | API連携 | 事前ビルド済みパッケージのダウンロード |

## 機能種別

データ連携 / 外部API連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| build.template | "remote" | Yes | ビルドタイプをremoteに設定 | - |
| パッケージ名 | String | Yes | ダウンロードするパッケージ名 | 有効なパッケージ名 |

### 入力データソース

- recipe.toml: buildセクションのtemplate="remote"
- リモートサーバー：パッケージファイル、メタデータ、公開鍵

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id_ed25519.pub.toml | File | Ed25519公開鍵 |
| source.pkgar | File | パッケージアーカイブ |
| source.toml | File | パッケージメタデータ |
| source_info.toml | File | ソース識別子情報 |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/target/{target}/`

## 処理フロー

### 処理シーケンス

```
1. ターゲットディレクトリの作成
   └─ {レシピディレクトリ}/target/{target}/を作成
2. 公開鍵のダウンロード
   └─ {REMOTE_PKG_SOURCE}/id_ed25519.pub.tomlをダウンロード
3. 各パッケージ（メイン + オプショナル）について：
   ├─ .pkgarファイルのダウンロード
   │   └─ {REMOTE_PKG_SOURCE}/{target}/{パッケージ名}.pkgar
   └─ .tomlファイルのダウンロード
       └─ {REMOTE_PKG_SOURCE}/{target}/{パッケージ名}.toml
4. ソース情報の記録
   └─ source.tomlからsource_identifierを抽出してsource_info.tomlに保存
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[ターゲットディレクトリ作成]
    B --> C{オフラインモードか}
    C -->|No| D[公開鍵をダウンロード]
    C -->|Yes| E{公開鍵が存在するか}
    E -->|No| F[エラー: オフラインでファイル不在]
    E -->|Yes| G[パッケージリストを取得]
    D --> G
    G --> H[各パッケージについてループ]
    H --> I{オフラインモードか}
    I -->|No| J[.pkgarをダウンロード]
    I -->|Yes| K{.pkgarが存在するか}
    K -->|No| F
    K -->|Yes| L[.toml処理へ]
    J --> L
    L --> M{オフラインモードか}
    M -->|No| N[.tomlをダウンロード]
    M -->|Yes| O{.tomlが存在するか}
    O -->|No| F
    O -->|Yes| P[source_info.tomlを更新]
    N --> P
    P --> Q{次のパッケージがあるか}
    Q -->|Yes| H
    Q -->|No| R[終了]
    F --> R
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | リモートサーバーURL | https://static.redox-os.org/pkg を使用 | 常時 |
| BR-02 | ターゲット別パス | /{target}/ サブディレクトリからダウンロード | 常時 |
| BR-03 | オプショナルパッケージ対応 | メインパッケージとオプショナルパッケージを全てダウンロード | オプショナルパッケージ存在時 |
| BR-04 | オフラインモード対応 | ファイルが存在すればダウンロードをスキップ | COOKBOOK_OFFLINE時 |
| BR-05 | ソース識別子継承 | リモートパッケージのsource_identifierをそのまま使用 | 常時 |

### 計算ロジック

**リモートURL生成ロジック**：
```rust
fn get_remote_url(name: &str, ext: &str) -> String {
    format!("{}/{}/{}.{}", REMOTE_PKG_SOURCE, redoxer::target(), name, ext)
}
// REMOTE_PKG_SOURCE = "https://static.redox-os.org/pkg"
// 例: https://static.redox-os.org/pkg/x86_64-unknown-redox/gcc.pkgar
```

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムとHTTPダウンロードのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ダウンロードエラー | wgetコマンド失敗 | ネットワーク接続確認、URL確認 |
| - | オフラインエラー | オフラインモードでファイルが存在しない | 事前にファイルをダウンロード |
| - | パースエラー | source.tomlの解析失敗 | サーバー上のファイルを確認 |

### リトライ仕様

- TUIモードではユーザーにリトライ/スキップ/終了を選択させる

## トランザクション仕様

特になし（ダウンロードは非アトミック）

## パフォーマンス要件

- ダウンロード時間はネットワーク帯域とファイルサイズに依存
- ソースからビルドするよりも大幅に高速

## セキュリティ考慮事項

- Ed25519署名による真正性検証
- HTTPS通信による転送中の保護
- 公開鍵の信頼性確認

## 備考

- このビルドタイプはソースコードを使用しない
- パッケージサーバーのバージョンとローカル環境の互換性に注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Remoteビルドタイプを表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | BuildKind::Remote variantの定義 |

**読解のコツ**:
- `#[serde(tag = "template")]`によりtemplate値でビルドタイプが判定される
- template="remote"でBuildKind::Remoteとしてデシリアライズされる

**主要処理フロー**:
- **96-98行目**: BuildKind::Remote variant定義

#### Step 2: エントリーポイントを理解する

リモートパッケージ取得処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch_remote()関数 |

**主要処理フロー**:
- **517-580行目**: fetch_remote()関数
  - ターゲットディレクトリの作成
  - 公開鍵のダウンロード
  - パッケージリストの処理
  - .pkgarと.tomlのダウンロード
  - source_info.tomlの更新

#### Step 3: URL生成とダウンロードを理解する

リモートURLの生成とダウンロード処理を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fetch.rs | `src/cook/fetch.rs` | get_remote_url(), get_pubkey_url() |
| 3-2 | fs.rs | `src/cook/fs.rs` | download_wget() |

**主要処理フロー**:
- **503-511行目**: get_remote_url() - パッケージURLの生成
- **513-515行目**: get_pubkey_url() - 公開鍵URLの生成
- **278-288行目** (fs.rs): download_wget() - wgetによるダウンロード

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    └─ BuildKind::Remote処理
           │
           └─ fetch_remote(recipe_dir, recipe, offline_mode, logger)
                  │
                  ├─ create_target_dir(recipe_dir, target)
                  │
                  ├─ download_wget(get_pubkey_url(), source_pubkey, logger)
                  │   └─ オフライン時: offline_check_exists()
                  │
                  └─ for package in packages
                         │
                         ├─ package_source_paths(package, target_dir)
                         │
                         ├─ download_wget(get_remote_url(name, "pkgar"), source_pkgar, logger)
                         │
                         ├─ download_wget(get_remote_url(name, "toml"), source_toml, logger)
                         │
                         └─ fetch_apply_source_info_from_remote(recipe, &SourceIdentifier{...})
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ build.template=remote ─▶ fetch_remote() ──────────────▶ ダウンロード開始
                                    │
                                    ▼
リモートサーバー               get_pubkey_url() ───────────────▶ id_ed25519.pub.toml
  │                                │
  │                                ▼
  │                         get_remote_url(name, "pkgar") ─▶ source.pkgar
  │                                │
  │                                ▼
  │                         get_remote_url(name, "toml") ──▶ source.toml
  │                                │
  │                                ▼
  └────────────────────────▶ fetch_apply_source_info_from_remote() ─▶ source_info.toml
                                    │
                                    └─ commit_identifier
                                    └─ source_identifier
                                    └─ time_identifier
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | Remote取得のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | BuildKind::Remote定義 |
| lib.rs | `src/lib.rs` | ソース | REMOTE_PKG_SOURCE定数定義 |
| fs.rs | `src/cook/fs.rs` | ソース | download_wget(), offline_check_exists() |
| package.rs | `src/cook/package.rs` | ソース | package_source_paths()関数 |
| recipe.toml | 各レシピディレクトリ | 設定 | build.template="remote"設定 |
