# 機能設計書 7-パッチ適用

## 概要

本ドキュメントは、Redox OSビルドシステムにおけるソースコードへのパッチファイル適用機能の設計を記載する。patchコマンドを使用したソース修正について詳述する。

### 本機能の処理概要

パッチ適用機能は、ソース取得後にパッチファイルをソースコードに適用する機能である。外部ライブラリやツールをRedox OS向けに修正するために必要な変更を、再現可能な形で管理・適用する。

**業務上の目的・背景**：多くの外部ライブラリやツールは、Linux/Windows向けに開発されており、Redox OSで動作させるためには修正が必要である。これらの修正をパッチファイルとして管理することで、オリジナルソースの更新時にも修正を容易に再適用でき、変更内容を明確に追跡できる。

**機能の利用シーン**：
- 外部ライブラリのRedox OS対応（プラットフォーム固有コードの修正）
- バグ修正パッチの適用
- セキュリティパッチの適用
- コンパイラやビルドシステムの互換性修正

**主要な処理内容**：
1. パッチファイルの存在確認
2. パッチファイルの読み込み
3. patchコマンドによるパッチ適用
4. 複数パッチの順次適用

**関連システム・外部連携**：
- patchコマンド：GNU patch互換
- ファイルシステム：パッチファイルの読み込み

**権限による制御**：特になし（ファイルシステムの読み取り・書き込み権限が必要）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | repo fetchコマンド画面 | API連携 | ソースコードへのパッチファイル適用 |

## 機能種別

データ変換 / 外部コマンド実行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| patches | Vec<String> | No | 適用するパッチファイル名のリスト | ファイルが存在すること |
| source_dir | PathBuf | Yes | パッチを適用するソースディレクトリ | ディレクトリが存在すること |

### 入力データソース

- recipe.toml: sourceセクションのpatches配列
- ファイルシステム：レシピディレクトリ内のパッチファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 修正されたソースファイル | Files | パッチが適用されたソースファイル群 |

### 出力先

- ファイルシステム：`{レシピディレクトリ}/source/`内の修正されたファイル

## 処理フロー

### 処理シーケンス

```
1. パッチリストの取得
   └─ recipe.tomlのpatches配列を取得
2. 各パッチについてループ
   ├─ パッチファイルの存在確認
   │   └─ {レシピディレクトリ}/{パッチファイル名}
   ├─ パッチファイルの読み込み
   │   └─ fs::read_to_string()
   └─ patchコマンドの実行
       └─ patch --directory {source_dir} --strip=1
       └─ 標準入力からパッチ内容を渡す
3. エラー時は即座に中断
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{patchesリストが空か}
    B -->|Yes| C[終了（パッチなし）]
    B -->|No| D[次のパッチを取得]
    D --> E{パッチファイルが存在するか}
    E -->|No| F[エラー: パッチファイル不在]
    E -->|Yes| G[パッチファイルを読み込み]
    G --> H[patchコマンド実行]
    H --> I{適用成功か}
    I -->|No| J[エラー: パッチ適用失敗]
    I -->|Yes| K{次のパッチがあるか}
    K -->|Yes| D
    K -->|No| L[終了（成功）]
    F --> M[終了]
    J --> M
    C --> M
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | strip-level | --strip=1でパッチを適用（a/file → file形式対応） | 常時 |
| BR-02 | 順次適用 | パッチはpatches配列の順序で適用 | 複数パッチ存在時 |
| BR-03 | 失敗時中断 | 1つでもパッチ適用に失敗したら処理を中断 | 常時 |
| BR-04 | Git用リセット | Gitソースの場合、パッチ適用前にgit reset --hard実行 | SourceRecipe::Git時 |

### 計算ロジック

特になし

## データベース操作仕様

本機能ではデータベース操作は行わない（ファイルシステムとpatchコマンドのみ）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイル不在 | パッチファイルが存在しない | パッチファイルを確認 |
| - | 読込エラー | パッチファイルの読み込み失敗 | ファイル権限を確認 |
| - | 適用失敗 | patchコマンドがエラー終了 | パッチの互換性を確認 |

### リトライ仕様

リトライは行わない（パッチ適用エラーは即時返却）

## トランザクション仕様

特になし（部分的なパッチ適用は元に戻せない）

## パフォーマンス要件

- 小さなパッチファイルは即座に適用
- 大規模なパッチでもほぼ即座に完了

## セキュリティ考慮事項

- パッチファイルの信頼性：信頼できるソースからのパッチのみ使用
- パストラバーサル：パッチ内のパスを検証

## 備考

- パッチはUnified diff形式（diff -u）を想定
- --strip=1はgit diff出力のa/b/プレフィックスを除去
- ソースが更新された場合、パッチの再適用が必要な場合がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、パッチ設定を表現するデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | recipe.rs | `src/recipe.rs` | SourceRecipe内のpatchesフィールド |

**読解のコツ**:
- patchesはVec<String>型で、パッチファイル名のリスト
- Gitソース、Tarソースの両方でpatches指定可能

**主要処理フロー**:
- **49-50行目**: Git sourceのpatches
- **62-63行目**: Tar sourceのpatches

#### Step 2: エントリーポイントを理解する

パッチ適用処理の起点となる関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | fetch.rs | `src/cook/fetch.rs` | fetch_apply_patches()関数 |

**主要処理フロー**:
- **606-646行目**: fetch_apply_patches()関数
  - パッチファイルの存在確認
  - パッチファイルの読み込み
  - patchコマンドの実行
  - スクリプトの実行（存在する場合）

#### Step 3: コマンド実行を理解する

patchコマンドの実行方法を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | fs.rs | `src/cook/fs.rs` | run_command_stdin()関数 |

**主要処理フロー**:
- **220-252行目** (fs.rs): run_command_stdin() - 標準入力付きコマンド実行

### プログラム呼び出し階層図

```
fetch(recipe, logger)
    │
    ├─ SourceRecipe::Git処理 または SourceRecipe::Tar処理
    │
    └─ fetch_apply_patches(recipe_dir, patches, script, source_dir, logger)
           │
           ├─ for patch_name in patches
           │   │
           │   ├─ patch_file = recipe_dir.join(patch_name)
           │   │
           │   ├─ patch_file.is_file() → 存在確認
           │   │
           │   ├─ fs::read_to_string(patch_file) → 内容読み込み
           │   │
           │   └─ run_command_stdin(Command::new("patch"), patch_content, logger)
           │       │
           │       ├─ command.arg("--directory").arg(source_dir)
           │       ├─ command.arg("--strip=1")
           │       └─ stdin.write_all(patch_content)
           │
           └─ if let Some(script) = script
               │
               └─ run_command_stdin(Command::new("bash"), script, logger)
```

### データフロー図

```
[入力]                       [処理]                              [出力]

recipe.toml
  └─ source.patches ─────▶ パッチファイルリスト取得 ─────────▶ Vec<String>

{レシピディレクトリ}
  └─ *.patch ────────────▶ fs::read_to_string() ─────────────▶ パッチ内容

                          ▼
                    Command::new("patch")
                          │
                          ├─ --directory {source_dir}
                          ├─ --strip=1
                          └─ stdin: パッチ内容
                          │
                          ▼
source/
  └─ 修正前ファイル ──────▶ patchコマンド実行 ─────────────▶ 修正後ファイル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| fetch.rs | `src/cook/fetch.rs` | ソース | パッチ適用のメインロジック |
| recipe.rs | `src/recipe.rs` | ソース | patchesフィールド定義 |
| fs.rs | `src/cook/fs.rs` | ソース | run_command_stdin()関数 |
| *.patch | 各レシピディレクトリ | パッチ | Unified diff形式のパッチファイル |
| recipe.toml | 各レシピディレクトリ | 設定 | patches配列の設定 |
