# API設計書

## 概要

本ドキュメントは、Roslyn Language Server Protocol (LSP) の API 設計書です。Roslyn は .NET Compiler Platform であり、C# および Visual Basic のコンパイラ機能を提供します。本 LSP サーバーは、IDE やエディタに対してコード補完、定義への移動、リファクタリングなどの言語サービス機能を JSON-RPC ベースの Language Server Protocol を通じて提供します。

## 共通仕様

### ベースURL

```
stdio:// または websocket://
```

本プロトコルは Language Server Protocol (LSP) に準拠しており、標準入出力（stdio）または WebSocket を介した JSON-RPC 2.0 通信を使用します。

### 認証方式

LSP 通信は通常ローカルプロセス間通信のため、明示的な認証は不要です。Service Broker 接続時は別途認証が必要となる場合があります。

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Content-Length | ○ | メッセージボディのバイト長 |
| Content-Type | - | application/vscode-jsonrpc; charset=utf-8 |

### 共通エラーレスポンス

| エラーコード | 説明 |
| --- | --- |
| -32700 | Parse error - JSON のパースエラー |
| -32600 | Invalid Request - 不正なリクエスト形式 |
| -32601 | Method not found - 存在しないメソッド |
| -32602 | Invalid params - 不正なパラメータ |
| -32603 | Internal error - サーバー内部エラー |
| -32002 | Server not initialized - サーバー未初期化 |
| -32001 | Unknown error code - 不明なエラー |
| -32800 | Request cancelled - リクエストキャンセル |

## API一覧

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| サーバーライフサイクル | initialize | Request | サーバー初期化 |
| サーバーライフサイクル | initialized | Notification | 初期化完了通知 |
| ドキュメント同期 | textDocument/didOpen | Notification | ドキュメントオープン通知 |
| ドキュメント同期 | textDocument/didChange | Notification | ドキュメント変更通知 |
| ドキュメント同期 | textDocument/didClose | Notification | ドキュメントクローズ通知 |
| ドキュメント同期 | textDocument/didSave | Notification | ドキュメント保存通知 |
| 言語機能 | textDocument/completion | Request | コード補完 |
| 言語機能 | completionItem/resolve | Request | 補完項目解決 |
| 言語機能 | textDocument/hover | Request | ホバー情報取得 |
| 言語機能 | textDocument/signatureHelp | Request | シグネチャヘルプ |
| 言語機能 | textDocument/definition | Request | 定義への移動 |
| 言語機能 | textDocument/typeDefinition | Request | 型定義への移動 |
| 言語機能 | textDocument/references | Request | 参照検索 |
| 言語機能 | textDocument/documentHighlight | Request | ドキュメントハイライト |
| 言語機能 | textDocument/documentSymbol | Request | ドキュメントシンボル |
| 言語機能 | workspace/symbol | Request | ワークスペースシンボル検索 |
| 言語機能 | textDocument/codeAction | Request | コードアクション |
| 言語機能 | codeAction/resolve | Request | コードアクション解決 |
| 言語機能 | textDocument/codeLens | Request | CodeLens 取得 |
| 言語機能 | codeLens/resolve | Request | CodeLens 解決 |
| 言語機能 | textDocument/formatting | Request | ドキュメントフォーマット |
| 言語機能 | textDocument/rangeFormatting | Request | 範囲フォーマット |
| 言語機能 | textDocument/onTypeFormatting | Request | タイプ時フォーマット |
| 言語機能 | textDocument/rename | Request | リネーム |
| 言語機能 | textDocument/prepareRename | Request | リネーム準備 |
| 言語機能 | textDocument/foldingRange | Request | 折りたたみ範囲 |
| 言語機能 | textDocument/semanticTokens/full | Request | セマンティックトークン（全体） |
| 言語機能 | textDocument/semanticTokens/range | Request | セマンティックトークン（範囲） |
| 言語機能 | textDocument/inlayHint | Request | インレイヒント |
| 言語機能 | inlayHint/resolve | Request | インレイヒント解決 |
| 診断 | textDocument/diagnostic | Request | ドキュメント診断（プル型） |
| 診断 | workspace/diagnostic | Request | ワークスペース診断（プル型） |
| テスト | textDocument/runTests | Request | テスト実行 |
| ワークスペース | solution/open | Notification | ソリューションオープン |
| ワークスペース | project/open | Notification | プロジェクトオープン |

## 各APIエンドポイント定義

### サーバーライフサイクル

#### 1. initialize

サーバーを初期化し、クライアント・サーバー間のケーパビリティをネゴシエートします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `initialize` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/ServerLifetime/InitializeHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| processId | number \| null | ○ | 親プロセスID |
| rootUri | DocumentUri \| null | ○ | ルートフォルダURI |
| capabilities | ClientCapabilities | ○ | クライアントのケーパビリティ |
| trace | string | - | トレースレベル（off, messages, verbose） |
| workspaceFolders | WorkspaceFolder[] | - | ワークスペースフォルダ |

**リクエストボディ**

```json
{
  "processId": 1234,
  "rootUri": "file:///path/to/workspace",
  "capabilities": {
    "textDocument": {
      "completion": {
        "completionItem": {
          "snippetSupport": true
        }
      }
    }
  },
  "trace": "off"
}
```

**レスポンス（成功時）**

```json
{
  "capabilities": {
    "textDocumentSync": 2,
    "completionProvider": {
      "triggerCharacters": [".", "<"],
      "resolveProvider": true
    },
    "hoverProvider": true,
    "definitionProvider": true,
    "referencesProvider": true,
    "documentSymbolProvider": true,
    "workspaceSymbolProvider": true,
    "codeActionProvider": true,
    "documentFormattingProvider": true,
    "renameProvider": true,
    "foldingRangeProvider": true,
    "semanticTokensProvider": {
      "full": true,
      "range": true
    }
  }
}
```

---

### ドキュメント同期

#### 2. textDocument/didOpen

ドキュメントがクライアントで開かれたことをサーバーに通知します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/didOpen` |
| タイプ | Notification |
| ソースファイル | `src/LanguageServer/Protocol/Handler/DocumentChanges/DidOpenHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentItem | ○ | 開かれたドキュメント情報 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs",
    "languageId": "csharp",
    "version": 1,
    "text": "using System;\n\nclass Program { }"
  }
}
```

---

#### 3. textDocument/didChange

ドキュメントの内容が変更されたことをサーバーに通知します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/didChange` |
| タイプ | Notification |
| ソースファイル | `src/LanguageServer/Protocol/Handler/DocumentChanges/DidChangeHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | VersionedTextDocumentIdentifier | ○ | ドキュメント識別子とバージョン |
| contentChanges | TextDocumentContentChangeEvent[] | ○ | 変更内容 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs",
    "version": 2
  },
  "contentChanges": [
    {
      "range": {
        "start": { "line": 2, "character": 0 },
        "end": { "line": 2, "character": 0 }
      },
      "text": "    static void Main() { }\n"
    }
  ]
}
```

---

#### 4. textDocument/didClose

ドキュメントがクライアントで閉じられたことをサーバーに通知します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/didClose` |
| タイプ | Notification |
| ソースファイル | `src/LanguageServer/Protocol/Handler/DocumentChanges/DidCloseHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 閉じられたドキュメント |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  }
}
```

---

### 言語機能

#### 5. textDocument/completion

指定位置でのコード補完候補を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/completion` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Completion/CompletionHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | カーソル位置 |
| context | CompletionContext | - | 補完コンテキスト |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 10,
    "character": 5
  },
  "context": {
    "triggerKind": 1
  }
}
```

**レスポンス（成功時）**

```json
{
  "isIncomplete": false,
  "items": [
    {
      "label": "Console",
      "kind": 7,
      "detail": "System.Console",
      "documentation": "Represents the standard input, output, and error streams...",
      "insertText": "Console"
    },
    {
      "label": "WriteLine",
      "kind": 2,
      "detail": "void Console.WriteLine(string value)",
      "insertText": "WriteLine"
    }
  ]
}
```

---

#### 6. textDocument/hover

指定位置でのホバー情報（シンボル情報、ドキュメント）を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/hover` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Hover/HoverHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | カーソル位置 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 5,
    "character": 10
  }
}
```

**レスポンス（成功時）**

```json
{
  "contents": {
    "kind": "markdown",
    "value": "```csharp\nclass Program\n```\nThe main program class."
  },
  "range": {
    "start": { "line": 5, "character": 6 },
    "end": { "line": 5, "character": 13 }
  }
}
```

---

#### 7. textDocument/definition

シンボルの定義位置へ移動するための情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/definition` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Definitions/GoToDefinitionHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | カーソル位置 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 10,
    "character": 15
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "uri": "file:///path/to/definition.cs",
    "range": {
      "start": { "line": 5, "character": 0 },
      "end": { "line": 5, "character": 20 }
    }
  }
]
```

---

#### 8. textDocument/references

シンボルの全参照箇所を検索します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/references` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/References/FindAllReferencesHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | カーソル位置 |
| context | ReferenceContext | ○ | 参照検索コンテキスト |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 10,
    "character": 15
  },
  "context": {
    "includeDeclaration": true
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "uri": "file:///path/to/file1.cs",
    "range": {
      "start": { "line": 10, "character": 10 },
      "end": { "line": 10, "character": 20 }
    }
  },
  {
    "uri": "file:///path/to/file2.cs",
    "range": {
      "start": { "line": 25, "character": 5 },
      "end": { "line": 25, "character": 15 }
    }
  }
]
```

---

#### 9. textDocument/signatureHelp

関数・メソッドのシグネチャ情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/signatureHelp` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/SignatureHelp/SignatureHelpHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | カーソル位置 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 15,
    "character": 20
  }
}
```

**レスポンス（成功時）**

```json
{
  "signatures": [
    {
      "label": "void Console.WriteLine(string value)",
      "documentation": "Writes the specified string value, followed by the current line terminator.",
      "parameters": [
        {
          "label": "value",
          "documentation": "The value to write."
        }
      ]
    },
    {
      "label": "void Console.WriteLine(int value)",
      "parameters": [
        {
          "label": "value"
        }
      ]
    }
  ],
  "activeSignature": 0,
  "activeParameter": 0
}
```

---

#### 10. textDocument/codeAction

指定範囲で利用可能なコードアクション（クイックフィックス、リファクタリング）を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/codeAction` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/CodeActions/CodeActionsHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| range | Range | ○ | 対象範囲 |
| context | CodeActionContext | ○ | コードアクションコンテキスト |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "range": {
    "start": { "line": 10, "character": 0 },
    "end": { "line": 10, "character": 50 }
  },
  "context": {
    "diagnostics": [
      {
        "range": {
          "start": { "line": 10, "character": 10 },
          "end": { "line": 10, "character": 15 }
        },
        "message": "The variable 'x' is declared but never used",
        "severity": 2,
        "code": "CS0168"
      }
    ]
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "title": "Remove unused variable",
    "kind": "quickfix",
    "diagnostics": [
      {
        "code": "CS0168"
      }
    ],
    "isPreferred": true
  },
  {
    "title": "Extract method",
    "kind": "refactor.extract"
  }
]
```

---

#### 11. textDocument/rename

シンボルのリネームを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/rename` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Rename/RenameHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| position | Position | ○ | リネーム対象位置 |
| newName | string | ○ | 新しい名前 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "position": {
    "line": 5,
    "character": 10
  },
  "newName": "NewClassName"
}
```

**レスポンス（成功時）**

```json
{
  "documentChanges": [
    {
      "textDocument": {
        "uri": "file:///path/to/file.cs",
        "version": 1
      },
      "edits": [
        {
          "range": {
            "start": { "line": 5, "character": 6 },
            "end": { "line": 5, "character": 18 }
          },
          "newText": "NewClassName"
        },
        {
          "range": {
            "start": { "line": 15, "character": 10 },
            "end": { "line": 15, "character": 22 }
          },
          "newText": "NewClassName"
        }
      ]
    }
  ]
}
```

---

#### 12. textDocument/formatting

ドキュメント全体のフォーマットを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/formatting` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Formatting/FormatDocumentHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| options | FormattingOptions | ○ | フォーマットオプション |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "options": {
    "tabSize": 4,
    "insertSpaces": true
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "range": {
      "start": { "line": 0, "character": 0 },
      "end": { "line": 100, "character": 0 }
    },
    "newText": "// Formatted content...\n"
  }
]
```

---

#### 13. textDocument/documentSymbol

ドキュメント内のシンボル一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/documentSymbol` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Symbols/DocumentSymbolsHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "name": "Program",
    "kind": 5,
    "range": {
      "start": { "line": 2, "character": 0 },
      "end": { "line": 20, "character": 1 }
    },
    "selectionRange": {
      "start": { "line": 2, "character": 6 },
      "end": { "line": 2, "character": 13 }
    },
    "children": [
      {
        "name": "Main",
        "kind": 6,
        "range": {
          "start": { "line": 4, "character": 4 },
          "end": { "line": 10, "character": 5 }
        },
        "selectionRange": {
          "start": { "line": 4, "character": 16 },
          "end": { "line": 4, "character": 20 }
        }
      }
    ]
  }
]
```

---

#### 14. workspace/symbol

ワークスペース全体からシンボルを検索します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `workspace/symbol` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Symbols/WorkspaceSymbolsHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| query | string | ○ | 検索クエリ |

**リクエストボディ**

```json
{
  "query": "Program"
}
```

**レスポンス（成功時）**

```json
[
  {
    "name": "Program",
    "kind": 5,
    "location": {
      "uri": "file:///path/to/Program.cs",
      "range": {
        "start": { "line": 5, "character": 0 },
        "end": { "line": 50, "character": 1 }
      }
    },
    "containerName": "MyNamespace"
  }
]
```

---

#### 15. textDocument/foldingRange

ドキュメントの折りたたみ可能な範囲を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/foldingRange` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/FoldingRanges/FoldingRangesHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "startLine": 0,
    "startCharacter": 0,
    "endLine": 5,
    "endCharacter": 0,
    "kind": "imports"
  },
  {
    "startLine": 7,
    "startCharacter": 0,
    "endLine": 50,
    "endCharacter": 1,
    "kind": "region"
  }
]
```

---

#### 16. textDocument/semanticTokens/range

指定範囲のセマンティックトークンを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/semanticTokens/range` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/SemanticTokens/SemanticTokensRangeHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| range | Range | ○ | 対象範囲 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "range": {
    "start": { "line": 0, "character": 0 },
    "end": { "line": 100, "character": 0 }
  }
}
```

**レスポンス（成功時）**

```json
{
  "data": [0, 0, 5, 0, 0, 1, 0, 10, 1, 0, 0, 15, 8, 2, 1]
}
```

---

#### 17. textDocument/inlayHint

インレイヒント（パラメータ名、型情報など）を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/inlayHint` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/InlayHint/InlayHintHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| range | Range | ○ | 対象範囲 |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  },
  "range": {
    "start": { "line": 0, "character": 0 },
    "end": { "line": 50, "character": 0 }
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "position": { "line": 10, "character": 15 },
    "label": "value:",
    "kind": 2,
    "paddingLeft": false,
    "paddingRight": true
  },
  {
    "position": { "line": 12, "character": 4 },
    "label": "int",
    "kind": 1,
    "paddingLeft": false,
    "paddingRight": true
  }
]
```

---

#### 18. textDocument/codeLens

CodeLens（参照数、テスト実行ボタンなど）を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/codeLens` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/CodeLens/CodeLensHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/file.cs"
  }
}
```

**レスポンス（成功時）**

```json
[
  {
    "range": {
      "start": { "line": 5, "character": 0 },
      "end": { "line": 5, "character": 20 }
    },
    "command": {
      "title": "3 references",
      "command": "editor.action.showReferences"
    }
  },
  {
    "range": {
      "start": { "line": 10, "character": 0 },
      "end": { "line": 10, "character": 30 }
    },
    "command": {
      "title": "Run Test",
      "command": "dotnet.test.run"
    }
  }
]
```

---

### 診断

#### 19. textDocument/diagnostic (VS Internal)

ドキュメントの診断情報をプル型で取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/_vs_diagnostic` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Protocol/Handler/Diagnostics/DocumentPullDiagnosticHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | 対象ドキュメント |
| previousResultId | string | - | 前回の結果ID（差分取得用） |

**レスポンス（成功時）**

```json
[
  {
    "resultId": "result-123",
    "diagnostics": [
      {
        "range": {
          "start": { "line": 10, "character": 10 },
          "end": { "line": 10, "character": 15 }
        },
        "severity": 1,
        "code": "CS0103",
        "source": "csharp",
        "message": "The name 'x' does not exist in the current context"
      }
    ]
  }
]
```

---

### テスト

#### 20. textDocument/runTests

テストを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `textDocument/runTests` |
| タイプ | Request |
| ソースファイル | `src/LanguageServer/Microsoft.CodeAnalysis.LanguageServer/Testing/RunTestsHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| textDocument | TextDocumentIdentifier | ○ | テストファイル |
| range | Range | ○ | テスト対象範囲 |
| attachDebugger | boolean | ○ | デバッガアタッチ有無 |
| runSettingsPath | string | - | runsettingsファイルパス |

**リクエストボディ**

```json
{
  "textDocument": {
    "uri": "file:///path/to/tests/MyTests.cs"
  },
  "range": {
    "start": { "line": 10, "character": 0 },
    "end": { "line": 20, "character": 0 }
  },
  "attachDebugger": false,
  "runSettingsPath": null
}
```

**レスポンス（成功時）**

```json
[
  {
    "stage": "Running tests",
    "message": "Test 'MyTest' passed (50ms)",
    "progress": {
      "current": 1,
      "total": 5
    }
  }
]
```

---

### ワークスペース

#### 21. solution/open

ソリューションファイルを開きます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `solution/open` |
| タイプ | Notification |
| ソースファイル | `src/LanguageServer/Microsoft.CodeAnalysis.LanguageServer/HostWorkspace/OpenSolutionHandler.cs` |

**リクエストパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| solution | Uri | ○ | ソリューションファイルURI |

**リクエストボディ**

```json
{
  "solution": "file:///path/to/MySolution.sln"
}
```

---

## 備考

### サポート言語

本 LSP サーバーは以下の言語をサポートしています:
- C# (csharp)
- Visual Basic (vb)

### Visual Studio 拡張

本実装は標準 LSP に加え、Visual Studio 固有の拡張機能（VS Internal）もサポートしています。これらは `_vs_` プレフィックスを持つメソッド名で識別されます。

### 部分結果（Partial Results）

多くのリクエストはストリーミング形式での部分結果をサポートしています。`partialResultToken` パラメータを使用して進捗状況をリアルタイムで受信できます。

### Roslyn Analyzer 統合

診断機能は Roslyn Analyzer と統合されており、カスタムアナライザーからの診断情報も LSP 経由で取得可能です。
