# アーキテクチャ設計書

## 概要

本ドキュメントは、Roslyn（.NET Compiler Platform）プロジェクトのアーキテクチャ設計について記述する。Roslynは、C#およびVisual Basicコンパイラのオープンソース実装であり、コード解析ツールを構築するためのAPI基盤を提供する。本プロジェクトは、コンパイラ機能、IDE機能、言語サーバー機能を統合した大規模な.NETプラットフォームである。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| Visual Studio | MEF (Managed Extensibility Framework) | IDE機能統合、エディタ拡張 |
| Visual Studio Code | Language Server Protocol (LSP) | 言語サービス提供 |
| MSBuild | MSBuild Task API | ビルド時コンパイル処理 |
| .NET SDK | CLI統合 | dotnet buildコマンド統合 |
| NuGet | パッケージ配布 | コンパイラ・アナライザー配布 |
| Azure DevOps | CI/CD パイプライン | ビルド・テスト・リリース自動化 |
| OmniSharp | External Access API | 外部言語サーバー統合 |
| Razor | External Access API | Razor言語サービス統合 |

## レイヤー構成

### アーキテクチャスタイル

本プロジェクトは**レイヤードアーキテクチャ**を採用しており、以下の特徴を持つ：

1. **言語別分離**: C#とVisual Basicの実装を分離しつつ、共通のCore層で機能を共有
2. **機能別分離**: Compiler、Workspaces、Features、EditorFeatures、VisualStudioの各層に機能を分割
3. **ポータビリティ重視**: Portable（クロスプラットフォーム）とDesktop（Windows専用）を明確に分離
4. **External Access パターン**: 外部パートナー向けに制限されたAPI公開層を設置

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Compilers | ソースコードの構文解析、意味解析、コード生成 | Microsoft.CodeAnalysis、Microsoft.CodeAnalysis.CSharp、Microsoft.CodeAnalysis.VisualBasic |
| Workspaces | プロジェクト・ソリューション管理、ドキュメント管理 | Microsoft.CodeAnalysis.Workspaces、MSBuild Integration |
| Features | コード補完、リファクタリング、診断機能 | Microsoft.CodeAnalysis.Features、Analyzers、CodeFixes |
| EditorFeatures | エディタ統合機能（ハイライト、ナビゲーション） | Microsoft.CodeAnalysis.EditorFeatures |
| LanguageServer | Language Server Protocol実装 | Microsoft.CodeAnalysis.LanguageServer.Protocol |
| VisualStudio | Visual Studio固有の統合機能 | Microsoft.VisualStudio.LanguageServices |
| ExternalAccess | 外部パートナー向けAPI公開層 | ExternalAccess.Razor、ExternalAccess.FSharp、ExternalAccess.OmniSharp |

### レイヤー間の依存関係ルール

**依存の方向性:**
```
VisualStudio → EditorFeatures → Features → Workspaces → Compilers
                     ↓
              LanguageServer
```

**許可される参照:**
- 上位レイヤーから下位レイヤーへの参照
- 同一レイヤー内のCore（共通）からCSharp/VisualBasic（言語固有）への参照
- ExternalAccessレイヤーからの公開APIのみの参照

**禁止される参照:**
- 下位レイヤーから上位レイヤーへの参照
- Compilers層からWorkspaces層以上への参照
- Portable層からDesktop固有APIへの直接参照

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Compilation | ソースコードのコンパイル処理全般 | Compilers/Core、Compilers/CSharp、Compilers/VisualBasic |
| Syntax | 構文木の解析・操作 | Syntax、SyntaxTree、SyntaxNode |
| Semantics | 意味解析・型解決 | SemanticModel、Symbol、Binding |
| CodeGen | IL生成・PDB出力 | Emit、CodeGen、PEWriter |
| Workspace | プロジェクト・ドキュメント管理 | Workspace、Solution、Project、Document |
| Diagnostics | 診断・エラー報告 | Diagnostic、DiagnosticAnalyzer、CodeFix |
| Scripting | C#スクリプト実行 | Microsoft.CodeAnalysis.Scripting |
| Interactive | REPL環境 | Interactive、InteractiveHost |

### パッケージ構造

```
src/
├── Analyzers/                    # コード解析ルール
│   ├── Core/                     # 言語非依存アナライザー
│   ├── CSharp/                   # C#固有アナライザー
│   └── VisualBasic/              # VB固有アナライザー
├── Compilers/                    # コンパイラコア
│   ├── Core/                     # 言語非依存コンパイラ基盤
│   │   ├── Portable/             # クロスプラットフォームAPI
│   │   ├── MSBuildTask/          # MSBuildタスク
│   │   └── AnalyzerDriver/       # アナライザー実行基盤
│   ├── CSharp/                   # C#コンパイラ
│   │   ├── Portable/             # C#コンパイラAPI
│   │   └── csc/                  # csc.exeエントリーポイント
│   ├── VisualBasic/              # VBコンパイラ
│   │   ├── Portable/             # VBコンパイラAPI
│   │   └── vbc/                  # vbc.exeエントリーポイント
│   ├── Server/                   # VBCSCompiler（共有コンパイラサーバー）
│   └── Shared/                   # 共有コード
├── Workspaces/                   # ワークスペース管理
│   ├── Core/                     # 言語非依存ワークスペース
│   │   ├── Portable/             # クロスプラットフォームAPI
│   │   └── Desktop/              # Windows固有機能
│   ├── CSharp/                   # C#ワークスペース
│   ├── VisualBasic/              # VBワークスペース
│   ├── MSBuild/                  # MSBuildプロジェクト読み込み
│   └── Remote/                   # リモートワークスペース（OOP）
├── Features/                     # IDE機能（リファクタリング等）
│   ├── Core/Portable/            # 言語非依存機能
│   ├── CSharp/                   # C#固有機能
│   └── VisualBasic/              # VB固有機能
├── EditorFeatures/               # エディタ統合機能
│   ├── Core/                     # 言語非依存エディタ機能
│   ├── CSharp/                   # C#エディタ機能
│   ├── VisualBasic/              # VBエディタ機能
│   └── Text/                     # テキストバッファ統合
├── LanguageServer/               # Language Server Protocol
│   ├── Protocol/                 # LSPプロトコル実装
│   ├── Microsoft.CodeAnalysis.LanguageServer/  # スタンドアロンLSP
│   └── Microsoft.CommonLanguageServerProtocol.Framework/  # LSPフレームワーク
├── VisualStudio/                 # Visual Studio統合
│   ├── Core/                     # VS共通機能
│   ├── CSharp/                   # C# VS統合
│   ├── VisualBasic/              # VB VS統合
│   └── Setup/                    # VSIXパッケージング
├── Scripting/                    # C#スクリプティング
├── Interactive/                  # 対話型環境
├── ExpressionEvaluator/          # デバッガ式評価
├── RoslynAnalyzers/              # Roslyn自身のアナライザー
└── Tools/                        # 開発ツール
```

### コンポーネント依存関係

**主要コンポーネント間の依存関係:**

1. **Microsoft.CodeAnalysis (Core)** - 最下層のコンパイラAPI
   - 依存先: System.Collections.Immutable、System.Reflection.Metadata
   - 依存元: 全ての上位コンポーネント

2. **Microsoft.CodeAnalysis.CSharp / VisualBasic** - 言語固有コンパイラ
   - 依存先: Microsoft.CodeAnalysis
   - 依存元: Workspaces、Features

3. **Microsoft.CodeAnalysis.Workspaces.Common** - ワークスペースAPI
   - 依存先: Microsoft.CodeAnalysis、System.Composition
   - 依存元: Features、EditorFeatures、LanguageServer

4. **Microsoft.CodeAnalysis.Features** - IDE機能
   - 依存先: Workspaces、Scripting
   - 依存元: EditorFeatures、LanguageServer

5. **Microsoft.CodeAnalysis.LanguageServer.Protocol** - LSP実装
   - 依存先: Features、StreamJsonRpc
   - 依存元: LanguageServer実行可能ファイル

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| 組み込みDB | SQLite (SQLitePCLRaw) | bundle_green | ワークスペース永続化キャッシュ |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| インメモリキャッシュ | 内部実装 | 構文木・意味モデルキャッシュ | セッション中 |
| RecoverableTextAndVersion | 内部実装 | ドキュメントテキストキャッシュ | LRU方式 |
| SolutionState | 内部実装 | ソリューション状態スナップショット | 不変オブジェクト |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| System.Threading.Channels | NuGetパッケージ | 非同期タスクキュー |
| System.IO.Pipelines | NuGetパッケージ | 高性能I/Oパイプライン |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Microsoft.CodeAnalysis.Elfie | NuGetパッケージ | シンボル検索・インデックス |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| StreamJsonRpc | NuGetパッケージ | JSON-RPC通信（LSP） |
| Humanizer.Core | NuGetパッケージ | 文字列人間可読化 |
| ICSharpCode.Decompiler | NuGetパッケージ | メタデータからソース復元 |
| Microsoft.DiaSymReader | NuGetパッケージ | PDBシンボル読み取り |

## データフロー

### リクエスト処理の流れ

**コンパイル処理フロー（csc/vbc）:**

1. コマンドライン引数を解析し、CompilationOptionsを構築
2. ソースファイルをパースしてSyntaxTreeを生成
3. メタデータ参照からシンボルをロード
4. Compilationオブジェクトを構築
5. SemanticModelを通じて意味解析を実行
6. Emitメソッドでアセンブリ（DLL/EXE）とPDBを生成
7. 診断結果をコンソールに出力

**IDE機能リクエストフロー（コード補完の例）:**

1. ユーザーが入力操作を行う
2. EditorFeaturesがテキスト変更イベントを受信
3. Workspaceがドキュメントを更新しCompilationを再構築
4. Features層のCompletionServiceがCompletionItemsを生成
5. 言語固有のCompletionProviderが候補を収集
6. EditorFeaturesが結果をエディタUIに返却
7. UIが補完リストを表示

### 非同期処理の流れ

**バックグラウンド診断処理:**

1. ドキュメント変更がWorkspaceに通知される
2. DiagnosticServiceがバックグラウンドキューに診断要求を追加
3. DiagnosticAnalyzerDriverが登録済みアナライザーを実行
4. 診断結果がDiagnosticUpdatedEventとして発行
5. サブスクライバー（エラー一覧、エディタ装飾）が結果を受信して更新

**Out-of-Process (OOP) 処理:**

1. ホストプロセス（VS/LSP）がServiceHubに接続
2. リモートサービス呼び出しをStreamJsonRpcで送信
3. ServiceHubプロセスで処理を実行
4. 結果をシリアライズしてホストに返却
5. ホストが結果をデシリアライズして利用

### データ永続化の流れ

**ワークスペースキャッシュ:**

1. Compilationまたはシンボル情報を生成
2. チェックサムを計算してキーを生成
3. SQLiteデータベースにバイナリ形式で保存
4. 次回アクセス時にキャッシュからロード
5. キャッシュミス時は再計算

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| InternalsVisibleTo | アセンブリ属性 | 内部APIアクセス制御 |
| RestrictedInternalsVisibleTo | アセンブリ属性 | パートナー向け制限付きAPI公開 |
| Public API Analyzers | ビルド時検証 | 公開API変更の追跡と検証 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | CodeAnalysisEventSource (ETW) | Windows ETWトレース |
| パフォーマンス計測 | PerformanceSensitiveAttribute | 開発時アナライザー警告 |
| テレメトリ | Visual Studio Telemetry | Azure Application Insights（VS統合時） |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| コンパイルエラー | Diagnostic発行 | DiagnosticSeverity.Error |
| 警告 | Diagnostic発行 | DiagnosticSeverity.Warning |
| 内部エラー | 例外キャッチ＋Diagnostic | CS8078等の内部エラーコード |
| キャンセル | OperationCanceledException | 処理中断、部分結果なし |
| タイムアウト | CancellationToken | 設定可能なタイムアウト値 |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| Compilation | 不変オブジェクト（Immutable） | スナップショット分離 |
| Workspace変更 | ApplyChanges API | 排他的ロック |
| ドキュメント更新 | VersionStamp | 楽観的並行制御 |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 不変性 (Immutability) | Compilation, SyntaxTree, Symbol | WithXxx()メソッドによる新インスタンス生成 |
| ファクトリーパターン | SyntaxFactory, SymbolFactory | 構文ノード・シンボル生成 |
| Visitorパターン | SyntaxVisitor, SymbolVisitor | 構文木・シンボル木の走査 |
| Flywightパターン | SyntaxToken, SyntaxTrivia | メモリ効率化 |
| 依存性注入 | MEF (System.Composition) | IDE機能の拡張性 |
| 内部DSL | Source Generator Attributes | コード生成メタデータ定義 |

### コーディング規約

**命名規則:**
- パブリックAPIはPascalCase
- プライベートフィールドは_camelCase
- 非同期メソッドはAsync接尾辞
- キャンセル可能メソッドはCancellationToken引数を最後に配置

**パフォーマンス規約:**
- ホットパスでのアロケーション最小化（ObjectPool使用）
- ImmutableArray優先（配列コピー回避）
- Span/Memory活用（ヒープアロケーション回避）
- PerformanceSensitiveAttributeでホットパス明示

**API設計規約:**
- PublicAPI.txtで公開API管理
- 破壊的変更禁止（既存APIの維持）
- オプショナルパラメータは新オーバーロードで追加
- Obsoleteは2リリース後に削除

## 備考

- 本プロジェクトは.NET Foundation傘下のオープンソースプロジェクトである
- .NET SDK 10.0.102、Visual Studio 17.14.0に対応
- Microsoft.DotNet.Arcade.Sdkを使用したビルドインフラストラクチャを採用
- Azure DevOpsによるCI/CDパイプラインで品質管理を実施
