# Roslyn (.NET Compiler Platform) コードリーディングガイドライン

## はじめに

このガイドラインは、Roslynのコードベースを効率的に理解するための手引きです。
C#に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

**プロジェクト概要:**
- Roslynは、C#およびVisual Basicコンパイラのオープンソース実装
- コード解析ツールを構築するためのAPIサーフェスを提供
- .NET Foundation配下のプロジェクト

---

## 1. 言語基礎

> このセクションでは、C#の基本構文と概念を解説します。

### 1.1 プログラム構造

C#のプログラムは名前空間(namespace)、クラス(class)、メソッドで構成されます。Roslynでは以下のような構造が一般的です。

```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:32-42
namespace Microsoft.CodeAnalysis
{
    /// <summary>
    /// The compilation object is an immutable representation of a single invocation of the
    /// compiler.
    /// </summary>
    public abstract partial class Compilation
    {
        // クラス本体
    }
}
```

**特徴:**
- `namespace` でコードを論理的にグループ化
- `abstract` キーワードで抽象クラスを定義
- `partial` キーワードでクラスを複数ファイルに分割可能
- XMLドキュメントコメント (`///`) でAPIドキュメントを記述

### 1.2 データ型と変数

Roslynでは不変(immutable)なデータ構造が多用されています。

```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:70-72
protected readonly IReadOnlyDictionary<string, string> _features;

private readonly Lazy<int?> _lazyDataSectionStringLiteralThreshold;
```

**主要なデータ型:**
| 型 | 説明 | 使用例 |
|---|---|---|
| `ImmutableArray<T>` | 不変配列 | シンボル、参照のコレクション |
| `IReadOnlyDictionary<K,V>` | 読み取り専用辞書 | 設定、オプション |
| `Lazy<T>` | 遅延初期化ラッパー | 計算コストの高いプロパティ |
| `T?` | Nullable参照型 | オプショナルな値 |

### 1.3 制御構造

```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:98-125
protected static IReadOnlyDictionary<string, string> SyntaxTreeCommonFeatures(IEnumerable<SyntaxTree> trees)
{
    IReadOnlyDictionary<string, string>? set = null;

    foreach (var tree in trees)
    {
        var treeFeatures = tree.Options.Features;
        if (set == null)
        {
            set = treeFeatures;
        }
        else
        {
            if ((object)set != treeFeatures && !set.SetEquals(treeFeatures))
            {
                throw new ArgumentException(CodeAnalysisResources.InconsistentSyntaxTreeFeature, nameof(trees));
            }
        }
    }

    if (set == null)
    {
        set = ImmutableDictionary<string, string>.Empty;
    }

    return set;
}
```

### 1.4 関数/メソッド定義

Roslynでは以下のパターンが頻出します。

```csharp
// 抽象メソッド
internal abstract AnalyzerDriver CreateAnalyzerDriver(
    ImmutableArray<DiagnosticAnalyzer> analyzers,
    AnalyzerManager analyzerManager,
    SeverityFilter severityFilter);

// プロパティ
public abstract string Language { get; }

// 静的ファクトリメソッド
public static CompletionService? GetService(Document? document)
    => document?.GetLanguageService<CompletionService>();
```

### 1.5 モジュール/インポート

```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:1-30
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using Microsoft.CodeAnalysis.CodeGen;
using Microsoft.CodeAnalysis.Collections;
using Microsoft.CodeAnalysis.Diagnostics;
using Roslyn.Utilities;
```

**名前空間の構造:**
- `Microsoft.CodeAnalysis` - コンパイラのコア機能
- `Microsoft.CodeAnalysis.CSharp` - C#固有の実装
- `Microsoft.CodeAnalysis.VisualBasic` - VB固有の実装
- `Roslyn.Utilities` - 内部ユーティリティ

---

## 2. プロジェクト固有の概念

> このセクションでは、Roslyn特有の概念を解説します。

### 2.1 コンパイラAPIの主要概念

#### Compilation (コンパイル)
コンパイラの1回の呼び出しを表す不変オブジェクトです。

```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:34-41
/// <summary>
/// The compilation object is an immutable representation of a single invocation of the
/// compiler. Although immutable, a compilation is also on-demand, and will realize and cache
/// data as necessary.
/// </summary>
public abstract partial class Compilation
```

#### Symbol (シンボル)
名前空間、クラス、メソッド、パラメータなどのプログラム要素を表します。

```csharp
// ファイル: src/Compilers/CSharp/Portable/Symbols/Symbol.cs:26-32
/// <summary>
/// The base class for all symbols (namespaces, classes, method, parameters, etc.) that are
/// exposed by the compiler.
/// </summary>
[DebuggerDisplay("{GetDebuggerDisplay(), nq}")]
internal abstract partial class Symbol : ISymbolInternal, IFormattable
```

#### Syntax (構文)
ソースコードの構文構造を表します。

```csharp
// ファイル: src/Compilers/CSharp/Portable/Syntax/SyntaxFactory.cs:22-26
/// <summary>
/// A class containing factory methods for constructing syntax nodes, tokens and trivia.
/// </summary>
public static partial class SyntaxFactory
```

#### Workspace (ワークスペース)
プロジェクトとドキュメントのアクティブなセットへのアクセスを提供します。

```csharp
// ファイル: src/Workspaces/Core/Portable/Workspace/Workspace.cs:30-37
/// <summary>
/// A workspace provides access to a active set of source code projects and documents and their
/// associated syntax trees, compilations and semantic models.
/// </summary>
public abstract partial class Workspace : IDisposable
```

### 2.2 プロジェクト独自のパターン

#### Binder パターン
名前を解決してシンボルに変換する責務を持ちます。

```csharp
// ファイル: src/Compilers/CSharp/Portable/Binder/Binder.cs:17-22
/// <summary>
/// A Binder converts names in to symbols and syntax nodes into bound trees. It is context
/// dependent, relative to a location in source code.
/// </summary>
internal partial class Binder
```

#### DiagnosticAnalyzer パターン
コード品質問題を検出するアナライザーの基底クラスです。

```csharp
// ファイル: src/Analyzers/CSharp/Analyzers/UseImplicitObjectCreation/CSharpUseImplicitObjectCreationDiagnosticAnalyzer.cs:19-27
[DiagnosticAnalyzer(LanguageNames.CSharp)]
internal sealed class CSharpUseImplicitObjectCreationDiagnosticAnalyzer()
    : AbstractBuiltInCodeStyleDiagnosticAnalyzer(
        IDEDiagnosticIds.UseImplicitObjectCreationDiagnosticId,
        EnforceOnBuildValues.UseImplicitObjectCreation,
        CSharpCodeStyleOptions.ImplicitObjectCreationWhenTypeIsApparent,
        // ...
    )
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `**/Portable/**` | .NET Standardターゲットのポータブルコード | `src/Compilers/Core/Portable/` |
| `**/Desktop/**` | デスクトップ固有の実装 | `src/Workspaces/Core/Desktop/` |
| `**Test**` | テストプロジェクト | `src/Compilers/CSharp/Test/` |
| `**Utilities**` | ユーティリティクラス群 | `src/Workspaces/CoreTestUtilities/` |
| `CSharp/` | C#言語固有の実装 | `src/Compilers/CSharp/` |
| `VisualBasic/` | VB言語固有の実装 | `src/Compilers/VisualBasic/` |
| `Core/` | 言語非依存のコア実装 | `src/Features/Core/` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Abstract*` | 抽象基底クラス | `AbstractBuiltInCodeStyleDiagnosticAnalyzer` |
| `*Provider` | サービスプロバイダー | `CompletionProvider`, `CodeFixProvider` |
| `*Service` | サービスクラス | `CompletionService`, `CodeFixService` |
| `*Factory` | ファクトリクラス | `SyntaxFactory` |
| `*Handler` | イベント/リクエストハンドラ | `ILspServiceRequestHandler` |
| `*Analyzer` | 診断アナライザー | `CSharpUseImplicitObjectCreationDiagnosticAnalyzer` |
| `*CodeFix*` | コード修正プロバイダー | `CSharpUseImplicitObjectCreationCodeFixProvider` |
| `I*` | インターフェース | `ISymbol`, `ILanguageService` |
| `_*` | プライベートフィールド | `_lazyWorkspaceFixersMap` |
| `*Options` | 設定/オプションクラス | `CompletionOptions`, `CSharpCompilationOptions` |
| `*Info` | 情報保持クラス | `SolutionInfo`, `ProjectInfo` |
| `*Helpers` | ヘルパー関数群 | `PopulateSwitchStatementHelpers` |
| `*Extensions` | 拡張メソッド群 | `CSharpExtensions` |
| `*Internal` | 内部実装向けインターフェース | `ISymbolInternal` |

### 3.3 プログラム分類一覧

| 分類 | 説明 | ディレクトリ |
|-----|------|------------|
| Compiler | コンパイラコア | `src/Compilers/` |
| Workspace | ワークスペースAPI | `src/Workspaces/` |
| Features | IDE機能 | `src/Features/` |
| EditorFeatures | エディタ機能 | `src/EditorFeatures/` |
| Analyzers | コードアナライザー | `src/Analyzers/` |
| LanguageServer | LSPプロトコル実装 | `src/LanguageServer/` |
| VisualStudio | VS統合 | `src/VisualStudio/` |
| Scripting | C#スクリプティング | `src/Scripting/` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
roslyn-main/
├── .azure-devops/          # Azure DevOps CI/CD設定
├── .github/                # GitHub Actions, Issue templates
├── docs/                   # ドキュメント
│   ├── analyzers/         # アナライザードキュメント
│   ├── compilers/         # コンパイラドキュメント
│   ├── contributing/      # 貢献ガイド
│   └── wiki/              # Wiki
├── eng/                    # エンジニアリングスクリプト・設定
├── src/                    # ソースコード
│   ├── Analyzers/         # コードスタイルアナライザー
│   ├── CodeStyle/         # コードスタイル設定
│   ├── Compilers/         # C#/VBコンパイラ
│   │   ├── Core/          # 言語共通コード
│   │   ├── CSharp/        # C#コンパイラ
│   │   ├── VisualBasic/   # VBコンパイラ
│   │   └── Shared/        # 共有コード
│   ├── Dependencies/      # 内部依存関係
│   ├── EditorFeatures/    # エディタ機能
│   ├── Features/          # IDE機能
│   ├── Interactive/       # インタラクティブ機能
│   ├── LanguageServer/    # LSPサーバー実装
│   ├── Scripting/         # スクリプティングサポート
│   ├── VisualStudio/      # Visual Studio統合
│   └── Workspaces/        # ワークスペースAPI
└── Directory.Build.props   # MSBuild共通設定
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/Compilers/Core/Portable/` | コンパイラのコアロジック | `Compilation.cs`, `Symbol.cs` |
| `src/Compilers/CSharp/Portable/` | C#コンパイラ実装 | `CSharpCompilation.cs`, `Binder.cs` |
| `src/Workspaces/Core/Portable/` | ワークスペースAPI | `Workspace.cs`, `Solution.cs` |
| `src/Features/Core/Portable/` | IDE機能のコア | `CompletionService.cs`, `CodeFixService.cs` |
| `src/Analyzers/Core/Analyzers/` | 共通アナライザー基盤 | `AbstractBuiltInCodeStyleDiagnosticAnalyzer.cs` |
| `src/LanguageServer/Protocol/` | LSPプロトコル実装 | `Handler/*.cs` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Roslynは**レイヤードアーキテクチャ**を採用しています。

```
┌─────────────────────────────────────────────────────────────┐
│                    Visual Studio / Editor                   │
│                   (src/VisualStudio/, EditorFeatures/)      │
├─────────────────────────────────────────────────────────────┤
│                    Language Server Protocol                 │
│                      (src/LanguageServer/)                  │
├─────────────────────────────────────────────────────────────┤
│                       IDE Features                          │
│           (src/Features/ - Completion, CodeFix, etc.)       │
├─────────────────────────────────────────────────────────────┤
│                        Workspaces                           │
│              (src/Workspaces/ - Solution, Project)          │
├─────────────────────────────────────────────────────────────┤
│                        Compilers                            │
│    (src/Compilers/ - Syntax, Symbols, Binding, Emit)        │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Presentation | VS統合、エディタ機能 | `src/VisualStudio/` |
| Language Server | LSPプロトコル実装 | `src/LanguageServer/Protocol/Handler/` |
| Features | IDE機能（補完、修正、リファクタリング） | `src/Features/Core/Portable/` |
| Workspaces | ソリューション/プロジェクト管理 | `src/Workspaces/Core/Portable/` |
| Compilers | コンパイル、構文解析、セマンティック解析 | `src/Compilers/` |

### 5.3 データフロー

1. **ソースコード入力** -> Syntaxレイヤーで構文木生成
2. **構文木** -> Binderで名前解決、型チェック
3. **バインド済みツリー** -> セマンティックモデル構築
4. **セマンティックモデル** -> Featuresレイヤーで補完・診断
5. **診断結果** -> EditorFeaturesでUI表示

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

#### コンパイラのエントリーポイント
```csharp
// ファイル: src/Compilers/CSharp/csc/Program.cs
// C#コンパイラのメインエントリーポイント
```

#### ワークスペースのエントリーポイント
```csharp
// ファイル: src/Workspaces/Core/Portable/Workspace/Workspace.cs:70-72
protected Workspace(HostServices host, string? workspaceKind)
{
    Kind = workspaceKind;
    Services = host.CreateWorkspaceServices(this);
}
```

### 6.2 ビジネスロジック

#### Compilation（コンパイル）
```csharp
// ファイル: src/Compilers/Core/Portable/Compilation/Compilation.cs:77-96
internal Compilation(
    string? name,
    ImmutableArray<MetadataReference> references,
    IReadOnlyDictionary<string, string> features,
    bool isSubmission,
    SemanticModelProvider? semanticModelProvider,
    AsyncQueue<CompilationEvent>? eventQueue)
{
    this.AssemblyName = name;
    this.ExternalReferences = references;
    // ...
}
```

#### Binder（名前解決）
```csharp
// ファイル: src/Compilers/CSharp/Portable/Binder/Binder.cs:30-36
internal Binder(CSharpCompilation compilation)
{
    RoslynDebug.Assert(compilation != null);
    this.Flags = compilation.Options.TopLevelBinderFlags;
    this.Compilation = compilation;
}
```

### 6.3 データアクセス

Roslynはファイルシステムへの直接アクセスではなく、抽象化レイヤーを使用します。

```csharp
// ファイル: src/Workspaces/Core/Portable/Workspace/Workspace.cs:56-57
/// <summary>
/// Current solution. Must be locked with <see cref="_serializationLock"/> when writing to it.
/// </summary>
private Solution _latestSolution;
```

### 6.4 ユーティリティ/共通機能

```csharp
// Roslyn.Utilities - 内部ユーティリティ
using Roslyn.Utilities;

// PooledObjects - オブジェクトプーリング
using Microsoft.CodeAnalysis.PooledObjects;

// ImmutableCollections - 不変コレクション
using System.Collections.Immutable;
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Abstract Template Method | 言語別実装の共通基盤 | 高 | `Abstract*Analyzer.cs` |
| Factory Method | オブジェクト生成の抽象化 | 高 | `SyntaxFactory.cs` |
| MEF Export/Import | 依存性注入 | 高 | `CodeFixService.cs` |
| Lazy Initialization | 遅延初期化 | 高 | `_lazy*` フィールド |
| Immutable Object | 不変オブジェクト | 高 | `Compilation`, `Solution` |

### 各パターンの詳細

#### パターン1: Abstract Template Method

**目的:** C#とVBで共通のロジックを共有しつつ、言語固有の実装を分離

**実装例:**
```csharp
// ファイル: src/Analyzers/Core/Analyzers/UseConditionalExpression/AbstractUseConditionalExpressionDiagnosticAnalyzer.cs
internal abstract class AbstractUseConditionalExpressionDiagnosticAnalyzer<...>
    : AbstractBuiltInCodeStyleDiagnosticAnalyzer
{
    protected abstract bool TryMatchPattern(...);
}

// ファイル: src/Analyzers/CSharp/Analyzers/UseConditionalExpression/CSharpUseConditionalExpressionForReturnDiagnosticAnalyzer.cs
internal sealed class CSharpUseConditionalExpressionForReturnDiagnosticAnalyzer
    : AbstractUseConditionalExpressionForReturnDiagnosticAnalyzer<...>
{
    protected override bool TryMatchPattern(...) { /* C#固有実装 */ }
}
```

**解説:** 共通ロジックは基底クラスで定義し、言語固有の部分を派生クラスでオーバーライドします。

#### パターン2: MEF (Managed Extensibility Framework)

**目的:** 拡張性と疎結合を実現

**実装例:**
```csharp
// ファイル: src/Features/Core/Portable/CodeFixes/Service/CodeFixService.cs:39-63
[Export(typeof(ICodeFixService)), Shared]
internal sealed partial class CodeFixService : ICodeFixService
{
    [ImportingConstructor]
    public CodeFixService(
        [ImportMany] IEnumerable<Lazy<IErrorLoggerService>> loggers,
        [ImportMany] IEnumerable<Lazy<CodeFixProvider, CodeChangeProviderMetadata>> fixers,
        [ImportMany] IEnumerable<Lazy<IConfigurationFixProvider, CodeChangeProviderMetadata>> configurationProviders)
    {
        _errorLoggers = [.. loggers];
        _fixers = [.. fixers];
    }
}
```

**解説:** `[Export]`/`[Import]`属性でコンポーネントを自動的に発見・注入します。

#### パターン3: Immutable Objects with WithXxx

**目的:** スレッドセーフな不変オブジェクトの操作

**実装例:**
```csharp
// Compilationは不変で、変更は新しいオブジェクトを返す
var newCompilation = compilation.WithOptions(newOptions);
var newCompilation = compilation.AddSyntaxTrees(tree);
```

**解説:** オブジェクトを直接変更せず、新しいインスタンスを生成します。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: コード補完（IntelliSense）

**概要:** ユーザーが`.`を入力した時の補完候補生成

**処理フロー:**
```
CompletionService.GetCompletionsAsync()
    → CompletionProvider.ProvideCompletionsAsync()
    → SemanticModel.GetSymbolInfo()
    → 補完アイテム生成
```

**詳細な追跡:**

1. **CompletionService** (`src/Features/Core/Portable/Completion/CompletionService.cs:31-52`)
   ```csharp
   public abstract partial class CompletionService : ILanguageService
   {
       public static CompletionService? GetService(Document? document)
           => document?.GetLanguageService<CompletionService>();
   }
   ```

2. **CompletionProvider呼び出し** - 各プロバイダーが補完候補を生成

3. **SemanticModel利用** - シンボル情報を取得して候補をフィルタリング

#### 例2: 診断アナライザーの実行

**概要:** コードスタイル問題の検出

**処理フロー:**
```
DiagnosticAnalyzer登録
    → RegisterSyntaxNodeAction()
    → AnalyzeSyntax()
    → ReportDiagnostic()
```

**詳細な追跡:**

1. **アナライザー初期化** (`src/Analyzers/CSharp/Analyzers/UseImplicitObjectCreation/CSharpUseImplicitObjectCreationDiagnosticAnalyzer.cs:31-32`)
   ```csharp
   protected override void InitializeWorker(AnalysisContext context)
       => context.RegisterSyntaxNodeAction(AnalyzeSyntax, SyntaxKind.ObjectCreationExpression);
   ```

2. **構文ノード解析** (`src/Analyzers/CSharp/Analyzers/UseImplicitObjectCreation/CSharpUseImplicitObjectCreationDiagnosticAnalyzer.cs:34-60`)
   ```csharp
   private void AnalyzeSyntax(SyntaxNodeAnalysisContext context)
   {
       // 解析ロジック
       context.ReportDiagnostic(DiagnosticHelper.Create(...));
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `docs/compilers/` - コンパイラアーキテクチャ
3. `docs/contributing/` - 開発ガイド

#### 特定機能を理解したい場合
1. `docs/features/` - 機能別ドキュメント
2. 対象の`src/Features/*/`ディレクトリ
3. テストコード（`*Test/`）

#### 改修作業を行う場合
1. 対象機能のドキュメント
2. 関連するソースコード
3. テストケース
4. `CONTRIBUTING.md`

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 初回参照 |
| `CONTRIBUTING.md` | 貢献ガイド | 開発開始時 |
| `docs/compilers/` | コンパイラ設計 | コンパイラ理解時 |
| `docs/features/` | IDE機能設計 | 機能開発時 |
| `docs/wiki/` | 詳細Wiki | 詳細調査時 |
| `docs/analyzers/` | アナライザーガイド | アナライザー開発時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `partial class`が複数ファイルに分散していて、どこに何があるかわからない
A: ファイル名に機能別サフィックスが付いています。例: `Compilation.cs`, `Compilation.EmitStream.cs`。Visual Studioの「定義へ移動」やGrepで探してください。

#### Q: `Abstract*`クラスの言語固有実装がどこにあるかわからない
A: 基底クラスが`Core/`にあれば、実装は`CSharp/`や`VisualBasic/`の同名パスにあります。

#### Q: MEFの依存関係がどう解決されているかわからない
A: `[Export]`属性を持つクラスを検索し、`[Import]`で注入される箇所を確認してください。

#### Q: テストを探したい
A: 対応するテストは`*Test/`ディレクトリにあり、本体と同じパス構造になっています。

### デバッグのヒント

1. **デバッガアタッチ**: Visual Studioから`devenv.exe`を選択してRoslynをデバッグ
2. **ログ出力**: `Internal.Log`名前空間のロギング機能を活用
3. **構文木ビューア**: Syntax Visualizer拡張機能でASTを可視化
4. **シンボルビューア**: SemanticModelの`GetSymbolInfo`で型情報を確認

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Compilation | コンパイラの1回の呼び出しを表す不変オブジェクト |
| Symbol | プログラム要素（名前空間、型、メンバー等）を表す |
| SyntaxTree | ソースコードの構文木 |
| SemanticModel | 構文木にセマンティック情報を付加したモデル |
| Workspace | プロジェクト/ドキュメントの集合を管理 |
| Solution | 複数プロジェクトを含むソリューション |
| Binder | 名前解決と型チェックを行うコンポーネント |
| DiagnosticAnalyzer | コード問題を検出するアナライザー |
| CodeFixProvider | コード問題を修正するプロバイダー |
| LSP | Language Server Protocol（エディタ統合プロトコル） |
| MEF | Managed Extensibility Framework（依存性注入） |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/Compilers/Core/Portable/Compilation/Compilation.cs` | コンパイルの核心 | Compilation抽象クラス |
| `src/Compilers/CSharp/Portable/Symbols/Symbol.cs` | シンボル基底 | Symbol抽象クラス |
| `src/Compilers/CSharp/Portable/Binder/Binder.cs` | 名前解決 | Binderクラス |
| `src/Compilers/CSharp/Portable/Syntax/SyntaxFactory.cs` | 構文生成 | SyntaxFactory静的クラス |
| `src/Workspaces/Core/Portable/Workspace/Workspace.cs` | ワークスペース | Workspace抽象クラス |
| `src/Features/Core/Portable/Completion/CompletionService.cs` | コード補完 | CompletionService |
| `src/Features/Core/Portable/CodeFixes/Service/CodeFixService.cs` | コード修正 | CodeFixService |
| `src/LanguageServer/Protocol/Handler/` | LSPハンドラ | 各種リクエストハンドラ |

### C. 参考資料

- [Roslyn公式ドキュメント](https://docs.microsoft.com/en-us/dotnet/csharp/roslyn-sdk/)
- [Roslyn GitHub Wiki](https://github.com/dotnet/roslyn/wiki)
- [C# Language Specification](https://docs.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/)
- [.NET API Reference](https://docs.microsoft.com/en-us/dotnet/api/)
- [MEF Programming Guide](https://docs.microsoft.com/en-us/dotnet/framework/mef/)
