# セキュリティ設計書

## 概要

本ドキュメントは、Roslyn (.NET Compiler Platform) プロジェクトにおけるセキュリティ設計を記載します。Roslynはコンパイラプラットフォームであり、ユーザー認証やセッション管理といった一般的なWebアプリケーションのセキュリティ機能ではなく、コンパイラ固有のセキュリティ機能（アセンブリ署名、コードアクセスセキュリティ属性の処理、暗号化処理等）を提供します。

## 認証設計

### 認証方式

本プロジェクトはコンパイラであり、エンドユーザー認証機能は提供していません。代わりに、アセンブリレベルでの信頼性を確保するための機能を提供します。

**アセンブリ署名による信頼性確認**

| 項目 | 説明 |
| --- | --- |
| StrongName署名 | アセンブリの発行元を証明し、改ざんを検出する機能 |
| PublicKey検証 | 公開鍵を使用したアセンブリ同一性の検証 |
| InternalsVisibleTo検証 | friend assemblyアクセス時の公開鍵照合 |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 該当なし | コンパイラはステートレスで動作 |
| セッション固定化対策 | 該当なし | セッション概念なし |

## 認可設計

### 権限体系

Roslynはアセンブリ間のアクセス制御を以下の仕組みで実現しています。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| Strong-named Assembly | フル信頼アクセス | 署名済みアセンブリは明示的に許可された場合、内部メンバーへアクセス可能 |
| Weak-named Assembly | 制限付きアクセス | 署名なしアセンブリは公開メンバーのみアクセス可能 |
| Friend Assembly | 内部アクセス | InternalsVisibleTo属性で指定されたアセンブリは内部メンバーへアクセス可能 |

### アクセス制御

**InternalsVisibleTo (IVT) チェック機構**

Roslynは `ISymbolExtensions.PerformIVTCheck` メソッドにより、アセンブリ間のfriendアクセスを検証します。

検証パターン:

| ケース | グラント側署名 | グラント先指定 | 要求側署名 | 公開鍵一致 | 結果 |
| --- | --- | --- | --- | --- | --- |
| 1 | 署名あり | 強名 | 署名あり | 一致 | アクセス許可 |
| 2 | 署名あり | 強名 | 署名あり | 不一致 | アクセス拒否 |
| 3 | 署名あり | 強名 | 署名なし | - | アクセス拒否 |
| 4 | 署名なし | - | 署名あり | - | 警告付きアクセス許可 |
| 5 | 署名なし | - | 署名なし | - | アクセス許可 |

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | 該当なし（コンパイラはネットワーク通信を直接行わない） |
| HSTS | 該当なし |

Roslynはスタンドアロンのコンパイラであり、ネットワーク通信は行いません。NuGetパッケージの取得やソース管理との連携はビルドシステムの責任範囲となります。

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| アセンブリ署名 | RSA署名 (SHA-1/SHA-256) |
| ソースファイルハッシュ | SHA-1, SHA-256 |
| アセンブリハッシュ | SHA-1, SHA-256, SHA-384, SHA-512, MD5 |
| 決定論的ビルドキー | XxHash128 (非暗号学的ハッシュ) |

**署名ユーティリティ (`SigningUtilities.cs`)**

```
- RSA署名: PKCS#1パディングによるRSA署名
- ハッシュアルゴリズム: ECMA-335準拠のためSHA-1を使用（レガシー互換性）
- 強名署名サイズ: 公開鍵長に基づいて128バイト以上
```

**暗号学的ハッシュプロバイダ (`CryptographicHashProvider.cs`)**

| ハッシュアルゴリズム | 用途 | サイズ |
| --- | --- | --- |
| SHA-1 | アセンブリ署名、ソースハッシュ（レガシー） | 160 bit |
| SHA-256 | ソースハッシュ（デフォルト）、PDBチェックサム | 256 bit |
| SHA-384 | アセンブリハッシュ（オプション） | 384 bit |
| SHA-512 | アセンブリハッシュ（オプション） | 512 bit |
| MD5 | アセンブリハッシュ（レガシー互換） | 128 bit |

### 機密情報管理

**暗号鍵の管理**

| 鍵タイプ | 管理方法 | ファイル/コンテナ |
| --- | --- | --- |
| KeyPair (秘密鍵) | SNK/PFXファイル | `StrongNameKeys.KeyPair` |
| PublicKey (公開鍵) | メタデータ埋め込み | `StrongNameKeys.PublicKey` |
| KeyContainer | Windows CSP | `StrongNameKeys.KeyContainer` |

**CryptoBlobParser による鍵解析**

- PUBLICKEYBLOB (0x06): 公開鍵のみ
- PRIVATEKEYBLOB (0x07): 秘密鍵を含む鍵ペア
- ECMA特殊キー: .NET Framework標準アセンブリ用の特殊公開鍵

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | 該当なし（コンパイラはHTML出力を行わない） |
| SQLインジェクション | 該当なし（データベース接続なし） |
| CSRF | 該当なし（Webインターフェースなし） |
| コードインジェクション | ソースコードの構文解析によりコード構造を検証 |
| バッファオーバーフロー | .NETマネージドコードによる自動境界チェック |

**入力検証**

| 入力タイプ | 検証方法 |
| --- | --- |
| ソースファイル | Lexer/Parserによる構文解析 |
| 公開鍵文字列 | 16進数形式検証 (`StrongNameKeys.IsValidPublicKeyString`) |
| 暗号ブロブ | 構造検証 (`CryptoBlobParser.IsValidPublicKey`) |
| アセンブリ参照 | メタデータ読み取り時の整合性検証 |

## セキュリティ属性の処理

Roslynは.NETのセキュリティ属性を認識・処理します。

| 属性 | 名前空間 | 処理内容 |
| --- | --- | --- |
| SecurityCriticalAttribute | System.Security | セキュリティクリティカルコードのマーキング |
| SecuritySafeCriticalAttribute | System.Security | 安全なセキュリティクリティカルコードのマーキング |
| SecurityTransparentAttribute | System.Security | セキュリティ透過コードのマーキング |
| SuppressUnmanagedCodeSecurityAttribute | System.Security | アンマネージドコードセキュリティチェックの抑制 |
| PrincipalPermissionAttribute | System.Security.Permissions | 主体ベースのアクセス制御 |
| HostProtectionAttribute | System.Security.Permissions | ホスト保護リソースの指定 |
| PermissionSetAttribute | System.Security.Permissions | 権限セットの指定 |

## 決定論的ビルド

**DeterministicKeyBuilder による再現性保証**

決定論的ビルドオプション (`/deterministic`) が有効な場合、同一入力から同一出力を保証します。

| 記録項目 | 目的 |
| --- | --- |
| コンパイラバージョン | ツールチェーン識別 |
| ソースファイルチェックサム | 入力ファイルの一意識別 |
| 公開鍵 | アセンブリ同一性の確認 |
| コンパイルオプション | ビルド設定の再現性 |
| 参照アセンブリMVID | 依存関係の一意識別 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| コンパイル診断 | エラー、警告、情報メッセージ | コンパイル実行時のみ |
| セキュリティ診断 | 署名失敗、鍵読み取りエラー | コンパイル実行時のみ |

Roslynはコンパイラとして診断情報を出力しますが、永続的なログ保存機能は提供していません。ログの保持はビルドシステムやCI/CDパイプラインの責任となります。

## エラー処理

| エラーコード | 説明 |
| --- | --- |
| ERR_PublicKeyFileFailure | 公開鍵ファイルの読み取り失敗 |
| ERR_PublicKeyContainerFailure | 鍵コンテナへのアクセス失敗 |
| ERR_BadCompilationOptionValue | 不正なコンパイルオプション値 |
| ERR_SecurityAttributeMissingAction | セキュリティ属性にSecurityActionが指定されていない |
| ERR_SecurityAttributeInvalidAction | 無効なSecurityAction値 |
| ERR_PrincipalPermissionInvalidAction | PrincipalPermissionに無効なSecurityAction |
| ERR_FriendAssemblySNReq | strong-named assemblyがweak-named friendを指定 |
| CS7029 | strong-named assemblyがweak-named assemblyを参照 |

## 備考

- Roslynは.NET Compiler Platformとして、セキュリティ機能の「実装」ではなく「コンパイル時の検証と処理」を担当します
- ECMA-335標準への準拠のため、一部でSHA-1が使用されますが、これは仕様要件によるものです
- 決定論的ビルド機能により、ビルドの再現性とサプライチェーンセキュリティを向上させることができます
- CoreCLR/非Windows環境ではIClrStrongNameインターフェースが利用できないため、鍵コンテナ署名は使用できません
