# 機能設計書 124-Copilot連携

## 概要

本ドキュメントは、RoslynプロジェクトにおけるGitHub Copilot連携機能の設計仕様を記載する。GitHub Copilotと統合し、AIによるコード分析、リネーム候補生成、ドキュメント生成などの支援機能を提供する。

### 本機能の処理概要

Copilot連携機能は、GitHub CopilotのAI機能をRoslyn IDE機能と統合し、開発者の生産性を向上させる各種AI支援機能を提供する。コード分析、Smart Rename、On-The-Fly Docs、ドキュメントコメント生成などの機能を提供する。

**業務上の目的・背景**：現代の開発環境ではAI支援が不可欠となっている。GitHub CopilotをRoslynの言語サービスと統合することで、コンテキストを理解した高精度なAI支援が可能となる。これにより、開発者はより効率的にコードを記述・リファクタリング・文書化できる。

**機能の利用シーン**：
- リネーム操作時にAIによる適切な名前候補を取得（Smart Rename）
- ホバー時にシンボルの説明をAIで生成（On-The-Fly Docs）
- ドキュメントコメントのAI生成
- コード分析とCopilot診断
- NotImplementedException実装の自動生成

**主要な処理内容**：
1. Copilotサービスの可用性確認
2. コード分析リクエストの送信とキャッシュ管理
3. リネーム候補の生成と表示
4. オンデマンドドキュメント生成
5. 変更分析とリファインメントセッション

**関連システム・外部連携**：GitHub Copilot API、Visual Studio Copilot拡張、Roslyn言語サービス。

**権限による制御**：Copilot利用にはGitHub Copilotサブスクリプションが必要。クォータ管理あり。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 21 | リネームフライアウト | 補助機能 | SmartRename機能によるAI支援リネーム候補の表示 |
| 24 | スマートリネームユーザー入力コンボボックス | 主機能 | Copilotによるリネーム候補生成と表示 |
| 25 | スマートリネームステータスコントロール | 補助機能 | Copilot処理状態の表示 |
| 26 | オンザフライドキュメントビュー | 補助機能 | Copilot連携によるドキュメント生成 |

## 機能種別

AI支援 / コード分析 / ドキュメント生成 / リファクタリング支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 対象ドキュメント | 有効なRoslynドキュメント |
| span | TextSpan | No | 対象範囲 | ドキュメント内の有効な範囲 |
| promptTitle | string | Yes | 分析プロンプトタイトル | GetAvailablePromptTitlesで取得可能な値 |
| onTheFlyDocsInfo | OnTheFlyDocsInfo | Yes | ドキュメント生成用コンテキスト | シンボル情報とコード情報 |
| proposal | DocumentationCommentProposal | Yes | ドキュメントコメント提案 | 構造化されたコメント情報 |

### 入力データソース

- 現在編集中のドキュメント
- カーソル位置のシンボル情報
- セマンティックモデル
- GitHub Copilot API

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Diagnostic | array | Copilot分析による診断結果 |
| responseString | string | AI生成テキスト |
| isQuotaExceeded | bool | クォータ超過フラグ |
| responseDictionary | Dictionary | 構造化された応答 |
| ImplementationDetails | ImmutableDictionary | 実装詳細 |

### 出力先

- Visual Studio診断ウィンドウ
- リネームフライアウト（候補表示）
- QuickInfoホバー表示
- ドキュメントコメント挿入

## 処理フロー

### 処理シーケンス

```
1. Copilot可用性確認（IsAvailableAsync）
   └─ Copilotサービスの状態確認
2. 利用可能プロンプト取得（GetAvailablePromptTitlesAsync）
   └─ 対象ドキュメントで利用可能な分析タイプを取得
3. 分析実行（AnalyzeDocumentAsync）
   ├─ Copilotへのリクエスト送信
   └─ 結果のキャッシュ
4. 結果取得（GetCachedDocumentDiagnosticsAsync）
   └─ キャッシュから診断結果を取得
5. リファインメント（StartRefinementSessionAsync）
   └─ 変更に基づく追加分析
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[IsAvailableAsync]
    B --> C{Copilot利用可能?}
    C -->|No| D[終了]
    C -->|Yes| E[GetAvailablePromptTitlesAsync]
    E --> F[AnalyzeDocumentAsync]
    F --> G{キャッシュあり?}
    G -->|Yes| H[GetCachedDocumentDiagnosticsAsync]
    G -->|No| I[Copilot API呼び出し]
    I --> J[結果キャッシュ]
    J --> H
    H --> K[結果表示]
    K --> L{リファインメント必要?}
    L -->|Yes| M[StartRefinementSessionAsync]
    L -->|No| D
    M --> D
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | クォータ管理 | APIリクエストにはクォータ制限がある | Copilot API呼び出し時 |
| BR-002 | ファイル除外 | ワークスペースで除外されたファイルはCopilot対象外 | IsFileExcludedAsync判定 |
| BR-003 | プロンプトID | プロンプトタイトルがプロンプトのIDとして機能 | 分析タイプ選択時 |
| BR-004 | キャッシュ利用 | 同一ドキュメント・範囲の分析結果はキャッシュから取得 | GetCachedDocumentDiagnosticsAsync |

### 計算ロジック

テキスト変更の正規化:
- `TryNormalizeCopilotTextChanges`: 変更をソート・オーバーラップチェック
- `GetTextSpansFromTextChanges`: 変更からTextSpanを計算

```csharp
// 変更適用後の新しいSpan計算
totalDelta += newTextLength - change.Span.Length;
newSpans.Add(new TextSpan(change.Span.Start + totalDelta, newTextLength));
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし（インメモリキャッシュ） |

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | isQuotaExceeded | APIクォータ超過 | クォータリセットまで待機 |
| - | IsAvailable=false | Copilotサービス利用不可 | Copilotサブスクリプション確認 |
| - | オーバーラップエラー | 変更がオーバーラップ | 正規化失敗時はdefault返却 |

### リトライ仕様

クォータ超過時のリトライは実装されていない。ユーザーに状態を通知し、クォータリセット後に再試行を促す。

## トランザクション仕様

トランザクション管理なし。API呼び出しは非同期・非トランザクション。

## パフォーマンス要件

- キャッシュ活用によるAPI呼び出し削減
- 非同期処理によるUI応答性維持
- CancellationTokenによるキャンセルサポート

## セキュリティ考慮事項

- コードはGitHub Copilot APIに送信される
- ワークスペース除外設定によるファイル制御
- クォータ管理による乱用防止
- 機密ファイルの除外機能

## 備考

- Copilot機能はVisual Studio拡張として提供
- クォータ超過時はisQuotaExceeded=trueを返却
- ドキュメントコメント生成は構造化された形式で返却

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: サービスインターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ICopilotCodeAnalysisService.cs | `src/Features/Core/Portable/Copilot/ICopilotCodeAnalysisService.cs` | Copilot分析サービスのメインインターフェース |

**主要処理フロー**:
- **26行目**: `IsAvailableAsync` - Copilot利用可能性確認
- **36行目**: `GetAvailablePromptTitlesAsync` - 利用可能プロンプト取得
- **46行目**: `AnalyzeDocumentAsync` - ドキュメント分析実行
- **57行目**: `GetCachedDocumentDiagnosticsAsync` - キャッシュ結果取得
- **65行目**: `StartRefinementSessionAsync` - リファインメントセッション開始
- **72-78行目**: `GetOnTheFlyDocsPromptAsync`/`GetOnTheFlyDocsResponseAsync` - On-The-Fly Docs
- **83行目**: `IsFileExcludedAsync` - ファイル除外判定
- **89行目**: `GetDocumentationCommentAsync` - ドキュメントコメント生成
- **100-103行目**: `ImplementNotImplementedExceptionsAsync` - 実装生成

#### Step 2: ユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CopilotUtilities.cs | `src/Features/Core/Portable/Copilot/CopilotUtilities.cs` | テキスト変更処理ユーティリティ |

**主要処理フロー**:
- **19-32行目**: `GetNewTextAndChangedSpans` - 変更適用と新Span計算
- **34-51行目**: `GetTextSpansFromTextChanges` - 変更からSpan計算
- **60-81行目**: `TryNormalizeCopilotTextChanges` - 変更の正規化
- **83-93行目**: `ThrowIfNotNormalized` - 正規化検証

**読解のコツ**: `totalDelta`による位置調整のロジックに注目

#### Step 3: 関連サービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ICopilotOptionsService.cs | `src/Features/Core/Portable/Copilot/ICopilotOptionsService.cs` | Copilotオプション設定 |
| 3-2 | CopilotOptions.cs | `src/Features/Core/Portable/Copilot/CopilotOptions.cs` | オプション定義 |
| 3-3 | ICopilotChangeAnalysisService.cs | `src/Features/Core/Portable/Copilot/ICopilotChangeAnalysisService.cs` | 変更分析サービス |

### プログラム呼び出し階層図

```
ICopilotCodeAnalysisService
    │
    ├─ IsAvailableAsync
    │      └─ Copilotサービス状態確認
    │
    ├─ GetAvailablePromptTitlesAsync
    │      └─ ドキュメント種別に応じたプロンプト一覧
    │
    ├─ AnalyzeDocumentAsync
    │      ├─ Copilot API呼び出し
    │      └─ 結果キャッシュ保存
    │
    ├─ GetCachedDocumentDiagnosticsAsync
    │      └─ キャッシュから診断結果取得
    │
    ├─ GetOnTheFlyDocsPromptAsync
    │      └─ ドキュメント生成用プロンプト構築
    │
    ├─ GetOnTheFlyDocsResponseAsync
    │      └─ Copilot応答取得
    │
    └─ GetDocumentationCommentAsync
           └─ ドキュメントコメント生成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Document ─────────────▶ AnalyzeDocumentAsync ──────▶ Diagnostic[]
    │                         │
    │                         ▼
    │                    キャッシュ
    │                         │
TextSpan ─────────────▶ GetCachedDiagnosticsAsync ─▶ Diagnostic[]

OnTheFlyDocsInfo ─────▶ GetOnTheFlyDocsPromptAsync ─▶ prompt
    │                         │
    └─────────────────▶ GetOnTheFlyDocsResponseAsync ─▶ (response, isQuotaExceeded)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ICopilotCodeAnalysisService.cs | `src/Features/Core/Portable/Copilot/ICopilotCodeAnalysisService.cs` | ソース | メインサービスインターフェース |
| CopilotUtilities.cs | `src/Features/Core/Portable/Copilot/CopilotUtilities.cs` | ソース | ユーティリティ関数 |
| ICopilotOptionsService.cs | `src/Features/Core/Portable/Copilot/ICopilotOptionsService.cs` | ソース | オプションサービス |
| CopilotOptions.cs | `src/Features/Core/Portable/Copilot/CopilotOptions.cs` | ソース | オプション定義 |
| ICopilotChangeAnalysisService.cs | `src/Features/Core/Portable/Copilot/ICopilotChangeAnalysisService.cs` | ソース | 変更分析サービス |
| CopilotChangeAnalysis.cs | `src/Features/Core/Portable/Copilot/CopilotChangeAnalysis.cs` | ソース | 変更分析実装 |
| IRemoteCopilotChangeAnalysisService.cs | `src/Features/Core/Portable/Copilot/IRemoteCopilotChangeAnalysisService.cs` | ソース | リモート分析サービス |
| ImplementationDetails.cs | `src/Features/Core/Portable/Copilot/ImplementationDetails.cs` | ソース | 実装詳細データ |
| Extensions.cs | `src/Features/Core/Portable/Copilot/Extensions.cs` | ソース | 拡張メソッド |
| IProposalAdjusterService.cs | `src/Features/Core/Portable/Copilot/IProposalAdjusterService.cs` | ソース | 提案調整サービス |
