# 機能設計書 128-名前付き引数使用

## 概要

本ドキュメントは、Roslynプロジェクトにおける名前付き引数使用機能の設計仕様を記載する。位置引数に明示的な引数名を追加するリファクタリング機能を提供する。

### 本機能の処理概要

名前付き引数使用機能は、メソッド呼び出しやコンストラクタ呼び出しにおいて、位置引数に名前を付加するコードリファクタリングを提供する。コードの可読性向上と、引数の順序に依存しない堅牢なコードへの変換を支援する。

**業務上の目的・背景**：多くの引数を持つメソッド呼び出しでは、各引数が何を意味するのかが分かりにくくなる。名前付き引数を使用することで、コードの意図が明確になり、引数の順序変更によるバグを防止できる。特にブール引数やマジックナンバーを含む呼び出しで効果的である。

**機能の利用シーン**：
- 多引数メソッド呼び出しの可読性向上
- ブール引数の意味を明示化
- 属性コンストラクタの引数名明示
- コードレビューでの指摘対応

**主要な処理内容**：
1. 引数位置からパラメータ情報を取得
2. 名前付き引数への変換可否判定
3. 単一引数または末尾引数への名前付加
4. 構文木の更新とドキュメント変更

**関連システム・外部連携**：特になし。Roslyn言語サービス内で完結。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コードアクション（クイックアクション/電球アイコン） |

## 機能種別

コードリファクタリング / コードアクション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 対象ドキュメント | 有効なRoslynドキュメント |
| textSpan | TextSpan | Yes | 選択範囲 | 引数上の位置 |
| cancellationToken | CancellationToken | Yes | キャンセルトークン | 有効なトークン |

### 入力データソース

- 選択された引数ノード
- メソッド/コンストラクタのシンボル情報
- パラメータリスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CodeAction | array | リファクタリングアクション |

### 出力先

- コードアクションリスト（電球アイコン）

## 処理フロー

### 処理シーケンス

```
1. 引数ノードの取得
   └─ 選択位置から引数を特定
2. 位置引数判定
   └─ IsPositionalArgument確認
3. レシーバー取得と診断確認
   └─ 呼び出し式のレシーバー取得
4. シンボル解決
   └─ メソッド/コンストラクタのパラメータ情報取得
5. 変換可否判定
   └─ IsLegalToAddNamedArguments確認
6. リファクタリングアクション登録
   ├─ 単一引数に名前追加
   └─ 末尾引数含め全て名前追加（オプション）
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数ノード取得]
    B --> C{位置引数?}
    C -->|No| D[終了]
    C -->|Yes| E[レシーバー取得]
    E --> F{診断エラー?}
    F -->|Yes| D
    F -->|No| G[シンボル解決]
    G --> H{パラメータあり?}
    H -->|No| D
    H -->|Yes| I[引数リスト取得]
    I --> J{変換可能?}
    J -->|No| D
    J -->|Yes| K[リファクタリング登録]
    K --> L{複数引数?}
    L -->|Yes| M[末尾含む選択肢追加]
    L -->|No| N[単一アクション]
    M --> D
    N --> D
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 位置引数のみ | 名前付き引数は変換対象外 | IsPositionalArgument判定 |
| BR-002 | 引数リスト内 | TArgumentListSyntaxの子要素のみ対象 | 構文チェック |
| BR-003 | パラメータ範囲内 | 引数インデックスがパラメータ数以内 | インデックス確認 |
| BR-004 | 非末尾名前付き | C# 7.2以降で末尾以外にも名前付き可能 | 言語バージョン |
| BR-005 | 暗黙インデクサ除外 | 暗黙のIndex/Range インデクサは対象外 | IsImplicitIndexOrRangeIndexer |

### 計算ロジック

引数インデックス計算:
```csharp
var arguments = GetArguments(argumentList);
var argumentIndex = arguments.IndexOf(argument);
if (argumentIndex >= parameters.Length)
    return; // 対象外
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | リファクタリング不可 | 位置引数でない | アクション非表示 |
| - | リファクタリング不可 | レシーバーに診断エラー | アクション非表示 |
| - | リファクタリング不可 | シンボル解決失敗 | アクション非表示 |

### リトライ仕様

リトライ処理なし。条件を満たさない場合はリファクタリングを提供しない。

## トランザクション仕様

Roslynのイミュータブルモデルにより、変更は新しいドキュメントとして返却。

## パフォーマンス要件

- コードアクション計算は軽量に実行
- シンボル解決はキャッシュを活用

## セキュリティ考慮事項

- ソースコード変更のみ
- 機密情報の処理なし

## 備考

- C#とVBで個別の実装（言語固有のAnalyzer継承）
- 属性引数にも対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 抽象基底クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractUseNamedArgumentsCodeRefactoringProvider.cs | `src/Features/Core/Portable/UseNamedArguments/AbstractUseNamedArgumentsCodeRefactoringProvider.cs` | リファクタリングプロバイダーの実装 |

**主要処理フロー**:
- **15-20行目**: IAnalyzerインターフェース定義
- **22-184行目**: Analyzer<T>抽象クラス
  - **27-133行目**: `ComputeRefactoringsAsync`メイン処理
    - **33-37行目**: 引数ノード取得
    - **39-42行目**: 位置引数判定
    - **44-53行目**: レシーバー取得と診断チェック
    - **55-67行目**: シンボル解決とパラメータ取得
    - **69-86行目**: 変換可否判定
    - **92-132行目**: リファクタリングアクション登録

**読解のコツ**:
- `TBaseArgumentSyntax`, `TSimpleArgumentSyntax`, `TArgumentListSyntax`は言語固有の型
- `IsPositionalArgument`で名前なし引数を判定
- `SupportsNonTrailingNamedArguments`でC# 7.2対応を確認

#### Step 2: ヘルパーメソッドを理解する

**GetOrSynthesizeNamedArguments読解ポイント**:
- **149-172行目**: 名前付き引数への変換ロジック
- **163-171行目**: `ShouldAddName`で変換対象を判定

**AddNamedArgumentsAsync読解ポイント**:
- **135-147行目**: ドキュメント更新処理

#### Step 3: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| - | (同ファイル内) | - | ComputeRefactoringsAsyncオーバーライド |

**読解ポイント**:
- **197-213行目**: `ComputeRefactoringsAsync`
- **200-203行目**: MiscellaneousFilesワークスペースを除外
- **207-211行目**: 引数アナライザーと属性引数アナライザーを実行

### プログラム呼び出し階層図

```
AbstractUseNamedArgumentsCodeRefactoringProvider.ComputeRefactoringsAsync
    │
    ├─ _argumentAnalyzer.ComputeRefactoringsAsync
    │      ├─ GetRelevantNodesAsync
    │      ├─ IsPositionalArgument
    │      ├─ GetReceiver
    │      ├─ SemanticModel.GetSymbolInfo
    │      ├─ GetParameters
    │      ├─ IsLegalToAddNamedArguments
    │      └─ RegisterRefactoring
    │             └─ AddNamedArgumentsAsync
    │                    └─ GetOrSynthesizeNamedArguments
    │                           └─ WithName
    │
    └─ _attributeArgumentAnalyzer.ComputeRefactoringsAsync
           └─ (同上)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

TextSpan ─────────────▶ GetRelevantNodesAsync ─────▶ TSimpleArgumentSyntax
    │                         │
    │                         ▼
    │                    IsPositionalArgument
    │                         │
    │                         ▼
    │                    GetSymbolInfo
    │                         │
    │                         ▼
    │                    GetOrSynthesizeNamedArguments
    │                         │
    └─────────────────────────┘
                              ▼
                         CodeAction[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractUseNamedArgumentsCodeRefactoringProvider.cs | `src/Features/Core/Portable/UseNamedArguments/AbstractUseNamedArgumentsCodeRefactoringProvider.cs` | ソース | 抽象基底クラス |
| CSharpUseNamedArgumentsCodeRefactoringProvider.cs | `src/Features/CSharp/Portable/UseNamedArguments/` | ソース | C#実装 |
| VisualBasicUseNamedArgumentsCodeRefactoringProvider.cs | `src/Features/VisualBasic/Portable/UseNamedArguments/` | ソース | VB実装 |
