# 機能設計書 129-タプル要素命名

## 概要

本ドキュメントは、Roslynプロジェクトにおけるタプル要素命名機能の設計仕様を記載する。タプルリテラルの要素に明示的な名前を付与するリファクタリング機能を提供する。

### 本機能の処理概要

タプル要素命名機能は、タプル式の要素に対して、型定義から推論される名前を明示的に付加するコードリファクタリングを提供する。タプルの可読性向上とコードの自己文書化を支援する。

**業務上の目的・背景**：C# 7.0で導入されたタプルは、複数の値を軽量に返す手段として広く使用される。しかし、要素名を省略した場合、Item1, Item2などの汎用的な名前が使用され、コードの意図が分かりにくくなる。本機能は、変換先の型から推論される適切な名前を自動的に付加し、コードの可読性を向上させる。

**機能の利用シーン**：
- タプルリテラルへの名前付加
- 型推論された名前の明示化
- コードレビューでの可読性向上指摘対応

**主要な処理内容**：
1. タプル式の要素位置特定
2. 変換先タプル型の取得
3. 対応するタプル要素名の抽出
4. 名前付き引数形式への変換
5. 構文木の更新

**関連システム・外部連携**：特になし。Roslyn言語サービス内で完結。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コードアクション（クイックアクション/電球アイコン） |

## 機能種別

コードリファクタリング / コードアクション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 対象ドキュメント | 有効なRoslynドキュメント |
| span | TextSpan | Yes | 選択範囲 | タプル要素上の位置 |
| cancellationToken | CancellationToken | Yes | キャンセルトークン | 有効なトークン |

### 入力データソース

- 選択された引数ノード（タプル要素）
- タプル式の変換先型情報
- セマンティックモデル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CodeAction | array | リファクタリングアクション |

### 出力先

- コードアクションリスト（電球アイコン）

## 処理フロー

### 処理シーケンス

```
1. 引数ノードの取得
   └─ 選択位置からタプル要素を特定
2. タプル式の特定
   └─ 親ノードがTTupleExpressionSyntaxか確認
3. 単純引数判定
   └─ IsSimpleArgument確認（名前なし）
4. 変換先型の取得
   └─ SemanticModel.GetTypeInfoでConvertedType取得
5. 要素名の抽出
   └─ TupleElementsから対応する名前を取得
6. 名前付き引数生成
   └─ WithNameで名前付き引数に変換
7. リファクタリングアクション登録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[引数ノード取得]
    B --> C{タプル要素?}
    C -->|No| D[終了]
    C -->|Yes| E{単純引数?}
    E -->|No| D
    E -->|Yes| F[変換先型取得]
    F --> G{INamedTypeSymbol?}
    G -->|No| D
    G -->|Yes| H[TupleElements取得]
    H --> I{要素名あり?}
    I -->|No| D
    I -->|Yes| J[名前付き引数生成]
    J --> K[リファクタリング登録]
    K --> D
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | タプル要素のみ | タプル式内の要素のみ対象 | 親がTTupleExpressionSyntax |
| BR-002 | 単純引数のみ | 既に名前付きの引数は対象外 | IsSimpleArgument判定 |
| BR-003 | 名前推論可能 | CorrespondingTupleFieldと異なる名前が必要 | element.Equals(element.CorrespondingTupleField) |
| BR-004 | 有効な要素インデックス | 引数インデックスがTupleElements範囲内 | argumentIndex < elements.Length |

### 計算ロジック

要素名取得:
```csharp
var element = elements[argumentIndex];
if (element.Equals(element.CorrespondingTupleField))
    return; // 名前なし
var elementName = element.Name;
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | リファクタリング不可 | タプル要素でない | アクション非表示 |
| - | リファクタリング不可 | 既に名前付き | アクション非表示 |
| - | リファクタリング不可 | 型情報取得失敗 | アクション非表示 |

### リトライ仕様

リトライ処理なし。

## トランザクション仕様

Roslynのイミュータブルモデルにより、変更は新しいドキュメントとして返却。

## パフォーマンス要件

- コードアクション計算は軽量に実行
- セマンティックモデルのキャッシュ活用

## セキュリティ考慮事項

- ソースコード変更のみ
- 機密情報の処理なし

## 備考

- C#とVBで個別の実装
- タプル型は.NET 4.7以降またはSystem.ValueTupleパッケージ必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 抽象基底クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractNameTupleElementCodeRefactoringProvider.cs | `src/Features/Core/Portable/NameTupleElement/AbstractNameTupleElementCodeRefactoringProvider.cs` | リファクタリングプロバイダーの実装 |

**主要処理フロー**:
- **16-19行目**: ジェネリック型パラメータ定義（TArgumentSyntax, TTupleExpressionSyntax）
- **20行目**: `WithName`抽象メソッド定義
- **22-38行目**: `ComputeRefactoringsAsync`メイン処理
  - **24行目**: TryGetArgumentInfoAsyncで情報取得
  - **32-37行目**: リファクタリングアクション登録

**TryGetArgumentInfoAsync読解ポイント**:
- **40-80行目**: 引数情報取得ロジック
  - **43-46行目**: MiscellaneousFilesワークスペース除外
  - **48-54行目**: 引数ノード取得と検証
  - **58-62行目**: タプル型情報取得
  - **64-68行目**: 要素情報取得とインデックス確認
  - **72-76行目**: 名前推論チェック

**AddNamedElementAsync読解ポイント**:
- **82-89行目**: 名前付き引数生成とドキュメント更新

### プログラム呼び出し階層図

```
AbstractNameTupleElementCodeRefactoringProvider.ComputeRefactoringsAsync
    │
    ├─ TryGetArgumentInfoAsync
    │      ├─ GetRelevantNodesAsync
    │      ├─ ISyntaxFactsService.IsSimpleArgument
    │      ├─ SemanticModel.GetTypeInfo
    │      ├─ ISyntaxFactsService.GetPartsOfTupleExpression
    │      └─ INamedTypeSymbol.TupleElements
    │
    └─ RegisterRefactoring
           └─ AddNamedElementAsync
                  └─ WithName
```

### データフロー図

```
[入力]                    [処理]                         [出力]

TextSpan ─────────────▶ TryGetArgumentInfoAsync ───▶ (root, argument, elementName)
    │                         │
    │                         ├─ GetRelevantNodesAsync
    │                         ├─ GetTypeInfo
    │                         └─ TupleElements取得
    │                         │
    └─────────────────────────┘
                              ▼
                    AddNamedElementAsync
                              │
                              ▼
                         CodeAction
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractNameTupleElementCodeRefactoringProvider.cs | `src/Features/Core/Portable/NameTupleElement/AbstractNameTupleElementCodeRefactoringProvider.cs` | ソース | 抽象基底クラス |
| CSharpNameTupleElementCodeRefactoringProvider.cs | `src/Features/CSharp/Portable/NameTupleElement/` | ソース | C#実装 |
| VisualBasicNameTupleElementCodeRefactoringProvider.cs | `src/Features/VisualBasic/Portable/NameTupleElement/` | ソース | VB実装 |
