# 機能設計書 132-フレームワーク型優先

## 概要

本ドキュメントは、Roslynの「フレームワーク型優先（PreferFrameworkType）」機能の設計を記述する。この機能は、C#の組み込み型キーワード（int, string等）をBCL（Base Class Library）のフレームワーク型（Int32, String等）に変換するコードスタイル機能である。

### 本機能の処理概要

この機能は、ユーザーのコードスタイル設定に基づき、C#の組み込み型キーワード（`int`、`string`、`bool`等）を対応するフレームワーク型（`System.Int32`、`System.String`、`System.Boolean`等）に変換する。逆方向の変換（フレームワーク型からキーワードへ）は「型名の簡略化」機能が担当する。

**業務上の目的・背景**：
コーディング規約によっては、組み込み型キーワードよりもフレームワーク型名を優先するスタイルが採用される場合がある。例えば、静的メンバーアクセス（`Int32.MaxValue`）やCref参照での一貫性を保つため、フレームワーク型を統一使用する規約が存在する。本機能により、このようなスタイルの自動適用と診断が可能となる。

**機能の利用シーン**：
- 宣言時のコードスタイル設定（PreferIntrinsicPredefinedTypeKeywordInDeclaration = false）
- メンバーアクセス時のコードスタイル設定（PreferIntrinsicPredefinedTypeKeywordInMemberAccess = false）
- コードスタイル設定画面での設定変更後
- .editorconfigファイルでの設定反映時

**主要な処理内容**：
1. 組み込み型キーワード（PredefinedTypeSyntax）の検出
2. セマンティック解析による型シンボルの取得
3. ユーザー設定（コンテキスト別）の参照
4. 診断メッセージ生成と表示
5. フレームワーク型表現への変換（SyntaxGenerator使用）

**関連システム・外部連携**：
- EditorConfig：コードスタイル設定の読み込み
- SyntaxGenerator：フレームワーク型構文の生成

**権限による制御**：
特に権限による制御はない。エディタ機能として全ユーザーに提供される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタウィンドウ | 主画面 | 診断結果の表示とクイックアクション |
| - | オプション画面 | 設定画面 | コードスタイル設定の変更 |

## 機能種別

コード診断・コードスタイル変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| typeNode | TTypeSyntax | Yes | 解析対象の型構文ノード | PredefinedTypeSyntaxまたはIdentifierNameSyntax |
| semanticModel | SemanticModel | Yes | セマンティックモデル | null不可 |
| options | AnalyzerOptions | Yes | アナライザーオプション | null不可 |

### 入力データソース

- ソースコードドキュメント（SyntaxTree）
- EditorConfig設定ファイル
- ユーザーオプション設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Diagnostic | Diagnostic | フレームワーク型使用を推奨する診断 |
| replacementNode | SyntaxNode | 変換後のフレームワーク型構文 |

### 出力先

- IDE診断結果（エディタ波線）
- 修正適用時は編集後のドキュメント

## 処理フロー

### 処理シーケンス

```
1. RegisterSyntaxNodeAction登録
   └─ SyntaxKindsOfInterestで対象ノード種別を指定
2. AnalyzeNode実行
   └─ 各対象ノードに対して解析
3. 設定確認
   └─ IsFrameworkTypePreferredでユーザー設定を確認
4. 型シンボル取得
   └─ GetSymbolInfoでSpecialType確認
5. コンテキスト判定
   └─ IsInMemberAccessOrCrefReferenceContextで使用箇所判定
6. 診断報告
   └─ ReportDiagnosticで結果出力
```

### フローチャート

```mermaid
flowchart TD
    A[型ノード取得] --> B{PredefinedType or IdentifierName?}
    B -->|その他| Z[スキップ]
    B -->|Yes| C{IsPredefinedTypeReplaceableWithFrameworkType?}
    C -->|No| Z
    C -->|Yes| D[型シンボル取得]
    D --> E{SpecialType != None?}
    E -->|No| Z
    E -->|Yes| F{IsInMemberAccessContext?}
    F -->|Yes| G[MemberAccess設定確認]
    F -->|No| H[Declaration設定確認]
    G --> I{IsFrameworkTypePreferred?}
    H --> I
    I -->|No| Z
    I -->|Yes| J[Diagnostic報告]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-132-01 | 宣言時設定 | 変数宣言等でフレームワーク型を優先 | PreferIntrinsicPredefinedTypeKeywordInDeclaration = false |
| BR-132-02 | メンバーアクセス時設定 | 静的メンバーアクセスでフレームワーク型を優先 | PreferIntrinsicPredefinedTypeKeywordInMemberAccess = false |
| BR-132-03 | IntPtr/UIntPtr特殊処理 | nint/nuintはGlobal::System.IntPtrに変換 | SpecialType.System_IntPtr or System_UIntPtr |

### 計算ロジック

IsFrameworkTypePreferred判定:
```csharp
!optionValue.Value && optionValue.Notification.Severity != ReportDiagnostic.Suppress
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし（メモリ上のコード解析のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | シンボル解決失敗 | GetSymbolInfoがnullを返す | 診断をスキップ |
| - | 設定未定義 | オプション値が取得できない | デフォルト動作（診断しない） |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

トランザクション管理なし。

## パフォーマンス要件

- SemanticSpanAnalysisカテゴリで実装
- RegisterSyntaxNodeActionで特定ノードのみ解析

## セキュリティ考慮事項

特になし。

## 備考

- 診断ID: IDE0049（PreferBuiltInOrFrameworkType）
- 「型名の簡略化」機能と逆方向の変換を担当

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PreferFrameworkTypeConstants.cs | `src/Features/Core/Portable/PreferFrameworkType/` | プロパティキー定義 |
| 1-2 | CodeStyleOptions2 | `src/Workspaces/SharedUtilitiesAndExtensions/Compiler/Core/CodeStyle/` | PreferIntrinsicPredefinedTypeKeyword設定 |

**読解のコツ**: `PreferFrameworkTypeConstants.PreferFrameworkType`がDiagnosticのPropertiesキーとして使用される。

#### Step 2: 診断アナライザーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PreferFrameworkTypeDiagnosticAnalyzerBase.cs | `src/Features/Core/Portable/PreferFrameworkType/` | 診断ロジックの基底クラス |

**主要処理フロー**:
1. **45-46行目**: RegisterSyntaxNodeActionで対象ノード登録
2. **48-108行目**: AnalyzeNodeで診断処理
3. **56-66行目**: 事前フィルタリング（設定未使用時の早期リターン）
4. **86-90行目**: 型シンボル取得とSpecialType確認
5. **94-104行目**: コンテキスト判定と診断報告

#### Step 3: コード修正プロバイダーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PreferFrameworkTypeCodeFixProvider.cs | `src/Features/Core/Portable/PreferFrameworkType/` | コード修正実装 |

**主要処理フロー**:
- **26-35行目**: RegisterCodeFixesAsyncでアクション登録
- **37-57行目**: FixAllAsyncで変換処理
- **51-53行目**: IntPtr/UIntPtrの特殊処理

### プログラム呼び出し階層図

```
PreferFrameworkTypeDiagnosticAnalyzerBase
    │
    ├─ InitializeWorker
    │      └─ RegisterSyntaxNodeAction(AnalyzeNode)
    │
    └─ AnalyzeNode
           ├─ IsPredefinedTypeReplaceableWithFrameworkType
           ├─ IsIdentifierNameReplaceableWithFrameworkType
           ├─ IsInMemberAccessOrCrefReferenceContext
           └─ ReportDiagnostic

PreferFrameworkTypeCodeFixProvider
    │
    ├─ RegisterCodeFixesAsync
    │
    └─ FixAllAsync
           └─ generator.TypeExpression / generator.QualifiedName
```

### データフロー図

```
[入力]                    [処理]                         [出力]

PredefinedTypeSyntax ───▶ IsPredefinedTypeReplaceable ───▶ Diagnostic
        │                        │
        │                        │
   SemanticModel          CodeStyleOption2
        │                        │
        └────────────────────────┘
                  ▼
         FixAllAsync ───▶ TypeExpression/QualifiedName
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PreferFrameworkTypeDiagnosticAnalyzerBase.cs | `src/Features/Core/Portable/PreferFrameworkType/` | ソース | 診断アナライザー基底クラス |
| PreferFrameworkTypeCodeFixProvider.cs | `src/Features/Core/Portable/PreferFrameworkType/` | ソース | コード修正プロバイダー |
| PreferFrameworkTypeConstants.cs | `src/Features/Core/Portable/PreferFrameworkType/` | ソース | 定数定義 |
| CodeStyleOptions2.cs | `src/Workspaces/SharedUtilitiesAndExtensions/Compiler/Core/CodeStyle/` | ソース | コードスタイルオプション |
