# 機能設計書 34-一時変数のインライン化

## 概要

本ドキュメントは、Roslynの「一時変数のインライン化（Inline Temporary）」機能の設計仕様を記述する。この機能は、ローカル変数を参照している箇所をその初期化式で直接置換するリファクタリング機能である。

### 本機能の処理概要

**業務上の目的・背景**：一時的に導入されたローカル変数が不要になった場合や、変数が単に式のエイリアスとして機能している場合、変数を削除して式を直接使用することでコードの簡潔性が向上する。この機能は、そのような場面で変数を安全にインライン化する。

**機能の利用シーン**：開発者が「一度しか使われない変数」や「単純な式の代入のみの変数」をリファクタリングで削除したい場合に使用する。例えば、`var x = GetValue(); return x;`を`return GetValue();`に変換する場面で活用される。

**主要な処理内容**：
1. 選択されたローカル変数宣言の検証（インライン化可能かどうかの判定）
2. 変数の参照箇所の検索
3. 各参照箇所を初期化式で置換
4. 変数宣言の削除

**関連システム・外部連携**：シンボル検索サービス（SymbolFinder）を使用して参照を検索する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| N/A | N/A | N/A | コードエディタ上のコンテキストメニューから直接実行 |

## 機能種別

コードリファクタリング / コード変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 操作対象のドキュメント | null不可 |
| variableDeclarator | TVariableDeclaratorSyntax | Yes | インライン化する変数宣言子 | 有効な変数宣言子 |

### 入力データソース

- エディタで選択された変数宣言

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| modifiedDocument | Document | 変更が適用されたドキュメント |

### 出力先

- 対象ドキュメント（変数参照を初期化式で置換、変数宣言削除）

## 処理フロー

### 処理シーケンス

```
1. 変数宣言子の取得
   └─ TryGetRelevantNodeAsync<TVariableDeclaratorSyntax>で取得

2. ローカル変数の検証
   └─ セマンティックモデルからシンボル取得
   └─ 初期化式があることを確認
   └─ 読み取り専用でないことを確認

3. 参照箇所の検索
   └─ GetReferenceLocationsAsync()
   └─ SymbolFinder.FindReferencesAsyncで参照検索
   └─ Cascade: falseで直接参照のみ

4. 参照の妥当性チェック
   └─ 隠し領域内の参照がないことを確認

5. 各参照を初期化式で置換
   └─ 参照位置ごとにドキュメントエディタで置換
   └─ 必要に応じて括弧追加

6. 変数宣言の削除
   └─ 宣言全体または宣言子のみを削除
   └─ 複数宣言子の場合は該当宣言子のみ削除

7. コードクリーンアップ
   └─ 型の簡略化
   └─ 不要なキャストの削除
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[変数宣言子取得]
    B --> C{ローカル変数?}
    C -->|No| Z[終了]
    C -->|Yes| D{初期化式あり?}
    D -->|No| Z
    D -->|Yes| E[参照箇所検索]
    E --> F{隠し領域内参照?}
    F -->|Yes| Z
    F -->|No| G[各参照を式で置換]
    G --> H[変数宣言削除]
    H --> I[コードクリーンアップ]
    I --> Z[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-34-01 | 初期化式必須 | 初期化式を持つ変数のみ対象 | 常に適用 |
| BR-34-02 | ローカル変数限定 | ローカル変数のみ対象（フィールド不可） | 常に適用 |
| BR-34-03 | 隠し領域除外 | 隠し領域内の参照がある場合は対象外 | 常に適用 |
| BR-34-04 | 直接参照のみ | Cascadeなしで直接参照のみ検索 | 常に適用 |

### 計算ロジック

参照位置の取得ロジック：
```
参照リスト = SymbolFinder.FindReferences(ローカルシンボル, ソリューション, Cascade: false)
有効な参照 = 参照リスト.Where(位置が隠し領域内でない)
                      .Select(位置 -> 対応する識別子ノード)
```

## データベース操作仕様

該当なし（メモリ上のシンボルとソースコードの操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| N/A | 検証エラー | 初期化式がない変数 | リファクタリングを提供しない |
| N/A | 検証エラー | フィールドなど非ローカル変数 | リファクタリングを提供しない |
| N/A | 検証エラー | 隠し領域内の参照がある | リファクタリングを提供しない |

### リトライ仕様

該当なし（インタラクティブな操作のため）

## トランザクション仕様

ドキュメント単位での変更を適用。すべての変更が成功するか、すべて適用されないかのいずれかとなる。

## パフォーマンス要件

- シンボル参照の検索は非同期で実行
- UIのレスポンシブ性を維持するため、キャンセル可能な操作として実装

## セキュリティ考慮事項

- ソースコードへの書き込み権限が必要

## 備考

- 複数の宣言子を持つ宣言文（`int a = 1, b = 2;`）の場合、該当する宣言子のみを削除
- var使用時は型推論の結果を考慮
- 式の優先順位により必要に応じて括弧を追加

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractInlineTemporaryCodeRefactoringProvider.cs | `src/Features/Core/Portable/InlineTemporary/AbstractInlineTemporaryCodeRefactoringProvider.cs` | CodeRefactoringProviderの実装。ComputeRefactoringsAsyncがエントリーポイント |

**主要処理フロー**:
1. **16-21行目**: クラス定義 - ジェネリック型パラメータによる言語抽象化
2. **22行目**: CodeRefactoringKind.Inlineを指定
3. **24-53行目**: GetReferenceLocationsAsync - 参照位置の取得

**読解のコツ**: このクラスは共通処理のみを提供し、言語固有の処理はC#/VB実装に委譲される。

#### Step 2: 参照検索ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractInlineTemporaryCodeRefactoringProvider.cs | `src/Features/Core/Portable/InlineTemporary/AbstractInlineTemporaryCodeRefactoringProvider.cs` | GetReferenceLocationsAsyncメソッド |

**主要処理フロー**:
- **29-30行目**: セマンティックモデルからローカルシンボル取得
- **37行目**: FindReferencesSearchOptions.Default with { Cascade = false }
- **39-40行目**: SymbolFinder.FindReferencesAsync呼び出し
- **41行目**: 該当シンボルの参照のみをフィルタリング
- **44-48行目**: 隠し領域を除外し、識別子ノードを取得

### プログラム呼び出し階層図

```
CSharpInlineTemporaryCodeRefactoringProvider.ComputeRefactoringsAsync
    │
    ├─ TryGetRelevantNodeAsync<LocalDeclarationStatementSyntax>
    │
    ├─ SemanticModel.GetDeclaredSymbol (ローカルシンボル取得)
    │
    └─ AbstractInlineTemporaryCodeRefactoringProvider.GetReferenceLocationsAsync
           │
           ├─ SymbolFinder.FindReferencesAsync (Cascade: false)
           │
           ├─ 隠し領域フィルタリング
           │
           └─ 識別子ノード取得
                  │
                  └─ CodeAction.Create
                         │
                         └─ InlineTemporaryAsync (言語固有実装)
                                │
                                ├─ 参照箇所を初期化式で置換
                                ├─ 変数宣言削除
                                └─ コードクリーンアップ
```

### データフロー図

```
[入力]                          [処理]                              [出力]

変数宣言子 ───────────▶ AbstractInlineTemporaryCodeRefactoringProvider
                                 │
                                 ▼
                        GetReferenceLocationsAsync
                                 │
                        ┌────────┴────────┐
                        ▼                 ▼
              SymbolFinder          隠し領域フィルタ
                        │                 │
                        └────────┬────────┘
                                 ▼
                        参照ノードリスト ───▶ 言語固有InlineAsync
                                 │
                                 ▼
                        各参照を式で置換 + 宣言削除
                                 │
                                 ▼
                        更新されたドキュメント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractInlineTemporaryCodeRefactoringProvider.cs | `src/Features/Core/Portable/InlineTemporary/` | ソース | リファクタリングプロバイダーの基底クラス |
| CSharpInlineTemporaryCodeRefactoringProvider.cs | `src/Features/CSharp/Portable/InlineTemporary/` | ソース | C#固有の実装 |
| VisualBasicInlineTemporaryCodeRefactoringProvider.vb | `src/Features/VisualBasic/Portable/InlineTemporary/` | ソース | VB固有の実装 |
