# 機能設計書 40-インターフェース実装

## 概要

本ドキュメントは、Roslynの「インターフェース実装（Implement Interface）」機能の設計仕様を記述する。この機能は、クラスまたは構造体が実装するインターフェースのメンバーを自動的に生成するコード生成機能である。

### 本機能の処理概要

**業務上の目的・背景**：インターフェースを実装する際、すべてのメンバーを手動で実装するのは時間がかかり、タイプミスの可能性もある。この機能は、インターフェースのすべてのメンバー（またはまだ実装されていないメンバー）を自動生成することで、開発効率を向上させる。

**機能の利用シーン**：
1. 新しいクラスでインターフェースを実装する際
2. インターフェースに新しいメンバーが追加された際に既存の実装クラスを更新する場合
3. IDisposableなどの特殊なインターフェースをDisposeパターンで実装する場合

**主要な処理内容**：
1. 実装されていないインターフェースメンバーの検出
2. 暗黙的/明示的実装の選択
3. Disposeパターンの生成（IDisposable実装時）
4. 既存メンバーへの委譲実装
5. 抽象メンバーとしての実装

**関連システム・外部連携**：コード生成サービス（SyntaxGenerator）を使用してメンバーを生成する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| N/A | N/A | N/A | コードエディタ上のコンテキストメニューから直接実行 |

## 機能種別

コード生成 / クイック修正 / コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 操作対象のドキュメント | null不可 |
| interfaceType | SyntaxNode | Yes | 実装するインターフェースの型ノード | 有効なインターフェース型 |
| options | ImplementTypeOptions | No | 実装オプション（プロパティ生成動作等） | |
| configuration | ImplementInterfaceConfiguration | No | 実装設定（明示的/暗黙的、抽象、委譲先等） | |

### 入力データソース

- エディタ上のインターフェース型参照
- 実装するクラス/構造体の型宣言

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| modifiedDocument | Document | 変更が適用されたドキュメント |

### 出力先

- 対象ドキュメント（インターフェースメンバーの実装を挿入）

## 処理フロー

### 処理シーケンス

```
1. インターフェース型の解析
   └─ TryInitializeState()でコンテキスト情報を収集
   └─ 実装するクラス/構造体と対象インターフェースを特定

2. 未実装メンバーの検出
   └─ MembersWithoutExplicitOrImplicitImplementation
   └─ MembersWithoutExplicitImplementation
   └─ 既存の暗黙的/明示的実装をチェック

3. 実装オプションの決定
   a. 暗黙的実装（通常）
   b. 明示的実装（インターフェース.メンバー形式）
   c. 抽象メンバー実装（抽象クラスの場合）
   d. 既存メンバーへの委譲
   e. Disposeパターン（IDisposable実装時）

4. CodeActionの生成
   └─ GetCodeActionsAsync()
   └─ 各実装オプションに対するCodeAction生成

5. メンバーの生成
   └─ ImplementInterfaceGenerator.GenerateMembers()
   └─ メソッド、プロパティ、イベント、インデクサの生成
   └─ アクセシビリティとunsafe修飾子の処理

6. Disposeパターンの生成（該当する場合）
   └─ Dispose(bool disposing)メソッド
   └─ ファイナライザ（必要に応じて）
   └─ IDisposable.Dispose()の実装

7. ドキュメントの更新
   └─ SyntaxEditorで変更を適用
   └─ フォーマット適用
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[インターフェース型解析]
    B --> C{有効な型?}
    C -->|No| Z[終了]
    C -->|Yes| D[未実装メンバー検出]
    D --> E{未実装あり?}
    E -->|No| Z
    E -->|Yes| F[実装オプション決定]
    F --> G{IDisposable?}
    G -->|Yes| H[Disposeパターン追加]
    G -->|No| I[通常の実装オプション]
    H --> J[CodeAction生成]
    I --> J
    J --> K{ユーザー選択}
    K --> L{実装方式}
    L -->|暗黙的| M[暗黙的実装生成]
    L -->|明示的| N[明示的実装生成]
    L -->|抽象| O[抽象メンバー生成]
    L -->|委譲| P[委譲実装生成]
    M --> Q[ドキュメント更新]
    N --> Q
    O --> Q
    P --> Q
    Q --> Z[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-01 | クラス/構造体限定 | インターフェース実装はクラスまたは構造体のみ | 常に適用 |
| BR-40-02 | 暗黙的実装優先 | デフォルトは暗黙的実装（C#のみ） | 暗黙的実装可能な場合 |
| BR-40-03 | Disposeパターン | IDisposable実装時はDisposeパターンを提案 | IDisposable実装時 |
| BR-40-04 | 抽象メンバー | 抽象クラスの場合は抽象メンバー実装を提案 | 抽象クラスの場合 |
| BR-40-05 | メンバー委譲 | 同じインターフェースを実装するメンバーへの委譲を提案 | 委譲可能なメンバーがある場合 |
| BR-40-06 | 非公開メンバー | 非公開インターフェースメンバーは暗黙的実装不可（C# 8.0以前） | 言語バージョンによる |

### 計算ロジック

実装オプション生成ロジック：
```
if (未実装の暗黙的実装可能メンバーあり) {
    if (totalMemberCount != inaccessibleMemberCount) {
        yield "暗黙的実装"
    }
    if (ShouldImplementDisposePattern) {
        yield "Disposeパターン付き暗黙的実装"
    }
    for each (委譲可能メンバー) {
        yield "メンバーへの委譲実装"
    }
    if (抽象クラス) {
        yield "抽象メンバー実装"
    }
}
if (未実装の明示的実装メンバーあり) {
    yield "明示的実装"
    if (ShouldImplementDisposePattern) {
        yield "Disposeパターン付き明示的実装"
    }
}
if (一部暗黙的実装済み) {
    yield "残りを明示的に実装"
}
```

## データベース操作仕様

該当なし（メモリ上のシンボルとソースコードの操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| N/A | 検証エラー | 非インターフェース型を指定 | コード生成を提供しない |
| N/A | 検証エラー | すべてのメンバーが既に実装済み | コード生成を提供しない |

### リトライ仕様

該当なし（インタラクティブな操作のため）

## トランザクション仕様

ドキュメント単位での変更を適用。すべての変更が成功するか、すべて適用されないかのいずれかとなる。

## パフォーマンス要件

- UIのレスポンシブ性を維持するため、キャンセル可能な操作として実装

## セキュリティ考慮事項

- ソースコードへの書き込み権限が必要

## 備考

- VBでは暗黙的実装はサポートされない（常に明示的実装）
- 非公開インターフェースメンバーはC# 8.0以降でサポート
- プロパティ生成時はImplementTypeOptionsに基づいてauto-implementedまたはthrow実装を選択

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースとデータ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/IImplementInterfaceService.cs` | サービスインターフェースの定義 |
| 1-2 | ImplementInterfaceConfiguration | 同ファイル内 | 実装設定のデータ構造（ImplementDisposePattern、Explicitly、Abstractly、OnlyRemaining、ThroughMember） |

**読解のコツ**: ImplementInterfaceConfigurationは5つの主要オプションを持つ。これらの組み合わせで様々な実装パターンが表現される。

#### Step 2: リファクタリングプロバイダーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ImplementInterfaceCodeRefactoringProvider.cs | `src/Features/Core/Portable/ImplementInterface/ImplementInterfaceCodeRefactoringProvider.cs` | CodeRefactoringProviderの実装 |

**主要処理フロー**:
1. **22-47行目**: ComputeRefactoringsAsync - エントリーポイント
2. **31行目**: IsBetweenTypeMembersで位置チェック
3. **34行目**: IImplementInterfaceService取得
4. **37-43行目**: GetInterfaceTypesで実装インターフェース取得、GetCodeActionsAsyncでアクション生成

#### Step 3: メインサービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AbstractImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/AbstractImplementInterfaceService.cs` | サービスの基底クラス |

**主要処理フロー**:
- **26-27行目**: クラス定義 - TTypeDeclarationSyntax ジェネリック型パラメータ
- **35-40行目**: TryInitializeState - 状態初期化（抽象メソッド）
- **58-76行目**: ImplementInterfaceAsync - インターフェース実装のメイン処理
- **146-171行目**: GetCodeActionsAsync - CodeAction生成

#### Step 4: 実装オプション生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | AbstractImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/AbstractImplementInterfaceService.cs` | GetImplementOptionsAsync |

**主要処理フロー**:
- **173-221行目**: GetImplementOptionsAsync - 実装オプションの列挙
- **179-209行目**: 暗黙的実装可能メンバーに対するオプション生成
- **200-201行目**: Disposeパターンオプション
- **203-205行目**: 委譲実装オプション
- **207-208行目**: 抽象メンバーオプション
- **211-217行目**: 明示的実装オプション

#### Step 5: タイトル生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | AbstractImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/AbstractImplementInterfaceService.cs` | GetTitle |

**主要処理フロー**:
- **223-249行目**: GetTitle - CodeActionのタイトル生成
- 各オプションに応じたリソース文字列の選択

### プログラム呼び出し階層図

```
ImplementInterfaceCodeRefactoringProvider.ComputeRefactoringsAsync
    │
    ├─ IRefactoringHelpersService.IsBetweenTypeMembers
    │
    ├─ IImplementInterfaceService.GetInterfaceTypes
    │
    └─ IImplementInterfaceService.GetCodeActionsAsync
           │
           ├─ AbstractImplementInterfaceService.AnalyzeAsync
           │      └─ State.Generate
           │             └─ TryInitializeState (abstract)
           │
           └─ GetImplementOptionsAsync
                  │
                  ├─ ShouldImplementDisposePattern
                  ├─ GetDelegatableMembers
                  └─ AnyImplementedImplicitly
                         │
                         └─ CodeAction.Create
                                │
                                └─ ImplementInterfaceAsync
                                       │
                                       └─ ImplementInterfaceGenerator
                                              ├─ GenerateMembers
                                              └─ ImplementInterfaceAsync
```

### データフロー図

```
[入力]                          [処理]                              [出力]

インターフェース型 ───────▶ ImplementInterfaceCodeRefactoringProvider
                                 │
                                 ▼
                        IImplementInterfaceService.GetInterfaceTypes
                                 │
                                 ▼
                        AbstractImplementInterfaceService.AnalyzeAsync
                                 │
                                 ▼
                        GetImplementOptionsAsync
                                 │
                        ┌────────┼────────┬────────┐
                        ▼        ▼        ▼        ▼
                     暗黙的    明示的    抽象     委譲
                        │        │        │        │
                        └────────┼────────┴────────┘
                                 ▼
                        ImplementInterfaceGenerator
                                 │
                        ┌────────┴────────┐
                        ▼                 ▼
                  GenerateMembers   [Disposeパターン]
                        │                 │
                        └────────┬────────┘
                                 ▼
                        ドキュメント更新
                                 │
                                 ▼
                        更新されたドキュメント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/` | ソース | サービスインターフェース |
| AbstractImplementInterfaceService.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/` | ソース | サービス基底クラス |
| AbstractImplementInterfaceService.State.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/` | ソース | 状態管理 |
| ImplementInterfaceCodeRefactoringProvider.cs | `src/Features/Core/Portable/ImplementInterface/` | ソース | リファクタリングプロバイダー |
| AbstractImplementInterfaceCodeFixProvider.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/` | ソース | コード修正プロバイダー |
| CSharpImplementInterfaceService.cs | `src/Analyzers/CSharp/CodeFixes/ImplementInterface/` | ソース | C#固有の実装 |
| ImplementInterfaceGenerator.cs | `src/Analyzers/Core/CodeFixes/ImplementInterface/` | ソース | メンバー生成ロジック |
