# 機能設計書 43-比較演算子生成

## 概要

本ドキュメントは、Roslyn IDEの「比較演算子生成（Generate Comparison Operators）」機能の設計を記述したものである。この機能は、IComparable<T>インターフェースを実装した型に対して、比較演算子（<、<=、>、>=）を自動生成するコードリファクタリング機能である。

### 本機能の処理概要

**業務上の目的・背景**：IComparable<T>インターフェースを実装した型では、CompareTo()メソッドによる比較が可能だが、開発者は直感的な比較演算子（<、>など）の使用を期待することが多い。しかし、CompareTo()を使った比較演算子の実装は定型的であり、手動実装は冗長で誤りが発生しやすい。この機能は、IComparable<T>の実装から自動的に比較演算子を生成することで、開発者の生産性向上とコードの一貫性を確保する。

**機能の利用シーン**：IComparable<T>を実装した型（DateTime、Version、カスタム値オブジェクトなど）に対して、開発者がクラスまたは構造体の宣言部にカーソルを置き、コードアクションメニューから「Generate comparison operators」を選択する場面で利用される。

**主要な処理内容**：
1. カーソル位置が型宣言ヘッダー上またはメンバー間の空行にあるかを判定
2. 対象型がIComparable<T>を実装しているか確認
3. CompareTo()メソッドの実装を取得
4. 未定義の比較演算子（<、<=、>、>=）を特定
5. 各演算子の実装コードを生成（CompareTo()の結果を0と比較）
6. 生成されたコードを型に挿入

**関連システム・外部連携**：SyntaxGeneratorを使用して演算子コードを生成する。ICodeGenerationServiceを使用してドキュメントに挿入する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能には専用の関連画面はない |

## 機能種別

コード生成 / リファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| document | Document | Yes | 対象ドキュメント | null不可 |
| textSpan | TextSpan | Yes | カーソル位置 | 有効な範囲内 |
| containingType | INamedTypeSymbol | Yes | 対象の型 | IComparable<T>を実装 |
| comparableType | INamedTypeSymbol | Yes | IComparable<T>のインスタンス | エラー型でない |

### 入力データソース

- ソースコードのカーソル位置からセマンティックモデルを介して取得
- 型のインターフェース実装から収集

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| operator < | IMethodSymbol | 小なり演算子 |
| operator <= | IMethodSymbol | 小なりイコール演算子 |
| operator > | IMethodSymbol | 大なり演算子 |
| operator >= | IMethodSymbol | 大なりイコール演算子 |

### 出力先

ソースコードのクラス/構造体宣言内に挿入

## 処理フロー

### 処理シーケンス

```
1. カーソル位置の検証
   └─ 型宣言ヘッダー上またはメンバー間の空行かを判定
2. IComparable<T>実装の確認
   └─ 型のInterfacesからIComparable<T>を検索
3. CompareTo()メソッドの取得
   └─ インターフェースメンバーの実装を取得
4. 未定義演算子の特定
   └─ 4つの比較演算子それぞれの存在を確認
5. 演算子コード生成
   └─ left.CompareTo(right) < 0 形式のコードを生成
6. ドキュメント挿入
   └─ ICodeGenerationService.AddMembersAsyncで挿入
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{カーソル位置が有効?}
    B -->|No| C[終了:リファクタリング不可]
    B -->|Yes| D[IComparable<T>実装を検索]
    D --> E{IComparable<T>あり?}
    E -->|No| C
    E -->|Yes| F[CompareTo実装を取得]
    F --> G{未定義演算子あり?}
    G -->|No| C
    G -->|Yes| H{複数のIComparable<T>?}
    H -->|No| I[単一アクションを登録]
    H -->|Yes| J[ネストアクションを登録]
    I --> K[終了]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 対象インターフェース | IComparable<T>を実装した型のみ対象 | 型検証時 |
| BR-02 | エラー型除外 | TypeArgumentsにエラー型を含む場合は除外 | インターフェース検証時 |
| BR-03 | CompareTo必須 | CompareTo()メソッドの実装が必要 | メソッド取得時 |
| BR-04 | 演算子存在確認 | 既存の演算子は生成しない | 演算子生成時 |
| BR-05 | 複数IComparable | 複数のIComparable<T>実装時はサブメニューで選択 | UI登録時 |

### 計算ロジック

**演算子生成パターン**：
- `operator <`: `return left.CompareTo(right) < 0;`
- `operator <=`: `return left.CompareTo(right) <= 0;`
- `operator >`: `return left.CompareTo(right) > 0;`
- `operator >=`: `return left.CompareTo(right) >= 0;`

## データベース操作仕様

該当なし（ソースコード操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| E-01 | 位置エラー | 無効なカーソル位置 | リファクタリングを提供しない |
| E-02 | インターフェース不在 | IComparable<T>未実装 | リファクタリングを提供しない |
| E-03 | メソッド不在 | CompareTo()未実装 | リファクタリングを提供しない |
| E-04 | 演算子完備 | すべての演算子が定義済み | リファクタリングを提供しない |

### リトライ仕様

なし

## トランザクション仕様

ドキュメント変更は単一のCodeActionでアトミックに適用される

## パフォーマンス要件

- インターフェース検索と演算子チェックは同期で実行
- コード生成と挿入は非同期で実行

## セキュリティ考慮事項

特になし

## 備考

- C#とVisual Basicの両言語をサポート
- 明示的インターフェース実装の場合はキャストを含むコードを生成

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

演算子の種類と定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CodeGenerationOperatorKind.cs | `src/Workspaces/Core/Portable/CodeGeneration/` | 演算子種類の列挙型 |

**読解のコツ**: s_operatorKindsで使用される4種類の演算子（LessThan、LessThanOrEqual、GreaterThan、GreaterThanOrEqual）を理解する。

#### Step 2: エントリーポイントを理解する

リファクタリングプロバイダーが処理の起点となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GenerateComparisonOperatorsCodeRefactoringProvider.cs | `src/Features/Core/Portable/GenerateComparisonOperators/` | ComputeRefactoringsAsyncが処理の開始点 |

**主要処理フロー**:
1. **28-34行目**: s_operatorKindsで生成対象の演算子種類を定義
2. **42-56行目**: カーソル位置の検証
3. **61-63行目**: IComparable<T>型の取得
4. **70-87行目**: 各IComparable<T>インターフェースについて処理
5. **92-117行目**: 単一/複数IComparableに応じたアクション登録

#### Step 3: 演算子生成ロジックを理解する

演算子コードの生成ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GenerateComparisonOperatorsCodeRefactoringProvider.cs | `src/Features/Core/Portable/GenerateComparisonOperators/` | GenerateComparisonOperatorsAsyncメソッド |

**主要処理フロー**:
- **131-154行目**: GenerateComparisonOperatorsAsyncで全体処理
- **156-170行目**: GenerateLeftExpressionでleft式生成（キャスト含む）
- **172-204行目**: GenerateComparisonOperatorsで各演算子生成
- **206-225行目**: GenerateStatementで`left.CompareTo(right) < 0`形式のコード生成
- **238-252行目**: HasComparisonOperatorで既存演算子をチェック

### プログラム呼び出し階層図

```
GenerateComparisonOperatorsCodeRefactoringProvider.ComputeRefactoringsAsync
    │
    ├─ IRefactoringHelpersService.IsOnTypeHeader / IsBetweenTypeMembers
    │      └─ カーソル位置の検証
    │
    ├─ Compilation.GetTypeByMetadataName(IComparable<>)
    │      └─ IComparable<T>型の取得
    │
    ├─ containingType.Interfaces (foreach)
    │      │
    │      ├─ TryGetCompareMethodImpl
    │      │      └─ CompareTo()実装の取得
    │      │
    │      └─ HasAllComparisonOperators
    │             └─ 既存演算子の確認
    │
    └─ CodeAction.Create
           │
           └─ GenerateComparisonOperatorsAsync
                  │
                  ├─ GenerateLeftExpression
                  │      └─ left式の生成（キャスト含む）
                  │
                  ├─ GenerateComparisonOperators
                  │      └─ 各演算子の生成
                  │
                  └─ ICodeGenerationService.AddMembersAsync
                         └─ ドキュメントへの挿入
```

### データフロー図

```
[入力]               [処理]                    [出力]

カーソル位置 ───▶ 位置検証 ───▶ (検証結果)
      │
      ▼
型シンボル ───▶ IComparable検索 ───▶ IComparable<T>一覧
      │
      ▼
IComparable<T> ───▶ 未定義演算子特定 ───▶ 生成対象演算子
      │
      ▼
CompareTo実装 ───▶ 演算子コード生成 ───▶ 演算子メソッド
      │
      ▼
生成コード ───▶ ドキュメント挿入 ───▶ 更新ドキュメント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GenerateComparisonOperatorsCodeRefactoringProvider.cs | `src/Features/Core/Portable/GenerateComparisonOperators/` | ソース | リファクタリングプロバイダー本体 |
| CodeGenerationOperatorKind.cs | `src/Workspaces/Core/Portable/CodeGeneration/` | ソース | 演算子種類の定義 |
| CodeGenerationSymbolFactory.cs | `src/Workspaces/Core/Portable/CodeGeneration/` | ソース | シンボル生成ユーティリティ |
| ICodeGenerationService.cs | `src/Workspaces/Core/Portable/CodeGeneration/` | ソース | コード挿入サービス |
