# 機能設計書 52-数値リテラル変換

## 概要

本ドキュメントは、Roslyn機能「数値リテラル変換」の機能設計書である。16進数・2進数・10進数の相互変換、および桁区切り文字の追加・削除を提供するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：ソフトウェア開発において、数値リテラルの表記方法は用途によって最適な形式が異なる。ビット操作では2進数や16進数が読みやすく、一般的な計算では10進数が適切である。また、大きな数値には桁区切り文字（`_`）を使用することで可読性が向上する。本機能は、これらの表記形式を相互変換することで、コードの可読性向上と保守性の向上を支援する。

**機能の利用シーン**：
- 16進数で定義された定数を10進数で確認したい場合
- ビットフラグを2進数表記に変換して各ビットの意味を明確にしたい場合
- 大きな数値に桁区切り文字を追加して可読性を向上させたい場合
- コーディング規約に合わせて数値リテラルの形式を統一したい場合

**主要な処理内容**：
1. 10進数から16進数への変換（例：255 → 0xFF）
2. 10進数から2進数への変換（例：255 → 0b11111111）
3. 16進数から10進数への変換（例：0xFF → 255）
4. 2進数から10進数への変換（例：0b11111111 → 255）
5. 桁区切り文字の追加（例：1000000 → 1_000_000）
6. 桁区切り文字の削除（例：1_000_000 → 1000000）

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| NumericLiteralExpression | SyntaxNode | Yes | 数値リテラル式のAST | 整数型であること |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |

### 入力データソース

エディタ内の選択位置またはカーソル位置にある数値リテラル式

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ConvertedToken | SyntaxToken | 変換後の数値リテラルトークン |
| UpdatedDocument | Document | 更新後のドキュメント |

### 出力先

エディタ（ソースコードの置換）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のノードを特定

2. 数値リテラルの検証
   └─ 整数型であるかを確認
   └─ 診断エラーがないことを確認
   └─ 定数値を取得

3. 現在の表記形式を判定
   └─ プレフィックス（0x, 0b）を解析
   └─ サフィックス（L, UL等）を保持

4. 変換候補の生成
   └─ 10進数以外の場合: 10進数変換を追加
   └─ 2進数以外の場合: 2進数変換を追加
   └─ 16進数以外の場合: 16進数変換を追加
   └─ 桁区切り有無に応じた変換を追加

5. CodeActionの登録
   └─ 単一候補の場合: 直接登録
   └─ 複数候補の場合: グループ化して登録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[数値リテラルを取得]
    B --> C{整数型?}
    C -->|No| D[リファクタリング提示なし]
    C -->|Yes| E[現在の形式を判定]
    E --> F{10進数?}
    F -->|Yes| G[16進数・2進数変換を提案]
    F -->|No| H[10進数変換を提案]
    G --> I{桁区切り有?}
    H --> I
    I -->|Yes| J[区切り削除を提案]
    I -->|No| K[区切り追加を提案]
    J --> L[CodeAction登録]
    K --> L
    L --> M[終了]
    D --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-52-01 | 整数型のみ | 浮動小数点数は変換対象外 | 数値リテラルが整数型でない場合 |
| BR-52-02 | プレフィックス保持 | 言語固有のプレフィックス（0x, 0b）を使用 | 16進数・2進数変換時 |
| BR-52-03 | サフィックス保持 | 元の型サフィックス（L, UL等）を維持 | 変換実行時 |
| BR-52-04 | 桁区切り間隔 | 10進数は3桁、16進数・2進数は4桁で区切り | 桁区切り追加時 |
| BR-52-05 | 低優先度 | 数値変換は低優先度のリファクタリング | 常時 |

### 計算ロジック

- 桁区切りの追加位置: 右から指定間隔ごとに`_`を挿入
- 負の数の処理: Int64に変換して処理

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | 浮動小数点数の場合 | リファクタリングを提示しない |
| - | 診断エラー | 構文エラーがある場合 | リファクタリングを提示しない |
| - | 不明な形式 | 認識できないプレフィックス | リファクタリングを提示しない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 正規表現によるパターンマッチングを使用
- コンパイル済み正規表現でパフォーマンスを最適化

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

本機能はCodeActionRequestPriority.Lowとして設定されており、他のリファクタリングより低い優先度で表示される。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractConvertNumericLiteralCodeRefactoringProvider.cs | `src/Features/Core/Portable/ConvertNumericLiteral/` | 数値リテラル変換の抽象クラス |

**主要処理フロー**:
1. **19-25行目**: NumericKind列挙型で数値形式を定義（Unknown, Decimal, Binary, Hexadecimal）
2. **31-32行目**: ComputeRequestPriorityで低優先度を返却
3. **34-137行目**: ComputeRefactoringsAsyncでメインロジックを実行

#### Step 2: 変換ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractConvertNumericLiteralCodeRefactoringProvider.cs | `src/Features/Core/Portable/ConvertNumericLiteral/` | 変換処理の詳細 |

**主要処理フロー**:
- **52-53行目**: 整数型チェック（IsIntegralType）
- **63-67行目**: プレフィックスから現在の形式を判定
- **74-81行目**: 各形式への変換候補を生成
- **84-97行目**: 桁区切りの追加・削除候補を生成
- **114-120行目**: AddSeparatorsローカル関数で桁区切りを挿入
- **122-129行目**: ReplaceTokenAsyncでトークンを置換

**読解のコツ**: 正規表現パターン（25行目）がプレフィックス・数値本体・サフィックスを分離する仕組みを理解する。

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync
    │
    ├─ TryGetRelevantNodeAsync<TNumericLiteralExpression>
    │      └─ 対象ノードの取得
    │
    ├─ ISyntaxFactsService.IsNumericLiteralExpression
    │      └─ 数値リテラルの検証
    │
    ├─ SemanticModel.GetTypeInfo
    │      └─ 型情報の取得
    │
    ├─ GetNumericLiteralParts
    │      └─ プレフィックス・数値・サフィックスの分離
    │
    ├─ CreateCodeAction (複数回)
    │      └─ 各変換候補のCodeAction生成
    │
    └─ ReplaceTokenAsync
           └─ トークンの置換とドキュメント更新
```

### データフロー図

```
[入力]                [処理]                      [出力]

NumericLiteral ───▶ GetNumericLiteralParts ───▶ (prefix, number, suffix)
                          │
                          ▼
(prefix, number, suffix) ───▶ 変換ロジック ───▶ 新しいリテラル文字列
                          │
                          ▼
新しいリテラル ───▶ SyntaxGenerator.NumericLiteralToken ───▶ 更新済みDocument
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractConvertNumericLiteralCodeRefactoringProvider.cs | `src/Features/Core/Portable/ConvertNumericLiteral/` | ソース | 数値リテラル変換の抽象プロバイダ |
| IntegerUtilities.cs | `src/Workspaces/SharedUtilitiesAndExtensions/Compiler/Core/Utilities/` | ソース | 整数変換ユーティリティ |
