# 機能設計書 54-キャスト変換

## 概要

本ドキュメントは、Roslyn機能「キャスト変換」の機能設計書である。as演算子とキャストの相互変換を提供するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：C#では型変換に2つの方法がある。キャスト演算子（`(Type)expr`）とas演算子（`expr as Type`）である。キャストは失敗時に例外をスローし、asは失敗時にnullを返す。開発者は状況に応じて適切な方法を選択する必要があり、本機能はこれらの相互変換を支援することで、コードの意図の明確化とエラーハンドリングの改善を可能にする。

**機能の利用シーン**：
- キャストをasに変換してnullチェックによる安全な型変換に変更したい場合
- asをキャストに変換して型変換失敗時に例外を発生させたい場合
- コードレビューでチームの推奨パターンに統一したい場合
- Nullable参照型の導入に伴いnull安全性を向上させたい場合

**主要な処理内容**：
1. キャスト式（`(Type)expr`）をas式（`expr as Type`）に変換
2. as式（`expr as Type`）をキャスト式（`(Type)expr`）に変換
3. Nullable参照型コンテキストでの適切な型付け

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| FromExpression | SyntaxNode | Yes | 変換元の式（キャストまたはas） | 参照型またはNullable値型であること |
| TypeNode | SyntaxNode | Yes | 変換先の型ノード | エラー型でないこと |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |

### 入力データソース

エディタ内の選択位置またはカーソル位置にあるキャスト式またはas式

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ConvertedExpression | SyntaxNode | 変換後の式 |
| UpdatedDocument | Document | 更新後のドキュメント |

### 出力先

エディタ（ソースコードの置換）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のノードを特定

2. 対象式の検証
   └─ キャストまたはas式であるかを確認
   └─ 診断エラーがないことを確認

3. 型の検証
   └─ 型情報を取得
   └─ エラー型でないことを確認
   └─ 参照型またはNullable値型であることを確認

4. 変換の実行
   └─ キャスト → as: ConvertExpressionメソッドで変換
   └─ as → キャスト: ConvertExpressionメソッドで変換

5. ドキュメントの更新
   └─ 元の式を変換後の式で置換
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{式の種類}
    B -->|キャスト式| C[型を取得]
    B -->|as式| D[型を取得]
    C --> E{参照型 or Nullable?}
    D --> E
    E -->|Yes| F[変換を実行]
    E -->|No| G[リファクタリング提示なし]
    F --> H[ドキュメント更新]
    H --> I[終了]
    G --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-54-01 | 参照型のみ | 値型（Nullable除く）は変換対象外 | asは参照型のみ対応のため |
| BR-54-02 | Nullable対応 | Nullable<T>型は変換可能 | 値型でもNullableの場合 |
| BR-54-03 | エラー型除外 | 型解決エラーがある場合は変換不可 | TypeKind.Error |
| BR-54-04 | NullableContext | Nullable参照型コンテキストを考慮 | 変換時の型注釈 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | 値型（非Nullable）の場合 | リファクタリングを提示しない |
| - | 診断エラー | 構文エラーがある場合 | リファクタリングを提示しない |
| - | 型エラー | 型が解決できない場合 | リファクタリングを提示しない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 単純なノード置換のため高速に実行可能

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

- asへの変換時はnullチェックが必要になる可能性がある（別途警告アナライザーで対応）
- キャストへの変換時は例外処理が必要になる可能性がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractConvertCastCodeRefactoringProvider.cs | `src/Features/Core/Portable/ConvertCast/` | キャスト変換の抽象クラス |

**主要処理フロー**:
1. **14-22行目**: クラスコメントで変換例を説明
2. **29-34行目**: 抽象メンバー（GetTitle, FromKind, ConvertExpression, GetTypeNode）
3. **35-66行目**: ComputeRefactoringsAsyncでリファクタリング候補を計算
4. **48-53行目**: 型の検証（エラー型チェック、参照型チェック）
5. **68-78行目**: ConvertAsyncで実際の変換を実行

**読解のコツ**: FromKind抽象プロパティで変換元の種類（キャストまたはas）を識別する仕組みを理解する。

#### Step 2: 言語固有実装を確認

C#およびVBの具体的な実装は、それぞれの言語フィーチャーディレクトリに配置されている。

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync
    │
    ├─ GetRelevantNodesAsync<TFromExpression>
    │      └─ 対象ノードの取得
    │
    ├─ GetTypeNode
    │      └─ 型ノードの取得
    │
    ├─ SemanticModel.GetTypeInfo
    │      └─ 型情報の取得
    │
    ├─ 型の検証
    │      ├─ TypeKind.Error チェック
    │      └─ IsReferenceType / Nullable チェック
    │
    └─ ConvertAsync
           ├─ ConvertExpression
           └─ Document.WithSyntaxRoot
```

### データフロー図

```
[入力]              [処理]                    [出力]

(Type)expr ───▶ GetTypeNode ───▶ Type
                     │
                     ▼
Type ───▶ GetTypeInfo ───▶ ITypeSymbol
                     │
                     ▼
((Type)expr, Type) ───▶ ConvertExpression ───▶ expr as Type

or

expr as Type ───▶ ConvertExpression ───▶ (Type)expr
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractConvertCastCodeRefactoringProvider.cs | `src/Features/Core/Portable/ConvertCast/` | ソース | キャスト変換の抽象プロバイダ |
