# 機能設計書 55-条件式をステートメントに変換

## 概要

本ドキュメントは、Roslyn機能「条件式をステートメントに変換」の機能設計書である。三項演算子（条件式）をif-elseステートメントに変換するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：三項演算子（条件式）はコンパクトに条件分岐を記述できる一方、複雑な場合は可読性が低下する。特にネストした条件式や、各分岐の処理が長い場合は、if-elseステートメントの方が理解しやすい。本機能は、条件式をif-elseステートメントに変換することで、コードの可読性と保守性を向上させる。

**機能の利用シーン**：
- 複雑な三項演算子を展開してデバッグしやすくしたい場合
- 条件分岐にログ出力やその他の処理を追加したい場合
- コードレビューで可読性向上を求められた場合
- チームのコーディング規約で条件式を禁止している場合

**主要な処理内容**：
1. 条件式（`condition ? trueExpr : falseExpr`）をif-elseステートメントに変換
2. 代入式内の条件式を展開（`x = condition ? a : b` → `if (condition) x = a; else x = b;`）
3. return文内の条件式を展開（`return condition ? a : b;` → `if (condition) return a; else return b;`）
4. 変数宣言内の条件式を展開

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ConditionalExpression | SyntaxNode | Yes | 条件式のAST | 有効な三項演算子であること |
| ParentStatement | SyntaxNode | Yes | 親ステートメント | 変換可能なコンテキストであること |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |

### 入力データソース

エディタ内の選択位置またはカーソル位置にある条件式

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| IfStatement | SyntaxNode | 変換後のif-elseステートメント |
| UpdatedDocument | Document | 更新後のドキュメント |

### 出力先

エディタ（ソースコードの置換）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のノードを特定

2. 条件式の検証
   └─ 三項演算子であるかを確認
   └─ 変換可能なコンテキストかを確認

3. 親ステートメントの解析
   └─ 代入式、return文、変数宣言等を判定
   └─ 変換パターンを決定

4. if-elseステートメントの生成
   └─ 条件部分を抽出
   └─ trueブランチのステートメントを生成
   └─ falseブランチのステートメントを生成

5. ドキュメントの更新
   └─ 元のステートメントをif-elseで置換
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[条件式を取得]
    B --> C{変換可能なコンテキスト?}
    C -->|No| D[リファクタリング提示なし]
    C -->|Yes| E{親の種類}
    E -->|代入式| F[代入をif-elseに展開]
    E -->|return文| G[returnをif-elseに展開]
    E -->|変数宣言| H[宣言をif-elseに展開]
    E -->|式ステートメント| I[式をif-elseに展開]
    F --> J[ドキュメント更新]
    G --> J
    H --> J
    I --> J
    J --> K[終了]
    D --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-01 | ステートメント内のみ | 式の途中（メソッド引数等）では変換不可 | 条件式がステートメントのトップレベルでない場合 |
| BR-55-02 | ブロック必要時追加 | 単一ステートメントの場合もブロックで囲む | 常時（安全性のため） |
| BR-55-03 | トリビア保持 | コメントや空白を適切に保持 | 変換時 |
| BR-55-04 | 変数初期化対応 | 変数宣言と代入を分離して処理 | 変数宣言内の条件式 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | メソッド引数内の条件式 | リファクタリングを提示しない |
| - | 変換不可 | ラムダ式本体の条件式 | リファクタリングを提示しない |
| - | 診断エラー | 構文エラーがある場合 | リファクタリングを提示しない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 単純なAST変換のため高速に実行可能

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

- ネストした条件式は再帰的に処理可能
- 逆方向の変換（if-else → 条件式）は別機能として提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractReplaceConditionalWithStatementsCodeRefactoringProvider.cs | `src/Features/Core/Portable/ReplaceConditionalWithStatements/` | 条件式変換の抽象クラス |

**主要処理フロー**:
1. **ComputeRefactoringsAsync**: 条件式を検出してリファクタリング候補を登録
2. **GetReplacementAsync**: 条件式をif-elseステートメントに変換
3. **抽象メンバー**: HasSingleVariable, GetCondition, GetWhenTrue, GetWhenFalse等

**読解のコツ**: 条件式が配置されているコンテキスト（代入、return、変数宣言）を判定する仕組みを理解する。

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync
    │
    ├─ TryGetRelevantNodeAsync<TConditionalExpression>
    │      └─ 条件式の取得
    │
    ├─ GetContainingStatement
    │      └─ 親ステートメントの取得
    │
    ├─ IsValidConditional
    │      └─ 変換可能性の検証
    │
    └─ GetReplacementAsync
           ├─ GetCondition
           ├─ GetWhenTrue
           ├─ GetWhenFalse
           └─ Generator.IfStatement
```

### データフロー図

```
[入力]                    [処理]                           [出力]

condition ? a : b ───▶ GetCondition ───▶ condition
                   ───▶ GetWhenTrue ───▶ a
                   ───▶ GetWhenFalse ───▶ b
                            │
                            ▼
(condition, a, b) ───▶ Generator.IfStatement ───▶ if (condition) { a } else { b }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractReplaceConditionalWithStatementsCodeRefactoringProvider.cs | `src/Features/Core/Portable/ReplaceConditionalWithStatements/` | ソース | 条件式変換の抽象プロバイダ |
