# 機能設計書 56-メソッドをプロパティに変換

## 概要

本ドキュメントは、Roslyn機能「メソッドをプロパティに変換」の機能設計書である。パラメータなしのゲッターメソッドをプロパティに変換するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：C#ではプロパティとパラメータなしメソッドが似た用途で使用されることがある。プロパティはフィールドライクなアクセス構文を提供し、外部から見て値の取得・設定を直感的に行える。本機能は、ゲッター/セッターパターンのメソッドをプロパティに変換することで、APIの一貫性と使いやすさを向上させる。

**機能の利用シーン**：
- GetXxx()/SetXxx()パターンのメソッドをプロパティに統合したい場合
- レガシーコードをモダンなC#スタイルに移行したい場合
- .NETフレームワーク設計ガイドラインに準拠したい場合
- インテリセンスでプロパティとして表示させたい場合

**主要な処理内容**：
1. GetXxx()メソッドをXxxプロパティのgetterに変換
2. 対応するSetXxx(value)メソッドをsetterに統合
3. 全ての参照箇所をプロパティアクセスに更新
4. ソリューション全体での一括変換

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| MethodDeclaration | SyntaxNode | Yes | メソッド宣言のAST | パラメータなしのゲッターメソッド |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |
| Solution | Solution | Yes | ソリューション | null不可（参照更新用） |

### 入力データソース

エディタ内の選択位置またはカーソル位置にあるメソッド宣言

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PropertyDeclaration | SyntaxNode | 変換後のプロパティ宣言 |
| UpdatedSolution | Solution | 参照更新済みのソリューション |

### 出力先

エディタおよびソリューション全体（参照箇所の更新）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のメソッド宣言を特定

2. メソッドの検証
   └─ パラメータがないことを確認
   └─ 戻り値がvoidでないことを確認
   └─ GetXxxの命名パターンを確認

3. 対応するSetXxxメソッドの検索
   └─ 同じ型のパラメータを持つSetXxxを検索
   └─ 見つかった場合はgetter/setterペアとして処理

4. プロパティ名の決定
   └─ "Get"プレフィックスを除去
   └─ 既存メンバーとの名前衝突を確認

5. 参照の検索
   └─ SymbolFinderで全参照を検索
   └─ GetXxx()呼び出しをプロパティ読み取りに変換
   └─ SetXxx(value)呼び出しをプロパティ書き込みに変換

6. ドキュメントの更新
   └─ メソッドをプロパティに置換
   └─ 全参照箇所を更新
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[メソッド宣言を取得]
    B --> C{パラメータなし?}
    C -->|No| D[リファクタリング提示なし]
    C -->|Yes| E{戻り値がvoid?}
    E -->|Yes| D
    E -->|No| F[プロパティ名を決定]
    F --> G{対応するSetXxxあり?}
    G -->|Yes| H[getter/setterプロパティ生成]
    G -->|No| I[getterのみプロパティ生成]
    H --> J[全参照を検索]
    I --> J
    J --> K[参照をプロパティアクセスに更新]
    K --> L[ソリューション更新]
    L --> M[終了]
    D --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-01 | パラメータなし | パラメータを持つメソッドは変換不可 | 常時 |
| BR-56-02 | 非void戻り値 | voidを返すメソッドは変換不可 | 常時 |
| BR-56-03 | Getプレフィックス除去 | GetXxx → Xxxの命名変換 | プロパティ名決定時 |
| BR-56-04 | Setメソッド統合 | SetXxx(T value)をsetterに変換 | 対応メソッドが存在する場合 |
| BR-56-05 | 名前衝突回避 | 既存メンバーと衝突する場合は別名を生成 | 常時 |
| BR-56-06 | ソリューション全体更新 | 全ての参照箇所を更新 | 常時 |

### 計算ロジック

- プロパティ名生成: `GetXxx` → `Xxx`（先頭の"Get"を除去）
- 名前衝突回避: NameGeneratorで一意な名前を生成

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | パラメータありメソッド | リファクタリングを提示しない |
| - | 変換不可 | voidメソッド | リファクタリングを提示しない |
| - | 警告 | メタデータ参照あり | 警告アノテーションを追加 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 全参照の検索が必要なため、大規模ソリューションでは時間がかかる可能性あり
- SymbolFinderによる非同期参照検索

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

- インターフェースメンバーの場合は適切にAsInterfaceMemberとして生成
- 逆方向の変換（プロパティ → メソッド）は「プロパティをメソッドに変換」機能で提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IReplaceMethodWithPropertyService.cs | `src/Features/Core/Portable/ReplaceMethodWithProperty/` | 言語サービスインターフェース |
| 1-2 | GetAndSetMethods構造体 | 同上 | ゲッター・セッターペアの管理 |

**読解のコツ**: GetAndSetMethods構造体がゲッターとセッターのペアを保持する仕組みを理解する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ReplaceMethodWithPropertyCodeRefactoringProvider.cs | `src/Features/Core/Portable/ReplaceMethodWithProperty/` | メイン処理クラス |

**主要処理フロー**:
1. **32-73行目**: ComputeRefactoringsAsyncでメソッドを解析
2. **75-108行目**: ReplaceMethodsWithPropertyAsyncでプロパティ生成と参照更新
3. **110-152行目**: 対応するSetメソッドの検索ロジック
4. **154-224行目**: UpdateReferencesAsyncで参照の更新

**読解のコツ**: SymbolFinderを使用した参照検索と、SyntaxEditorによる一括更新の仕組みを理解する。

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync
    │
    ├─ GetMethodDeclarationAsync
    │      └─ メソッド宣言の取得
    │
    ├─ GetDeclaredSymbol
    │      └─ メソッドシンボルの取得
    │
    ├─ FindSetMethod
    │      └─ 対応するSetXxxメソッドの検索
    │
    └─ ReplaceMethodsWithPropertyAsync
           ├─ SymbolFinder.FindReferencesAsync
           │      └─ 全参照の検索
           ├─ UpdateReferencesAsync
           │      ├─ ReplaceGetReference
           │      └─ ReplaceSetReference
           └─ ReplaceDefinitionsWithPropertyAsync
                  └─ プロパティ宣言の生成
```

### データフロー図

```
[入力]                       [処理]                        [出力]

GetXxx() method ───▶ GetMethodDeclarationAsync ───▶ IMethodSymbol
                            │
                            ▼
IMethodSymbol ───▶ FindSetMethod ───▶ GetAndSetMethods
                            │
                            ▼
GetAndSetMethods ───▶ ReplaceMethodsWithPropertyAsync ───▶ Property
                            │
                            ▼
Property ───▶ UpdateReferencesAsync ───▶ Updated Solution
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ReplaceMethodWithPropertyCodeRefactoringProvider.cs | `src/Features/Core/Portable/ReplaceMethodWithProperty/` | ソース | メイン処理クラス |
| IReplaceMethodWithPropertyService.cs | `src/Features/Core/Portable/ReplaceMethodWithProperty/` | ソース | 言語サービスインターフェース |
