# 機能設計書 58-if文の分割・結合

## 概要

本ドキュメントは、Roslyn機能「if文の分割・結合」の機能設計書である。複合条件を持つif文の分割（`if (a && b)` → `if (a) if (b)`）と、連続するif文の結合（`if (a) if (b)` → `if (a && b)`）を提供するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：if文の条件が複雑になると可読性が低下する。また、デバッグ時に各条件を個別に確認したいケースもある。本機能は、論理AND（`&&`）や論理OR（`||`）で結合された条件を分割したり、逆に連続するif文を結合することで、コードの構造を柔軟に変更できるようにする。

**機能の利用シーン**：
- 複雑な条件式を分割してデバッグしやすくしたい場合
- 短絡評価の各条件にブレークポイントを設定したい場合
- 類似した条件の連続if文を1つにまとめたい場合
- コードレビューで条件の論理構造を明確化したい場合

**主要な処理内容**：
1. `if (a && b)` → `if (a) { if (b) { ... } }`（AND分割）
2. `if (a || b)` → `if (a) { ... } else if (b) { ... }`（OR分割）
3. 連続するif文の結合（`if (a) if (b)` → `if (a && b)`）
4. else-if チェーンの結合

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| IfStatement | SyntaxNode | Yes | if文のAST | 分割・結合可能な構造であること |
| BinaryExpression | SyntaxNode | Conditional | 論理AND/OR式 | 分割時に必要 |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |

### 入力データソース

エディタ内の選択位置またはカーソル位置にあるif文または論理演算子

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TransformedIfStatements | SyntaxNode[] | 変換後のif文（複数の場合あり） |
| UpdatedDocument | Document | 更新後のドキュメント |

### 出力先

エディタ（ソースコードの置換）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のノードを特定

2. 操作種別の判定
   └─ 分割: 論理AND/OR式を含むif文
   └─ 結合: 連続するif文

3. 分割処理（AND/ORの場合）
   └─ 二項式を左右に分解
   └─ ネストしたif文を生成（ANDの場合）
   └─ else-ifチェーンを生成（ORの場合）

4. 結合処理（連続if文の場合）
   └─ 条件を論理AND/ORで結合
   └─ 本体をマージ

5. ドキュメントの更新
   └─ 元のif文を変換後の構造で置換
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{操作種別}
    B -->|分割| C{演算子種別}
    B -->|結合| D[連続if文を解析]
    C -->|AND| E[ネストしたif文を生成]
    C -->|OR| F[else-ifチェーンを生成]
    D --> G[条件を結合]
    E --> H[ドキュメント更新]
    F --> H
    G --> H
    H --> I[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-58-01 | AND分割 | `if (a && b)` → ネストした`if (a) { if (b) {...} }` | AND式を含むif文 |
| BR-58-02 | OR分割 | `if (a \|\| b)` → `if (a) {...} else if (b) {...}` | OR式を含むif文 |
| BR-58-03 | 条件チェーン | 最上位の二項式のみ分割対象 | 分割時 |
| BR-58-04 | 結合方向 | 上方向・下方向の両方を提案 | 結合可能時 |
| BR-58-05 | スコープブロック | スコープブロックの階層を適切に処理 | 結合時 |

### 計算ロジック

- 左結合二項式の分割: `((a && b) && c)` → `a && b`（左）と`c`（右）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | 結合不可能な構造 | リファクタリングを提示しない |
| - | 診断エラー | 構文エラーがある場合 | リファクタリングを提示しない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 単純なAST変換のため高速に実行可能

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

- C#ではelse if（`if (a) {} else if (b) {}`）がネストしたif文として表現されるため、内部的にはフラット化して処理
- VBにはElseIfが独立した構文要素として存在

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: サービスインターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IIfLikeStatementGenerator.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | if文操作のインターフェース |

**主要メソッド**:
- **18-19行目**: IsIfOrElseIf - if/else-ifの判定
- **20行目**: IsCondition - 条件式の判定
- **26行目**: GetCondition - 条件の取得
- **36行目**: GetElseIfAndElseClauses - else-ifチェーンの取得
- **60-65行目**: InsertElseIfClause/RemoveElseIfClause - else-if操作

#### Step 2: 分割処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractSplitIfStatementCodeRefactoringProvider.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | 分割処理の抽象クラス |

**主要処理フロー**:
1. **33-58行目**: ComputeRefactoringsAsyncでリファクタリング候補を計算
2. **49-51行目**: IsPartOfBinaryExpressionChainで二項式チェーンを検出
3. **60-76行目**: RefactorAsyncで分割を実行
4. **78-92行目**: IsPartOfBinaryExpressionChainで最上位の二項式を特定
5. **94-110行目**: SplitBinaryExpressionChainで左右に分解

#### Step 3: 結合処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AbstractMergeIfStatementsCodeRefactoringProvider.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | 結合処理の抽象クラス |

**主要処理フロー**:
1. **35-62行目**: ComputeRefactoringsAsyncで結合候補を計算
2. **46-50行目**: CanBeMergedUp/CanBeMergedDownで結合可能性を判定
3. **64-85行目**: RefactorAsyncで結合を実行
4. **87-122行目**: WalkDownScopeBlocks/WalkUpScopeBlocksでスコープ処理

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync (Split)
    │
    ├─ IsPartOfBinaryExpressionChain
    │      └─ 二項式チェーンの検出
    │
    ├─ IsCondition
    │      └─ if文条件かの判定
    │
    └─ GetChangedRootAsync
           ├─ SplitBinaryExpressionChain
           └─ 新しいif文構造の生成

ComputeRefactoringsAsync (Merge)
    │
    ├─ IsApplicableSpan
    │      └─ 対象範囲の判定
    │
    ├─ CanBeMergedUp / CanBeMergedDown
    │      └─ 結合可能性の判定
    │
    └─ GetChangedRoot
           └─ 条件の結合とif文の再構築
```

### データフロー図

```
[入力]                    [処理]                      [出力]

if (a && b) { C } ───▶ SplitBinaryExpressionChain ───▶ if (a) { if (b) { C } }
                              │
                              └─ (left=a, right=b)

if (a) { if (b) { C } } ───▶ Merge ───▶ if (a && b) { C }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractSplitIfStatementCodeRefactoringProvider.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | ソース | 分割処理の抽象プロバイダ |
| AbstractMergeIfStatementsCodeRefactoringProvider.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | ソース | 結合処理の抽象プロバイダ |
| IIfLikeStatementGenerator.cs | `src/Features/Core/Portable/SplitOrMergeIfStatements/` | ソース | if文操作インターフェース |
