# 機能設計書 60-論理反転

## 概要

本ドキュメントは、Roslyn機能「論理反転」の機能設計書である。論理AND（`&&`）と論理OR（`||`）を相互変換し、ド・モルガンの法則を適用するコードリファクタリング機能について記述する。

### 本機能の処理概要

**業務上の目的・背景**：複雑な条件式は、否定形で書くと読みやすくなる場合がある。ド・モルガンの法則（`!(a && b)` = `!a || !b`）を適用することで、条件式の構造を変更しながら論理的等価性を保つことができる。本機能は、この変換を自動的に行うことで、コードの可読性向上と論理構造の最適化を支援する。

**機能の利用シーン**：
- `!a || !b` を `!(a && b)` に簡略化したい場合
- `!(a && b)` を `!a || !b` に展開してデバッグしやすくしたい場合
- 条件式の論理構造を変更して可読性を向上させたい場合
- 二重否定を除去したい場合

**主要な処理内容**：
1. `!a || !b` → `!(a && b)`（OR+否定をAND+外側否定に変換）
2. `!a && !b` → `!(a || b)`（AND+否定をOR+外側否定に変換）
3. `!(a && b)` → `!a || !b`（外側否定をOR+否定に展開）
4. `!(a || b)` → `!a && !b`（外側否定をAND+否定に展開）
5. 二重否定の自動除去

**関連システム・外部連携**：該当なし（IDE内機能）

**権限による制御**：該当なし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | エディタ | 主画面 | リファクタリング候補の表示とクイックアクション実行 |

## 機能種別

コードリファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| BinaryExpression | SyntaxNode | Yes | 論理AND/OR式のAST | LogicalAnd/LogicalOrであること |
| Document | Document | Yes | 対象ドキュメント | null不可 |
| SemanticModel | SemanticModel | Yes | セマンティック情報 | null不可 |

### 入力データソース

エディタ内の選択位置またはカーソル位置にある論理AND/OR式

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| InvertedExpression | SyntaxNode | 変換後の論理式 |
| UpdatedDocument | Document | 更新後のドキュメント |

### 出力先

エディタ（ソースコードの置換）

## 処理フロー

### 処理シーケンス

```
1. リファクタリングコンテキストの取得
   └─ カーソル位置のノードを特定

2. 対象式の検証
   └─ 論理AND/OR式であるかを確認
   └─ 最上位の二項式まで遡る

3. 内部否定の適用
   └─ SyntaxGenerator.Negateで式全体を否定
   └─ ド・モルガンの法則を適用

4. 外部否定の適用
   └─ 変換結果を再度否定
   └─ 二重否定の自動除去

5. ドキュメントの更新
   └─ 元の式を変換後の式で置換
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[論理式を取得]
    B --> C{AND/OR?}
    C -->|Yes| D[最上位の二項式を特定]
    C -->|No| E[リファクタリング提示なし]
    D --> F{選択範囲?}
    F -->|空| G[チェーン全体を対象]
    F -->|非空| H[選択部分を対象]
    G --> I[内部否定を適用]
    H --> I
    I --> J[外部否定を適用]
    J --> K[二重否定を除去]
    K --> L[ドキュメント更新]
    L --> M[終了]
    E --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-60-01 | ド・モルガン適用 | `!(a && b)` ↔ `!a \|\| !b` | AND/OR変換時 |
| BR-60-02 | 演算子反転 | `&&` ↔ `\|\|` | 変換時 |
| BR-60-03 | チェーン全体変換 | 空選択時は同種演算子のチェーン全体を対象 | 選択範囲が空の場合 |
| BR-60-04 | 部分選択不可 | 中間ノードの選択は対象外 | 部分選択時 |
| BR-60-05 | 二重否定除去 | `!!x` → `x` | 常時 |
| BR-60-06 | アノテーション保持 | 変換結果にアノテーションを付与して識別 | 内部処理 |

### 計算ロジック

- 反転後の演算子: LogicalAnd ↔ LogicalOr
- 否定の適用: 各オペランドに`!`を追加、演算子を反転

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（メモリ内操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 変換不可 | AND/OR以外の式 | リファクタリングを提示しない |
| - | 変換不可 | 部分選択されたチェーン | リファクタリングを提示しない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 2段階の否定処理を非同期で実行
- 中間ドキュメントを生成して正確なセマンティック解析を維持

## セキュリティ考慮事項

該当なし（コード変換のみでセキュリティ上のリスクなし）

## 備考

- 変換を2回適用すると元の形式に戻る（可逆変換）
- 括弧の追加・削除は自動的に最適化される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AbstractInvertLogicalCodeRefactoringProvider.cs | `src/Features/Core/Portable/InvertLogical/` | 論理反転の抽象クラス |

**主要処理フロー**:
1. **17-18行目**: クラスコメントで変換例（`!a || !b` → `!(a && b)`）を説明
2. **31行目**: s_annotationで変換結果を識別
3. **33行目**: GetOperatorText抽象メソッドで演算子テキストを取得
4. **35-80行目**: ComputeRefactoringsAsyncでリファクタリング候補を計算
5. **50-58行目**: 空選択時は最上位の二項式を対象
6. **60-70行目**: 非空選択時は完全選択のみ対象
7. **82-97行目**: InvertLogicalAsyncで2段階の否定を適用
8. **99-111行目**: InvertInnerExpressionAsyncで内部否定
9. **113-137行目**: InvertOuterExpressionAsyncで外部否定

**読解のコツ**: 2段階の否定処理（内部→外部）が実質的にド・モルガンの法則を適用する仕組みを理解する。

### プログラム呼び出し階層図

```
ComputeRefactoringsAsync
    │
    ├─ TryGetRelevantNodeAsync<TBinaryExpressionSyntax>
    │      └─ 対象ノードの取得
    │
    ├─ IsLogicalAndExpression / IsLogicalOrExpression
    │      └─ 演算子種別の判定
    │
    ├─ 最上位二項式の特定
    │      └─ Parent.RawKind == expression.RawKindを辿る
    │
    └─ InvertLogicalAsync
           ├─ InvertInnerExpressionAsync
           │      └─ Generator.Negate (内部否定)
           │             └─ ド・モルガン適用
           └─ InvertOuterExpressionAsync
                  └─ Generator.Negate (外部否定, negateBinary:false)
                         └─ 二重否定の除去
```

### データフロー図

```
[入力]                  [処理]                        [出力]

!a || !b ───▶ InvertInnerExpressionAsync ───▶ a && b (内部否定でド・モルガン適用)
                     │
                     └─ Generator.Negate
                            │
                            ▼
a && b ───▶ WithAdditionalAnnotations ───▶ (a && b) + s_annotation
                     │
                     ▼
(a && b) ───▶ InvertOuterExpressionAsync ───▶ !(a && b) (外部否定)
                     │
                     └─ Generator.Negate (negateBinary:false)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AbstractInvertLogicalCodeRefactoringProvider.cs | `src/Features/Core/Portable/InvertLogical/` | ソース | 論理反転の抽象プロバイダ |
