# API設計意図書

## 概要

本ドキュメントは、AureusERP（Laravel + Filament v4ベースのオープンソースERPシステム）におけるAPI設計の意図・判断根拠・選択理由を記載したものです。本システムは従来のREST APIを限定的に提供し、主要なデータ操作はFilament管理パネル（Livewireベース）を通じて行うアーキテクチャを採用しています。この設計方針に至った業務上の背景と技術的判断を詳細に説明します。

---

## 業務要件との関連（なぜこのAPIが必要か）

### 対応する業務要件

| 業務要件ID | 業務要件名 | APIに求められる要件 |
| --- | --- | --- |
| BR-001 | 管理者認証 | 管理者ユーザーのログイン・ログアウト機能を提供し、不正アクセスを防止する |
| BR-002 | 顧客認証 | 顧客ユーザーの認証・登録機能を提供し、顧客ポータルへのアクセスを制御する |
| BR-003 | ベンダー見積回答 | ベンダーがメール経由で見積依頼（RFQ）に対して承諾・拒否の回答ができる |
| BR-004 | ユーザー招待 | 新規ユーザーを招待し、安全にアカウントを作成できる仕組みを提供する |
| BR-005 | マルチテナント対応 | 複数会社のデータを分離し、ユーザーが所属会社に応じたデータにアクセスできる |
| BR-006 | ロールベース権限管理 | ユーザーの役割に応じたきめ細かいアクセス制御を実現する |

### 業務観点でのAPI要件

#### 想定されるAPIの利用者と利用シーン

本システムのAPI設計は、利用者の特性と利用シーンを慎重に分析した結果に基づいています。

| 利用者種別 | 利用目的 | 利用シーン |
| --- | --- | --- |
| 社内管理者 | ERP機能全般の操作・管理 | 管理パネル（/admin）からのブラウザ操作 |
| 顧客 | 自社関連情報の閲覧・操作 | 顧客ポータル（/）からのブラウザ操作 |
| ベンダー | 見積依頼への回答 | メールに記載された署名付きURLからの操作 |
| 招待されたユーザー | アカウント作成 | 招待メールに記載された署名付きURLからの操作 |

**重要な分析結果**: 本システムの主要な利用シーンは**ブラウザ経由のインタラクティブな操作**であり、外部システムからのAPI連携や、モバイルアプリからのデータアクセスは現時点で想定されていません。

#### 業務上のAPI品質要件

| 項目 | 業務要求 | 根拠 |
| --- | --- | --- |
| 応答時間 | 3秒以内 | 管理パネル操作のレスポンスとして体感できる速度（Livewireの部分更新により、実際の体感はより高速） |
| 可用性 | 99%以上 | 通常の営業時間帯（9:00-18:00）での利用を想定。24時間365日の運用は必須ではない |
| 同時リクエスト数 | 50件/秒 | 中小企業向けERPとして、同時利用者50名程度を想定 |
| セキュリティ | CSRF保護必須 | Webアプリケーションとして一般的なセキュリティ要件を満たす |

### 背景にあると推察する業務課題

コード分析から、以下の業務課題が存在していたと推察されます。

#### 1. 中小企業向けERPの導入・運用コストの課題

**課題の深刻度**: 高

ERPシステムは機能が豊富な反面、導入・カスタマイズ・運用のコストが高く、中小企業にとって導入障壁となっています。特に以下の点が課題と考えられます。

- **カスタマイズコスト**: 従来のERPは個別の要件に対応するために高額なカスタマイズ費用が発生
- **学習コスト**: 複雑なAPIを理解し、フロントエンド開発を行う技術者の確保が困難
- **保守コスト**: REST APIとフロントエンドの両方を保守する必要があり、運用負荷が高い

#### 2. ベンダーとのコミュニケーション効率の課題

**課題の深刻度**: 中

購買プロセスにおいて、見積依頼（RFQ）に対するベンダーからの回答を効率的に収集する必要がありました。

- **従来の方法**: メール・電話・FAXによる手動確認が中心で、回答の追跡・管理が煩雑
- **業務への影響**: 回答遅延による調達プロセスの遅延、見積比較の手間

#### 3. ユーザー招待・オンボーディングの課題

**課題の深刻度**: 中

新規ユーザーのアカウント作成プロセスにおいて、セキュリティと利便性のバランスが求められていました。

- **セキュリティ懸念**: 誰でもアカウントを作成できると不正アクセスのリスクが高まる
- **利便性の要求**: 招待されたユーザーが簡単にアカウントを作成できる仕組みが必要

### 業務課題に対するアプローチ方針

#### 1. Filament + Livewireによるフルスタック管理パネル

**アプローチの核心**: 従来のAPI + フロントエンド分離アーキテクチャではなく、Filament管理パネルによる統合アプローチを採用。

**期待される業務改善効果**:
- **導入コスト削減**: API設計・フロントエンド開発のコストを大幅に削減（推定50-70%減）
- **開発速度向上**: Filamentのリソース定義により、CRUDの90%以上をコード生成で対応
- **保守コスト削減**: 単一コードベースでの保守により、API・フロントエンド間の不整合リスクを排除

#### 2. 署名付きURLによるベンダー連携

**アプローチの核心**: ベンダーにログイン機能を要求せず、署名付きURLでの認証を採用。

**期待される業務改善効果**:
- **ベンダー負担軽減**: アカウント作成・パスワード管理が不要で、ワンクリックで回答可能
- **回答率向上**: 操作の簡易化により、ベンダーからの回答率向上が期待される
- **追跡性確保**: 署名付きURLにより、回答者の特定と有効期限管理が可能

#### 3. 招待ベースのユーザー登録

**アプローチの核心**: オープン登録ではなく、招待制のユーザー登録を採用。

**期待される業務改善効果**:
- **セキュリティ強化**: 管理者が承認したユーザーのみがアカウントを作成可能
- **適切な初期設定**: 招待時にロール・会社を事前設定し、オンボーディングを効率化

### 検討した実現方式（業務観点）

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | REST API + SPAフロントエンド | 柔軟性高、モバイル対応容易 | 開発コスト高、保守複雑 | 低（過剰投資） |
| B | GraphQL + SPAフロントエンド | データ取得効率良、型安全 | 学習コスト高、複雑性増大 | 低（過剰投資） |
| C（採用） | Filament管理パネル + 限定的Web API | 開発効率高、保守容易 | 外部連携制限、モバイル非対応 | **高** |
| D | 従来型MVC + Blade | シンプル、学習容易 | UX劣る、リアルタイム性低 | 中 |

### 選択理由（業務観点）

**案C（Filament管理パネル + 限定的Web API）を採用した理由**:

1. **業務要件との適合性（最重要）**
   - 本システムの主要ユーザーは社内管理者と顧客であり、ブラウザ操作が前提
   - 外部システム連携やモバイルアプリの要件は現時点で明確に存在しない
   - したがって、包括的なREST APIを提供する必要性が低い

2. **手動連携を継続する選択肢との比較（不採用理由）**
   - ベンダー見積回答：メール・電話による手動追跡は非効率で、回答漏れのリスクがある
   - ユーザー招待：管理者による手動アカウント作成は、パスワード共有などのセキュリティリスクを伴う
   - **結論**: 署名付きURLによる自動化が業務効率とセキュリティの両面で優位

3. **バッチ連携で実現する選択肢との比較（不採用理由）**
   - ベンダー見積回答：リアルタイム性が求められる業務であり、バッチ処理では回答の即時反映ができない
   - ユーザー招待：招待直後のアカウント作成要求に応えられない
   - **結論**: インタラクティブな操作が必要な業務には、同期的なWeb APIが適切

4. **ROI（投資対効果）の観点**
   - Filament + Livewireにより、フロントエンド開発工数を大幅削減
   - 限定的なWeb APIのみで、必要十分な外部連携機能を実現
   - 将来的な拡張（REST API追加）は、Laravelの標準機能で対応可能

---

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| API形式 | REST / GraphQL / gRPC | **Livewire + 限定的REST** | Filament統合による開発効率、ブラウザ操作が主要ユースケース |
| バージョニング | URLパス / ヘッダー / クエリ | **バージョニングなし** | 内部利用が中心で、外部APIとしてのバージョン管理は不要 |
| 認証方式 | JWT / OAuth2 / API Key / Session | **セッション認証 + 署名付きURL** | Webアプリケーションとしてのセキュリティ要件、ゲストユーザー対応 |
| エラーレスポンス | RFC 7807 / 独自形式 / HTTPステータスのみ | **Laravel標準形式（独自形式）** | Filament・Livewireとの統合、フレームワーク標準に準拠 |

---

## 設計判断と根拠

### 1. API形式の選択

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| REST | リソース指向のHTTP API | シンプル、広く普及、キャッシュ容易 | オーバーフェッチ/アンダーフェッチ |
| GraphQL | クエリ言語ベースのAPI | 柔軟なデータ取得、型安全 | 学習コスト、キャッシュ複雑 |
| gRPC | Protocol BuffersベースのRPC | 高パフォーマンス、型安全 | ブラウザ対応限定、デバッグ困難 |
| Livewire | Laravelの全二重通信フレームワーク | サーバーサイドロジック統合、リアルタイム更新 | SPA的なUX制限、外部連携不可 |

#### 採用した方針

**Livewire（Filament経由）を主軸とし、限定的なWebエンドポイントで補完**

本システムは、Filament v4の管理パネル機能を全面的に採用しています。Filamentは内部でLivewireを使用し、すべてのCRUD操作を`POST /livewire/update`エンドポイント経由で処理します。

**コード根拠**:
```php
// app/Providers/Filament/AdminPanelProvider.php
return $panel
    ->default()
    ->id('admin')
    ->path('admin')
    ->login()
    // ...
    ->plugins([
        FilamentShieldPlugin::make(),
        PluginManager::make(),
    ])
```

#### 選択理由

1. **開発効率の最大化**: Filamentのリソース定義により、CRUDの実装コードを大幅削減。UserResourceの例では、約500行のPHPコードで、フォーム・テーブル・詳細画面・権限制御をすべて実現しています。

2. **一貫したUX**: Livewireの部分更新により、SPAに近い操作感を実現しながら、サーバーサイドレンダリングの利点（SEO、初期読み込み速度）も維持。

3. **プラグインアーキテクチャとの親和性**: `PluginManager`によるモジュラー設計と、Filamentのリソース登録機能がシームレスに統合。

4. **業務要件への適合**: 主要ユーザーがブラウザ経由で操作する前提であり、外部API連携の優先度は低い。

#### 不採用とした選択肢の理由

- **REST API（包括的）**: 開発・保守コストが増大し、現時点の業務要件に対して過剰投資となる。将来必要になった場合は、Laravel Sanctum/Passportで追加可能。
- **GraphQL**: 学習コストが高く、チーム全体での習熟が必要。現時点でのメリットが限定的。
- **gRPC**: ブラウザからの直接利用が困難で、本システムの主要ユースケースに適さない。

---

### 2. RESTful設計レベル

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| Level 0（POE） | 単一エンドポイント、POSTのみ | シンプル | RESTの利点なし |
| Level 1（Resources） | リソース指向だがPOSTのみ | リソース概念あり | HTTPメソッド未活用 |
| Level 2（HTTP Verbs） | リソース + HTTPメソッド活用 | 直感的、標準的 | HATEOASなし |
| Level 3（HATEOAS） | ハイパーメディア駆動 | 自己文書化、発見可能 | 実装複雑、オーバーヘッド |

#### 採用した方針

**Level 0〜1相当（Livewire経由のRPCライク）+ Level 2（限定的Webエンドポイント）**

Filament/Livewireのエンドポイントは、`POST /livewire/update`という単一エンドポイントでRPCライクな通信を行います。これはRichardson Maturity ModelではLevel 0に相当しますが、これはLivewireの設計思想に基づくものです。

一方、限定的に提供されるWebエンドポイント（見積回答、招待承諾など）は、HTTPメソッドとURL設計の観点でLevel 2に近い設計となっています。

#### エンドポイント設計パターン

| パターン | 採用方針 | 選択理由 |
| --- | --- | --- |
| リソース命名 | 複数形・小文字（purchases, invitations） | Laravelの規約に準拠し、一貫性を確保 |
| ネスト深度 | 最大2階層（/purchase/{order}/{action}） | URLの可読性と管理性のバランス |
| アクション表現 | パスパラメータ（/accept, /decline） | 動詞的なアクションを明示的に表現 |

**コード根拠**:
```php
// plugins/webkul/purchases/routes/web.php
Route::middleware('signed')
    ->get('purchase/{order}/{action}', RespondQuotation::class)
    ->name('purchases.quotations.respond');
```

#### 選択理由

Livewireの採用により、従来のRESTful設計の議論（HTTPメソッドの使い分け、ステータスコードの活用）は限定的になりました。これは意図的な設計判断であり、以下の理由によります。

1. **Filamentとの統合優先**: Filamentの機能をフル活用するため、その内部通信方式（Livewire）をそのまま採用。
2. **開発効率**: RESTfulな設計に時間を費やすよりも、業務機能の実装に注力。
3. **実用性**: 外部からのAPI利用を前提としないため、RESTの標準化メリットが限定的。

#### 不採用とした選択肢の理由

- **Level 3（HATEOAS）**: 実装コストが高く、外部API連携を想定しない本システムでは過剰な設計。

---

### 3. バージョニング戦略

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| URLパス | `/v1/users` | 明示的、キャッシュ容易 | URLが冗長 |
| ヘッダー | `Accept-Version: v1` | URLクリーン | キャッシュ複雑、見えにくい |
| クエリパラメータ | `?version=1` | 柔軟 | キャッシュ複雑、非標準的 |
| バージョニングなし | バージョン管理を行わない | シンプル | 破壊的変更の管理困難 |

#### 採用した方針

**バージョニングなし**

本システムでは、APIのバージョニングを明示的に行っていません。

#### バージョン管理ルール

| 項目 | ルール | 理由 |
| --- | --- | --- |
| 現行バージョン | なし（暗黙的にv1相当） | 外部公開APIがないため、バージョン概念が不要 |
| 非推奨化ポリシー | 未定義 | 内部利用のみで、段階的な移行管理が不要 |
| 破壊的変更の扱い | アプリケーション全体のバージョンで管理 | composer.jsonのバージョンで管理 |

#### 選択理由

1. **内部利用が前提**: 外部開発者がAPIを利用する想定がなく、バージョン互換性の維持コストは不要。
2. **Filament/Livewire統合**: Livewireの通信プロトコルはFilamentのバージョンに依存し、個別のAPIバージョニングは不適切。
3. **シンプルさの維持**: 不要な複雑性を避け、保守性を向上。

#### 不採用とした選択肢の理由

- **URLパスバージョニング**: 外部APIを公開する場合には有効だが、現時点では不要。
- **ヘッダーバージョニング**: 同上。将来API公開時に検討。

---

### 4. 認証・認可方式

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| JWT | トークンベース認証 | ステートレス、スケーラブル | トークン失効が困難 |
| OAuth 2.0 | 認可フレームワーク | 標準的、柔軟 | 実装複雑 |
| API Key | 静的キーによる認証 | シンプル | セキュリティ弱い、ユーザー識別困難 |
| Session + Cookie | サーバーサイドセッション | 実装シンプル、失効容易 | スケーラビリティ課題 |
| 署名付きURL | 時限付き署名URLによる認証 | ゲストユーザー対応、共有容易 | 用途限定 |

#### 採用した方針

**セッション認証（Web/Customer Guard）+ 署名付きURL**

本システムでは、以下の認証方式を組み合わせて使用しています。

1. **管理者向け**: セッション認証（Web Guard）
2. **顧客向け**: セッション認証（Customer Guard）
3. **ゲストユーザー向け**: 署名付きURL（Signed URL）

**コード根拠**:
```php
// config/auth.php
'guards' => [
    'web' => [
        'driver'   => 'session',
        'provider' => 'users',
    ],
    'customer' => [
        'driver'   => 'session',
        'provider' => 'customers',
    ],
],
```

```php
// plugins/webkul/purchases/routes/web.php
Route::middleware('signed')
    ->get('purchase/{order}/{action}', RespondQuotation::class)
```

#### 認証フロー

**管理者認証フロー**:
1. `/admin/login`にアクセス
2. メールアドレス・パスワードを入力
3. セッションが作成され、Cookieに保存
4. 以降のリクエストでセッションCookieを送信

**署名付きURL認証フロー**:
1. システムが署名付きURLを生成（メール等で送信）
2. ユーザーがURLにアクセス
3. `signed`ミドルウェアが署名と有効期限を検証
4. 検証成功時、対応するアクションを実行

#### 選択理由

1. **Webアプリケーションに最適**: セッション認証はブラウザベースのアプリケーションに最も適した認証方式。Laravel/Filamentの標準機能をそのまま活用可能。

2. **セキュリティ要件の充足**:
   - CSRF保護との統合が容易
   - セッション失効による即時アクセス停止が可能
   - パスワード変更時の全セッション無効化が容易

3. **署名付きURLの有効性**: ベンダーや招待ユーザーに対して、アカウント作成を要求せずに限定的な操作を許可できる。

**コード根拠（ミドルウェア構成）**:
```php
// app/Providers/Filament/AdminPanelProvider.php
->middleware([
    EncryptCookies::class,
    StartSession::class,
    AuthenticateSession::class,
    VerifyCsrfToken::class,
    // ...
])
->authMiddleware([
    Authenticate::class,
])
```

#### 不採用とした選択肢の理由

- **JWT**: ステートレス認証のメリットは、分散システムやモバイルアプリで顕著。本システムは単一サーバー構成が前提であり、セッション管理の複雑さが低い。また、トークンの即時失効が困難な点がセキュリティ上の懸念。
- **OAuth 2.0**: 外部サービス連携やサードパーティアプリケーション認可には有効だが、内部利用が中心の本システムには過剰。
- **API Key**: ユーザー識別が困難で、セキュリティ要件を満たさない。

---

### 5. エラーレスポンス設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| RFC 7807 (Problem Details) | 標準化されたエラー形式 | 標準的、ツール対応 | やや冗長 |
| 独自形式 | プロジェクト固有の形式 | 柔軟、シンプル化可能 | 非標準、ドキュメント必要 |
| HTTPステータスのみ | ボディなし、ステータスコードのみ | 最シンプル | デバッグ困難 |

#### 採用した方針

**Laravel標準形式（独自形式）**

本システムでは、Laravelのバリデーションエラー形式をそのまま採用しています。

#### エラーレスポンス形式

```json
{
  "message": "These credentials do not match our records.",
  "errors": {
    "email": ["These credentials do not match our records."]
  }
}
```

#### HTTPステータスコード使用方針

| 状況 | ステータスコード | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| バリデーションエラー | 422 | 400 | Laravel標準に準拠、フィールド単位のエラー表示に適切 |
| 認証エラー | 401 | 403 | 未認証状態を明確に示す |
| 権限エラー | 403 | 401 | 認証済みだが権限不足を明確に示す |
| CSRFトークン不一致 | 419 | 403 | Laravel固有のステータスコードで、CSRF問題を明確に識別 |
| リソース未検出 | 404 | 400 | 標準的なHTTPセマンティクス |

#### 選択理由

1. **フレームワーク統合**: Laravel・Filamentの標準エラーハンドリングをそのまま利用することで、一貫したエラー処理を実現。

2. **Livewireとの親和性**: Livewireはエラーレスポンスを自動的に処理し、UIにフィードバックを表示。カスタム形式は統合を複雑化する。

3. **開発者体験**: Laravel開発者にとって馴染みのある形式で、デバッグ・トラブルシューティングが容易。

**コード根拠**:
```php
// 認証エラーレスポンスの例（API設計書より）
{
  "message": "These credentials do not match our records.",
  "errors": {
    "email": ["These credentials do not match our records."]
  }
}
```

---

### 6. ページネーション設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| オフセット | `?offset=20&limit=10` | シンプル、ページ番号対応 | 大量データで性能劣化 |
| カーソル | `?cursor=xxx&limit=10` | 大量データで高速、一貫性 | ページ番号不可、実装複雑 |
| キーセット | `?after_id=100&limit=10` | 高速、シンプル | ソート順固定 |
| ページ番号 | `?page=3&per_page=10` | 直感的 | 大量データで性能劣化 |

#### 採用した方針

**Filament標準のページ番号ベースページネーション**

Filamentテーブルコンポーネントが提供するページネーション機能をそのまま利用しています。

#### ページネーションパラメータ

| パラメータ | 説明 | デフォルト値 | 選択理由 |
| --- | --- | --- | --- |
| page | 現在のページ番号 | 1 | ユーザーにとって直感的 |
| per_page | 1ページあたりの件数 | 15（Filament標準） | 画面に収まる適切な件数 |

#### 選択理由

1. **ユーザー体験**: ページ番号ベースのページネーションは、ユーザーにとって最も直感的で、「3ページ目を見る」といった操作が容易。

2. **データ規模**: 中小企業向けERPとして、各テーブルの行数は数千〜数万件程度を想定。この規模ではオフセット方式のパフォーマンス問題は顕在化しにくい。

3. **Filament統合**: Filamentの標準機能を利用することで、実装コストを最小化。

#### 不採用とした選択肢の理由

- **カーソルベース**: 大規模データセットでは有効だが、実装複雑性が増し、ページ番号表示ができない。現時点のデータ規模では過剰。

---

### 7. レート制限・スロットリング

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 固定ウィンドウ | 時間枠ごとにリセット | シンプル | バースト許容 |
| スライディングウィンドウ | 移動平均で制限 | 滑らか | 実装複雑 |
| トークンバケット | トークン補充方式 | バースト対応可 | 実装複雑 |
| 制限なし | レート制限なし | シンプル | DoSリスク |

#### 採用した方針

**Laravel標準のレート制限（Throttle）**

LaravelのThrottleミドルウェアによる固定ウィンドウ方式を採用していると推測されます（明示的な設定は確認できず）。

#### 制限ルール

| エンドポイント種別 | 制限 | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| ログインエンドポイント | 5回/分（推測） | 10回/分 | ブルートフォース攻撃防止 |
| 一般Livewireリクエスト | 制限なし〜緩い制限 | 厳格な制限 | インタラクティブな操作を阻害しない |
| 署名付きURL | 制限なし | レート制限 | 一度限りの使用であり、制限不要 |

#### 選択理由

1. **内部利用中心**: 外部公開APIではないため、DoS攻撃のリスクは限定的。
2. **ユーザー体験優先**: 過度なレート制限は、正常な業務操作を阻害する。
3. **ログイン保護**: セキュリティ上重要なエンドポイントには適切な制限を適用。

---

### 8. リクエスト/レスポンス形式

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| JSON | テキストベース | 可読性高い、広く普及 | サイズ大きい |
| Protocol Buffers | バイナリ形式 | コンパクト、高速 | 可読性低い、スキーマ必須 |
| MessagePack | バイナリJSON | JSONより軽量 | 普及度低い |

#### 採用した方針

**JSON**

すべてのリクエスト・レスポンスでJSON形式を採用しています。

#### 命名規則

| 対象 | 採用規則 | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| フィールド名 | snake_case | camelCase | Laravel/PHP標準に準拠、データベースカラムとの一貫性 |
| 日時形式 | ISO 8601 | UNIXタイムスタンプ | 可読性、タイムゾーン対応 |
| 列挙値 | 小文字（accept/decline） | 大文字 | URLパラメータとしての可読性 |

**コード根拠**:
```php
// plugins/webkul/purchases/src/Livewire/RespondQuotation.php
if ($this->action === 'accept') {
    $order->update([
        'mail_reception_confirmed' => true,
    ]);
}
```

#### 選択理由

1. **可読性**: JSONはテキストベースで、デバッグやログ分析が容易。
2. **広範なサポート**: ブラウザ、JavaScript、PHPすべてで標準的にサポート。
3. **Laravelとの統合**: Laravelのレスポンスヘルパーや、Livewireの通信がJSONを前提として設計されている。

---

## 将来の拡張性考慮

### 想定される変更

1. **外部REST API追加**
   - **想定シナリオ**: 外部システム（基幹システム、ECサイト等）との連携要件が発生
   - **対応方針**: Laravel Sanctum/Passportを導入し、APIトークン認証を追加。既存のEloquentモデルを活用し、APIResourceクラスでレスポンス形式を定義。

2. **モバイルアプリ対応**
   - **想定シナリオ**: 営業担当者向けのモバイルアプリから、顧客情報や見積を参照したい
   - **対応方針**: JWT認証を導入し、モバイルネイティブアプリからのアクセスに対応。必要なエンドポイントのみを段階的に追加。

3. **Webhooks実装**
   - **想定シナリオ**: 注文確定時に外部システムへ通知したい
   - **対応方針**: Laravelのイベント・リスナー機構を活用し、Webhooks送信機能を追加。

### 後方互換性

**基本方針**: 現時点では外部APIを公開していないため、後方互換性の厳格な管理は不要。将来API公開時には以下を検討。

1. **セマンティックバージョニング**: メジャーバージョンアップ時のみ破壊的変更を許容
2. **非推奨化期間**: 最低6ヶ月の非推奨期間を設け、移行猶予を提供
3. **変更履歴の公開**: CHANGELOG形式での変更内容の明示

---

## 備考

### アーキテクチャ上の特記事項

1. **Filament v4採用**: 本システムはFilament v4（filament/filament: ^4.1）を採用しており、Filament v3との互換性はありません。

2. **プラグインアーキテクチャ**: 機能は`plugins/webkul/`配下にモジュール化されており、`PluginManager`を通じてFilamentパネルに登録されます。

3. **マルチテナント対応**: 会社（Company）単位でのデータ分離を実現しており、ユーザーは複数の会社に所属可能です。

### セキュリティ考慮事項

1. **CSRF保護**: すべてのPOSTリクエストにCSRFトークンが必須（`VerifyCsrfToken`ミドルウェア）
2. **セッション暗号化**: Cookie内容は暗号化されて保存（`EncryptCookies`ミドルウェア）
3. **パスワードポリシー**: 最小8文字、確認入力必須（Laravel Password Rule）
4. **署名付きURL**: 有効期限付きで、改ざん検知機能を持つ

### 推奨事項

1. **HTTPS必須**: 本番環境では必ずHTTPSを使用し、セッションCookieのSecure属性を有効化
2. **セッション有効期限**: 業務要件に応じて適切な有効期限を設定（デフォルト120分）
3. **権限設定**: Filament Shieldによる適切なロール・権限設定を実施
4. **定期更新**: Laravel、Filament、依存パッケージのセキュリティアップデートを定期的に適用
