# アーキテクチャ設計意図書

## 概要

本ドキュメントは、AureusERP（Laravel/FilamentベースのオープンソースERPソリューション）におけるアーキテクチャ設計の意図・選定理由を記述します。本システムは中小企業から大規模企業向けのERPソリューションであり、モジュラープラグインアーキテクチャを採用しています。アーキテクチャ選定にあたり、どのような業務課題に対応するためにこの構成が必要とされたのか、また他の選択肢と比較してなぜこの構成が最適であるかを詳述します。

---

## 業務要件との関連（なぜこのアーキテクチャが必要か）

### 対応する業務要件
| 業務要件ID | 業務要件名 | アーキテクチャに求められる要件 |
| --- | --- | --- |
| BR-001 | 複数業務モジュールの統合管理 | 販売・購買・在庫・会計・人事など多様な業務ドメインを単一システムで管理できる拡張可能なモジュール構造 |
| BR-002 | 段階的な機能導入 | 企業の成長に合わせて必要なモジュールのみをインストール・有効化できるプラグイン機構 |
| BR-003 | マルチカンパニー・マルチチーム対応 | 複数の会社・チームを単一インスタンスで管理できるマルチテナント設計 |
| BR-004 | 業務ルールの柔軟な変更対応 | 業務プロセス（販売フロー、承認フロー等）の変更に迅速に対応できる疎結合な設計 |
| BR-005 | 細粒度のアクセス制御 | リソース・操作単位での権限管理により、部門・役職に応じたアクセス制御を実現 |

### 業務観点でのアーキテクチャ要件

#### 業務の特性とシステムへの要求
| 業務特性 | システムへの要求 | 根拠 |
| --- | --- | --- |
| ERP業務の多様性 | モジュール単位での機能分離・独立性 | 販売、購買、在庫、会計、人事など異なる業務ドメインを統合的に管理する必要があり、各ドメインは独立して開発・保守できることが求められる |
| 導入企業規模の多様性 | スケーラブルかつ軽量な構成の両立 | 小規模企業から大規模企業まで対応するため、必要なモジュールのみを有効化できる柔軟性が必要 |
| 業務プロセスの企業差 | 設定・カスタマイズの容易性 | 各企業固有の業務フローに対応するため、コードを変更せずに設定で挙動を変更できる必要がある（Settings機構） |
| 組織構造の複雑性 | チーム・会社単位のデータ分離 | 複数部門・複数法人を持つ企業向けに、データの分離とアクセス制御が必要 |
| 継続的な機能拡張要求 | 後方互換性を保った拡張機構 | OSSとして継続的に機能追加されるため、既存機能を壊さずに新機能を追加できる設計が必要 |

#### 開発・運用体制との整合性
| 体制の特性 | アーキテクチャへの要求 | 根拠 |
| --- | --- | --- |
| オープンソースコミュニティ | 標準的な技術スタック・明確なコード規約 | 多数の開発者が参加するため、学習コストが低く標準的な技術（Laravel/Filament）を採用し、PSR-12準拠のコード規約を適用 |
| プラグイン開発者の分散 | 明確なプラグインインターフェース | サードパーティ開発者がプラグインを開発できるよう、Package/PackageServiceProviderという明確な基盤クラスを提供 |
| 長期運用（5年以上） | 保守性・アップグレード容易性 | Laravel LTSに追従し、レイヤー分離により特定層の変更が他層に影響しにくい設計を採用 |
| 多言語・多通貨対応の運用 | 国際化対応の基盤 | 翻訳ファイル機構（hasTranslations）、通貨テーブル設計により、グローバル展開に対応 |

### 検討したアーキテクチャ案（業務観点）

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A: 最小構成（モノリス、シンプルな構造） | 単一アプリケーションとして全機能を実装 | 低コスト、学習コスト低、デプロイ容易 | 機能追加時の影響範囲大、不要機能の除去困難 | 低：ERP業務の多様性に対応困難 |
| B: 標準構成（レイヤードアーキテクチャのみ） | 3層または4層の階層構造で責務分離 | バランス良い、理解しやすい | ドメイン間の境界が曖昧、大規模化時に複雑化 | 中：保守性は確保できるが拡張性に制限 |
| C（採用）: モジュラープラグイン + レイヤード | プラグイン単位でドメイン分離＋各プラグイン内でレイヤー構造 | 業務ドメインの独立性、段階的導入、拡張容易 | 初期設計コスト高、プラグイン間連携の複雑さ | 高：ERP業務の多様性と拡張性を両立 |
| D: マイクロサービス | 各ドメインを独立したサービスとして分離 | 完全な独立性、技術選択の自由度 | 運用複雑、分散トランザクション、中小企業には過剰 | 低：導入・運用コストがターゲット企業に不適合 |

### 選択理由（業務観点）

1. **業務要件との適合性**: ERPは販売・購買・在庫・会計・人事など多様な業務ドメインを統合的に管理するシステムであり、各ドメインを独立したプラグインとして分離することで、業務の境界を明確にし、ドメイン知識を持つ開発者が各プラグインを独立して開発・保守できる構造を実現している。

2. **業務ルール変更への対応力**: Spatie Laravel Settingsを活用した設定機構により、業務ルール（見積書の有効期限、承認フロー等）をコード変更なしに設定画面から変更可能。各プラグインが独立しているため、特定業務の変更が他業務に影響しにくい。

3. **開発・運用体制との整合性**: Laravel/Filamentという広く普及したフレームワークを採用することで、開発者の学習コストを抑え、コミュニティからの貢献を促進。Spatie Laravel Package Toolsを拡張した独自の`Package`/`PackageServiceProvider`クラスにより、プラグイン開発の標準パターンを提供。

4. **将来の業務拡大への対応**: プラグインの依存関係管理（`hasDependencies`）、インストール/アンインストールコマンドの標準化により、企業の成長に合わせて必要なモジュールを段階的に追加できる。新たな業務領域（例：製造管理）もプラグインとして追加可能。

5. **費用対効果（ROI）**: マイクロサービスほどの運用コストをかけずに、モジュール単位での独立性を確保。単一のLaravelアプリケーションとしてデプロイできるため、中小企業でも導入・運用が容易。OSSとして公開することでコミュニティの力を活用し、継続的な機能拡張を低コストで実現。

---

## システム特性・前提条件

| 項目 | 内容 | 補足 |
| --- | --- | --- |
| サービス種別 | BtoB SaaS / オンプレミス対応の業務システム | MITライセンスのOSSとして公開されており、SaaS提供とオンプレミス導入の両方に対応 |
| トランザクション特性 | 中頻度、読み取り・書き込み混在 | 通常のERP業務では高頻度トランザクションは発生しないが、請求書発行時など一部で複数テーブルへの同時書き込みが発生 |
| 可用性要件 | 業務時間内の稼働を保証（99.5%程度を想定） | クリティカルなリアルタイム処理はなく、ジョブキューによる非同期処理で対応可能 |
| チーム規模 | 小規模（5人以下）〜中規模（20人程度）のチームで保守可能 | プラグイン単位で責務分離されているため、各プラグインは2-3人で保守可能な規模 |
| 想定成長 | 機能拡張（新規プラグイン追加）とスケールアウト（ユーザー数増加）の両方 | プラグインアーキテクチャにより機能拡張に対応、標準的なLaravelスケーリング手法でスケールアウトに対応 |

> **この前提があるから、以下の選択をした**という流れで記載する。

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| アーキテクチャスタイル | レイヤード / クリーン / ヘキサゴナル / DDD / モノリス / マイクロサービス | モジュラープラグイン + レイヤード | 業務ドメイン単位での分離と段階的導入を両立するため |
| レイヤー分割 | 3層 / 4層 / オニオン型 | 4層（Presentation / Application / Domain / Infrastructure） | 各層の責務を明確にしつつ、Laravelエコシステムとの親和性を確保 |
| モジュール分割 | 機能別 / ドメイン別 / レイヤー別 | ドメイン別（業務プラグイン単位） | ERP業務の独立性を確保し、プラグインとして管理可能にするため |
| DB選定 | PostgreSQL / MySQL / MongoDB / DynamoDB | SQLite(開発) / MySQL・PostgreSQL(本番) | Laravelの標準サポートと幅広い環境での動作を優先 |
| キャッシュ選定 | Redis / Memcached / なし | Database（デフォルト）/ Redis（オプション） | 追加インフラなしで動作し、スケール時はRedisに切替可能 |
| MQ選定 | Kafka / RabbitMQ / SQS / なし | Database Queue（デフォルト）/ Redis・SQS（オプション） | 非同期処理のニーズに対応しつつ、最小構成でも動作可能に |

---

## 設計判断と根拠

### 1. アーキテクチャスタイル選定

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| レイヤードアーキテクチャ | 階層構造で責務を分離 | シンプル、学習コスト低 | ドメイン境界が曖昧、機能追加時の影響範囲大 |
| クリーンアーキテクチャ | 依存性逆転を適用 | テスタブル、変更に強い | 学習コスト高、冗長になりがち、Laravelとの親和性低い |
| ヘキサゴナルアーキテクチャ | ポート&アダプター | 外部依存を抽象化 | 設計の複雑化、ERPには過剰な場合が多い |
| DDD（ドメイン駆動設計） | ドメインモデル中心 | ビジネスロジック表現力 | ドメイン分析に時間がかかる、OSSコミュニティでの共有が難しい |
| モノリシック | 単一アプリケーション | シンプル、デプロイ容易 | スケール制限、機能追加時の影響範囲大 |
| マイクロサービス | サービス分割 | 独立デプロイ、スケール | 運用複雑、分散トランザクション、中小企業には過剰 |
| **モジュラープラグイン + レイヤード（採用）** | プラグイン単位でドメイン分離＋レイヤー構造 | 業務単位の独立性、段階的導入、運用容易 | プラグイン間連携の設計が必要 |

#### 採用した方針
**モジュラープラグインアーキテクチャ + レイヤードアーキテクチャ**

Spatie Laravel Package Toolsを拡張した独自の`Package`クラス・`PackageServiceProvider`クラスを基盤として、各業務ドメイン（販売、購買、在庫等）をプラグインとして実装。各プラグイン内はFilament Admin Panel（Presentation層）、Manager/Facadeクラス（Application層）、Eloquent Models（Domain層）、Migrations/Settings（Infrastructure層）というレイヤー構造を持つ。

#### 選択理由

1. **業務ドメインの独立性確保**: ERPは複数の業務ドメインを統合するシステムであり、各ドメインを独立したプラグインとして分離することで、ドメイン知識を持つ開発者が各プラグインを独立して開発・保守できる。`plugins/webkul/sales/`、`plugins/webkul/purchases/`、`plugins/webkul/inventories/`のように明確に分離されている。

2. **段階的導入の実現**: `bootstrap/plugins.php`で有効なプラグインを管理し、`Package::hasDependencies()`で依存関係を定義することで、企業のニーズに応じて必要なモジュールのみをインストール・有効化できる。

3. **Laravelエコシステムとの親和性**: Laravelの標準的なServiceProvider機構を活用することで、Laravelに精通した開発者がすぐに開発に参加できる。

4. **現行踏襲（単純なモノリス）を不採用とした理由**: ERP業務の多様性に対応するには、ドメイン間の境界を明確にする必要がある。単純なモノリスでは機能追加時の影響範囲が大きく、特定業務の変更が他業務に波及するリスクがある。

5. **マイクロサービスを不採用とした理由**: 中小企業向けERPとしては運用コストが過剰。分散トランザクションの複雑さ（販売→請求→在庫の連携等）を単一DBトランザクションで解決できるモジュラーモノリスの方が適切。

#### 不採用とした選択肢の理由
- **単純なモノリス**: 機能追加時の影響範囲が大きく、不要な機能の除去が困難。ERPの多様な業務ドメインを整理して管理できない。
- **マイクロサービス**: 中小企業向けには運用コストが過剰。分散トランザクション管理の複雑さがERP業務（販売→請求→在庫の連携）に適さない。
- **クリーン/ヘキサゴナルアーキテクチャ**: Laravelのエコシステム（Eloquent ORM、Filament Admin Panel）との親和性が低く、学習コストが高い。

---

### 2. レイヤー分割

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 3層（MVC） | Presentation / Business / Data | シンプル、標準的 | ビジネスロジックが肥大化しやすい |
| 4層 | Presentation / Application / Domain / Infrastructure | 責務明確、テスタブル | 学習コスト |
| オニオン型 | 中心にDomain、外側に依存 | 依存性逆転 | 設計の複雑化、Laravelとの親和性低い |

#### 採用した方針
**4層構造（Presentation / Application / Domain / Infrastructure）**

各プラグイン内で以下のレイヤー構造を採用している。

#### レイヤー構成の詳細
| レイヤー | 責務 | 主なコンポーネント | 選択理由 |
| --- | --- | --- | --- |
| Presentation | ユーザーインターフェース、画面表示 | Filament Panels, Livewire Components, Blade Views | Filament Admin Panelを活用することで、一貫したUI/UXと高速な画面開発を実現 |
| Application | ビジネスロジック調整、ユースケース実行 | ServiceProviders, Facades, Managers（SaleManager等） | 複雑なビジネスロジックをManagerクラスに集約し、コントローラーの肥大化を防止 |
| Domain | ビジネスルール、エンティティ | Models, Enums, Policies | Eloquent ORMを活用しつつ、ビジネスルールをEnumとPolicyで表現 |
| Infrastructure | データ永続化、外部サービス連携 | Migrations, Seeders, Settings, Mail | Laravelの標準機構を活用し、データベース操作・メール送信を抽象化 |

#### 選択理由

1. **責務の明確化**: 4層に分離することで、UI変更はPresentation層のみ、ビジネスロジック変更はApplication/Domain層のみに影響範囲を限定できる。

2. **テスタビリティの確保**: Application層のManager（`SaleManager`等）をFacade経由で呼び出す設計により、モック化が容易でユニットテストが書きやすい。

3. **3層（MVC）を不採用とした理由**: ERPのビジネスロジックは複雑（販売注文の確定処理、請求書発行、在庫引当等）であり、コントローラーに直接記述すると肥大化する。Application層（Manager）を設けることでロジックを整理。

4. **オニオン型を不採用とした理由**: Laravelの標準機構（Eloquent ORM、ServiceProvider）との親和性が低く、学習コストが高い。OSSコミュニティでの開発効率を優先。

#### 不採用とした選択肢の理由
- **3層（MVC）**: ビジネスロジックがコントローラーに集中し、肥大化のリスクがある。ERPの複雑な業務ロジックを整理するには不十分。
- **オニオン型**: 依存性逆転は理想的だが、Laravel/Eloquentとの親和性が低く、学習コストが高い。

---

### 3. モジュール分割

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 機能別 | 機能ごとにモジュール化 | 直感的、見通しが良い | 機能間の重複、ドメイン知識の分散 |
| ドメイン別 | ビジネスドメインで分割 | 凝集度高い | ドメイン分析が必要 |
| レイヤー別 | 技術レイヤーで分割 | 技術ごとに整理 | 機能が分散、変更時に複数層への修正が必要 |
| フィーチャー別 | ユーザー機能ごとに縦割り | 独立性高い | 共通化しにくい |

#### 採用した方針
**ドメイン別（業務プラグイン単位）**

ERP業務ドメインに対応するプラグインとして分割。24のプラグインが存在する。

#### モジュール構成の詳細
| モジュール | 責務 | 依存関係 | 選択理由 |
| --- | --- | --- | --- |
| support（コア） | 共通機能・基盤サービス提供 | なし（全プラグインから依存される） | プラグイン機構、共通モデル（Currency, Country等）、パーミッション管理を集約 |
| security（コア） | 認証・認可・ユーザー管理 | support | ユーザー、チーム、招待機能を分離し、認証関連を一元管理 |
| sales | 販売管理・見積・受注 | invoices, payments | 販売業務の独立性を確保しつつ、請求・支払との連携を明確化 |
| purchases | 購買管理・発注 | invoices, payments | 購買業務の独立性を確保しつつ、請求・支払との連携を明確化 |
| inventories | 在庫管理・倉庫管理 | products | 在庫業務と製品マスタの関係を明確化 |
| invoices | 請求書管理・会計連携 | accounts, payments | 請求書発行と会計連携を集約 |
| employees | 従業員管理・HR | なし | 人事業務を独立したドメインとして管理 |
| projects | プロジェクト管理 | なし | プロジェクト管理を独立したドメインとして管理 |

#### 選択理由

1. **ドメイン知識の集約**: 販売業務に関する知識（見積、受注、出荷指示）は`sales`プラグインに集約され、販売担当の開発者がそのプラグインのみを理解すれば開発できる。

2. **依存関係の明示**: `Package::hasDependencies()`により、プラグイン間の依存関係を明示。例：`sales`は`invoices`と`payments`に依存することが`SaleServiceProvider`で宣言されている。

3. **機能別を不採用とした理由**: ERP業務は複数の機能が密接に関連しており（販売注文→請求書発行→入金管理）、機能別に分割するとドメイン知識が分散する。

4. **レイヤー別を不採用とした理由**: 1つの業務変更（例：販売注文のステータス追加）で複数のディレクトリを修正する必要があり、変更の影響範囲が見えにくい。

#### 不採用とした選択肢の理由
- **機能別**: ERPの業務は機能横断的であり、機能別分割ではドメイン知識が分散する。
- **レイヤー別**: 1つの業務変更で複数ディレクトリの修正が必要になり、変更の影響範囲が見えにくい。

---

### 4. データベース選定

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| PostgreSQL | 高機能RDB | 拡張性、JSON対応、信頼性 | 設定複雑 |
| MySQL | 軽量RDB | 高速、普及率高い | 機能がやや限定的 |
| SQLite | ファイルベースRDB | セットアップ不要、開発向き | 本番環境では非推奨 |
| MongoDB | ドキュメントDB | スキーマ柔軟、スケール | トランザクション制限、ERP向きでない |
| DynamoDB | マネージドNoSQL | スケーラブル、サーバーレス | 柔軟性制限、コスト |

#### 採用した方針
**SQLite（開発）/ MySQL・MariaDB・PostgreSQL・SQL Server（本番）の選択制**

`config/database.php`で複数のRDBMSに対応し、環境変数（`DB_CONNECTION`）で切り替え可能。

#### データベース構成
| 用途 | データベース | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| 開発環境 | SQLite | MySQL | セットアップ不要で開発開始できる。`touch database/database.sqlite`のみで準備完了 |
| 本番環境 | MySQL（推奨）/ PostgreSQL / MariaDB / SQL Server | MongoDB | ERP業務はトランザクション整合性が重要であり、ACIDトランザクションをサポートするRDBMSが適切 |

#### 選択理由

1. **トランザクション整合性の確保**: ERP業務（販売注文の確定→在庫引当→請求書発行）では、複数テーブルへの同時更新が必要であり、ACIDトランザクションが不可欠。

2. **幅広い環境への対応**: オンプレミス導入からクラウドまで、顧客の環境に応じてDBMSを選択できる柔軟性を確保。

3. **開発効率の向上**: SQLiteをデフォルトとすることで、`composer create-project`直後に追加設定なしで動作開始できる。

4. **MongoDBを不採用とした理由**: ERPの業務データ（販売注文、請求書、在庫移動等）は構造化データであり、リレーショナルモデルが適切。トランザクション処理の観点でもRDBMSが優れている。

#### 不採用とした選択肢の理由
- **MongoDB**: ERP業務データは構造化されており、トランザクション処理が重要。ドキュメントDBは適さない。
- **DynamoDB**: クラウドロックインの問題があり、オンプレミス導入に対応できない。

---

### 5. キャッシュ選定

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| Redis | インメモリKVS | 高機能、永続化可能 | 運用コスト、追加インフラ |
| Memcached | シンプルKVS | 軽量、高速 | 機能限定 |
| Database | DBテーブルベース | 追加インフラ不要 | パフォーマンス限界 |
| File | ファイルベース | 追加インフラ不要 | スケール時に問題 |
| なし | キャッシュなし | シンプル | パフォーマンス課題 |

#### 採用した方針
**Database（デフォルト）/ Redis（本番環境オプション）**

`config/cache.php`で`CACHE_STORE=database`がデフォルト設定。

#### キャッシュ構成
| 用途 | キャッシュ | TTL | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- | --- |
| デフォルト | Database | 環境設定依存 | Redis | 追加インフラなしで動作し、中小企業でも導入容易 |
| 本番環境（高負荷時） | Redis | 環境設定依存 | Memcached | スケールアウト時にセッション共有、高速キャッシュが必要な場合 |

#### 選択理由

1. **最小構成での動作**: 中小企業向けERPとして、追加のインフラ（Redisサーバー）なしで動作することが重要。Databaseキャッシュにより、DBサーバーのみで完結。

2. **スケール時の選択肢確保**: `CACHE_STORE=redis`に変更するだけでRedisキャッシュに切り替え可能。負荷増大時にも対応できる設計。

3. **Redisをデフォルトにしなかった理由**: 中小企業向けERPとして、導入時のインフラ要件を最小化するため。Redis設定が必要になると導入障壁が上がる。

#### 不採用とした選択肢の理由
- **Redis（デフォルト）**: 追加インフラが必要となり、中小企業の導入障壁が上がる。
- **なし**: パフォーマンス問題が発生しやすく、特にFilament Admin Panelの動作に影響。

---

### 6. メッセージキュー選定

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| Kafka | 分散ストリーム基盤 | 高スループット、耐久性 | 運用複雑、ERPには過剰 |
| RabbitMQ | メッセージブローカー | 柔軟なルーティング | スケール制限 |
| AWS SQS | マネージドキュー | 運用不要、スケーラブル | クラウドロックイン |
| Database Queue | DBテーブルベースキュー | 追加インフラ不要 | パフォーマンス限界 |
| Redis Queue | インメモリキュー | 高速、低遅延 | 追加インフラ必要 |
| なし | 同期処理のみ | シンプル | レスポンス低下 |

#### 採用した方針
**Database Queue（デフォルト）/ Redis・SQS（本番環境オプション）**

`config/queue.php`で`QUEUE_CONNECTION=database`がデフォルト設定。

#### MQ構成
| 用途 | MQ | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| メール送信 | Database Queue | Redis | 見積書送信、請求書送信などの非同期処理に対応 |
| PDF生成 | Database Queue | Redis | 帳票生成をバックグラウンドで実行し、UIレスポンスを確保 |
| 大量データ処理 | Redis Queue（推奨） | SQS | 大量の請求書一括発行などでは高スループットが必要 |

#### 選択理由

1. **非同期処理のニーズ対応**: ERP業務では、メール送信（見積書、請求書）、PDF生成（帳票、請求書）など、レスポンス時間に影響を与えずにバックグラウンドで処理したいタスクが存在。

2. **最小構成での動作**: Database Queueにより、追加インフラなしで非同期処理を実現。`php artisan queue:listen`で処理開始。

3. **Kafkaを不採用とした理由**: ERPの非同期処理は主にメール送信・PDF生成であり、Kafkaのような高スループット基盤は過剰。運用コストに見合わない。

4. **同期処理のみを不採用とした理由**: PDF生成やメール送信は時間がかかるため、同期処理ではユーザー体験が低下する。

#### 不採用とした選択肢の理由
- **Kafka**: ERPの非同期処理ニーズには過剰。運用複雑さに見合わない。
- **なし（同期処理のみ）**: メール送信・PDF生成でレスポンス低下が発生。

---

### 7. フレームワーク選定

#### 想定される選択肢（言語・フレームワーク）

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| **Laravel + Filament（採用）** | PHPフレームワーク + Admin Panel | エコシステム充実、開発速度、Admin Panel統合 | PHPの制約 |
| Symfony | PHPフレームワーク | 柔軟性、エンタープライズ向け | 学習コスト高、Admin Panel別途 |
| Ruby on Rails | Rubyフレームワーク | 開発速度、Convention over Configuration | ホスティング制約、Rubyist以外には敷居高い |
| Django | Pythonフレームワーク | Admin自動生成、データ分析との親和性 | カスタマイズ性がやや低い |
| Spring Boot | Javaフレームワーク | エンタープライズ向け、型安全 | 開発速度、学習コスト |

#### 採用した方針
**Laravel 11.x + Filament 4.x**

PHP 8.2以上を要件として、Laravel 11.xをベースに、Filament 4.xをAdmin Panelとして採用。

#### 選択理由

1. **開発速度の最大化**: FilamentはLaravelに最適化されたAdmin Panelフレームワークであり、CRUD操作、フォーム、テーブル、ダッシュボードを迅速に構築できる。ERPの多数の管理画面を効率的に開発可能。

2. **エコシステムの活用**: Spatie Laravel Permission（権限管理）、Spatie Laravel Settings（設定管理）、DomPDF（PDF生成）など、Laravel向けの豊富なパッケージを活用。

3. **ホスティングの容易さ**: PHPはほぼ全てのホスティング環境で動作し、共用サーバーからクラウドまで幅広い環境に対応。中小企業の多様なインフラ環境に適合。

4. **コミュニティの規模**: LaravelはPHPフレームワークで最大のコミュニティを持ち、ドキュメント、チュートリアル、Stack Overflowの情報が豊富。OSSとして開発者を集めやすい。

5. **Symfonyを不採用とした理由**: 柔軟性は高いが、Admin Panelを別途構築する必要があり、開発速度でFilamentに劣る。

6. **Spring Bootを不採用とした理由**: エンタープライズ向けには適しているが、中小企業向けERPとしては開発コスト・学習コストが高い。

#### 不採用とした選択肢の理由
- **Symfony**: Admin Panel統合がなく、開発速度でFilamentに劣る。
- **Ruby on Rails**: Rubyist以外には参入障壁が高く、ホスティング環境も限定的。
- **Spring Boot**: 開発コスト・学習コストが高く、中小企業向けERPには過剰。

---

### 8. 横断的関心事の実装方式

#### 認証・認可

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| JWT | トークンベース | ステートレス | トークン無効化が難しい |
| **セッション（採用）** | サーバー側セッション | 即座に無効化可能 | スケール時に共有必要 |
| OAuth2 | 外部認証連携 | 標準的、SSO対応 | 実装複雑 |

**採用した方針**: セッションベース認証（Laravel標準）+ Spatie Permission + Filament Shield

**選択理由**:
- ERP業務ではセキュリティ上、ユーザーの即座の無効化（退職、権限変更）が重要であり、セッションベースが適切。
- Filament Shieldにより、リソース・ページ・ウィジェット単位での細粒度な権限管理を実現。
- マルチテナント（チーム・会社単位のデータ分離）は`security`プラグインで実装。

#### ロギング

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| **構造化ログ（採用）** | JSON形式 | 検索・分析容易 | 可読性低い |
| プレーンテキスト | 人間可読 | デバッグ容易 | 分析しにくい |

**採用した方針**: Monolog（Laravel標準）+ ActivityLog Model

**選択理由**:
- アプリケーションログはMonolog（Laravel標準）で`storage/logs/laravel.log`に出力。
- 監査ログ（誰がいつ何を変更したか）は`activity_logs`テーブルに保存し、業務上の追跡が可能。

#### エラーハンドリング

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| **例外ベース + グローバルハンドラ（採用）** | 例外をスロー、一括処理 | 明示的、一元管理 | 制御フローが複雑 |
| Result型 | 戻り値で表現 | 明示的なエラー処理 | コード量増加 |

**採用した方針**: 例外ベース + Laravel Exception Handler + Filament Form Validation

**選択理由**:
- Laravelの例外ハンドリング機構を活用し、認証エラー、認可エラー、バリデーションエラーを一元的に処理。
- Filament Formのバリデーション機構により、ユーザー入力エラーを即座にフィードバック。

---

## 将来の拡張性考慮

### 想定される変更
1. **新業務ドメインの追加（製造管理、CRM拡張等）**: プラグインとして追加することで、既存機能に影響を与えずに機能拡張可能。`Package`クラスを継承した新しいServiceProviderを作成し、`bootstrap/plugins.php`に登録するだけで有効化できる。

2. **外部システム連携（会計ソフト、EC、決済サービス等）**: Infrastructure層に新しいアダプターを追加することで対応。例えば、Stripeとの決済連携は`payments`プラグインにStripeアダプターを追加する形で実現可能。

3. **マルチテナントの強化（SaaS化）**: 現在のチーム・会社単位のデータ分離を拡張し、完全なテナント分離（DBスキーマ分離またはDB分離）に対応可能。Filament Shieldの`tenant_model`設定を活用。

### スケールアウト時の対応
1. **水平スケーリング**: Laravel標準のスケーリング手法（ロードバランサー + 複数Webサーバー）に対応。セッション共有にはRedisを使用。

2. **キャッシュの強化**: `CACHE_STORE=redis`に変更し、Redis Clusterでキャッシュを分散。

3. **キューワーカーの分散**: `QUEUE_CONNECTION=redis`に変更し、複数のキューワーカーを並列実行。

4. **読み取りレプリカの活用**: Laravelの`read`/`write`コネクション設定で、読み取りクエリを読み取りレプリカに分散。

---

## 備考

- **バージョン情報**: Aureus ERP v1.2.0
- **Laravel バージョン**: 11.x
- **PHP バージョン要件**: 8.2以上
- **Filament バージョン**: 4.x
- **ライセンス**: MIT License
- **本ドキュメントの作成日**: 2026-01-19
- **参照資料**:
  - アーキテクチャ設計書: `docs/code-to-docs/アーキテクチャ設計書/アーキテクチャ設計書.md`
  - composer.json
  - plugins/webkul/support/src/Package.php
  - plugins/webkul/support/src/PackageServiceProvider.php
  - config/database.php
  - config/cache.php
  - config/queue.php
  - config/filament-shield.php
