# セキュリティ設計意図書

## 概要

本ドキュメントは、AureusERP（Laravel/Filamentベースのエンタープライズリソースプランニングシステム）のセキュリティ設計における設計意図を記述します。各セキュリティ対策がなぜ選択されたのか、業務要件との関連性、代替案との比較を通じて、設計判断の根拠を明確にします。

---

## 業務要件との関連（なぜこのセキュリティ設計が必要か）

### 対応する業務要件

| 業務要件ID | 業務要件名 | セキュリティに求められる要件 |
| --- | --- | --- |
| BR-001 | 企業リソース管理 | 財務・在庫・人事等の機密データの適切なアクセス制御 |
| BR-002 | 複数会社対応 | 会社間のデータ隔離と適切な権限境界の設定 |
| BR-003 | 顧客向けポータル | 顧客データの保護と内部システムからの分離 |
| BR-004 | 従業員管理 | 個人情報保護法に準拠した従業員データの取り扱い |
| BR-005 | 取引先管理 | 取引情報・契約情報の機密性確保 |

### 業務観点でのセキュリティ要件

#### 取り扱うデータの機密性

ERPシステムとして本システムが扱うデータは、企業経営の根幹に関わる機密性の高い情報です。

| データ種別 | 機密性レベル | 業務上の保護理由 |
| --- | --- | --- |
| 従業員情報（人事・給与） | 高 | 個人情報保護法による法的義務、従業員からの信頼維持 |
| 財務・会計データ | 高 | 企業の経営状況の漏洩防止、取引先への信用維持 |
| 顧客・取引先情報 | 高 | 顧客の個人情報保護、取引関係の維持 |
| 在庫・商品情報 | 中 | 競合への情報漏洩防止、価格戦略の保護 |
| プロジェクト情報 | 中 | 進捗・工数情報の競合漏洩防止 |
| システム設定 | 低 | 運用情報の保護、システム攻撃のヒント遮断 |

#### 法令・規制への対応

| 法令/規制 | 概要 | 業務への影響 |
| --- | --- | --- |
| 個人情報保護法 | 従業員・顧客の個人情報の適切な取り扱い | 違反時の行政処分・損害賠償リスク、信用失墜 |
| 不正アクセス禁止法 | 不正アクセスの防止義務 | システム侵害時の法的責任、被害拡大防止 |
| 労働安全衛生法 | 従業員の健康情報等の適切な管理 | 勤怠・休暇情報の適切なアクセス制御 |

#### お客様からのセキュリティ要求

| 要求事項 | 内容 | 対応方針 |
| --- | --- | --- |
| ロールベースアクセス制御 | 職務に応じた最小権限の原則 | Spatie Permission + Filament Shieldで実装 |
| 監査ログ | 重要操作の記録と保持 | ログ記録チャネルの設定 |
| セッション管理 | セッションハイジャック対策 | データベースセッション、HTTPOnly Cookie |
| パスワードポリシー | 安全なパスワード要件 | Laravelのパスワードルール適用 |

### 検討したセキュリティレベル（業務観点）

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | 最小限のセキュリティ（パスワード認証のみ、単純なロール） | 低コスト、シンプル、導入容易 | 80以上のリソースに対する細かい制御不可、法令違反リスク | 低 |
| B | 標準的なセキュリティ（RBAC、基本的なセッション管理） | バランス良い、一般的なERP水準 | 大規模組織での柔軟性に欠ける可能性 | 中 |
| C（採用） | 包括的なセキュリティ（RBAC + リソース単位権限、強化されたセッション管理、Rate Limiting） | 法令準拠、細かいアクセス制御、エンタープライズ対応 | 実装・運用の複雑さ、初期設定の手間 | 高 |

### 選択理由（業務観点）

1. **業務要件との適合性**: ERPシステムは80以上のリソース（従業員、在庫、会計、採用等）を扱うため、リソース単位での細かい権限制御が不可欠です。案Cは各リソースに対してview_any, view, create, update, delete等の11種類の権限を個別に設定でき、「経理担当者は請求書の閲覧・編集は可能だが従業員情報にはアクセス不可」といった業務要件を正確に実装できます。

2. **法令・規制への準拠**: 個人情報保護法の要求する「アクセス制御」「適切な安全管理措置」を満たすため、最小権限の原則に基づいたRBACが必要です。案AやBでは組織の職務分掌に応じた細かい制御が困難です。

3. **顧客からの信頼確保**: エンタープライズ向けERPとして、セキュリティチェックシートへの対応が必要となる場面が想定されます。案CはRate Limiting、セッション固定化対策、CSRF保護等の包括的な対策により、取引先のセキュリティ要求に応えられます。

4. **インシデント発生時の業務影響**: 万一のセキュリティインシデント時、案Cでは監査ログによる追跡、セッションの即時無効化、権限の緊急変更等の対応が可能です。案A/Bでは事後対応が困難となります。

---

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| 認証方式 | パスワード / OAuth / SAML / MFA | セッションベースパスワード認証 + メール検証 | SME向けの実装シンプルさと十分なセキュリティのバランス |
| 認可モデル | RBAC / ABAC / ACL | RBAC（Spatie Permission + Filament Shield） | 80以上のリソースに対する細かい権限管理とFilamentとの統合 |
| セッション管理 | Cookie / JWT / サーバーサイド | データベースセッション | セッション無効化の容易さと水平スケーリング対応 |
| 暗号化戦略 | TLS / AES / KMS | AES-256-CBC + bcrypt | Laravel標準の信頼性と十分な暗号強度 |
| 監査ログ | 全記録 / 重要のみ / なし | 日次ローテーション（14日保持） | コストと追跡可能性のバランス |

---

## 設計判断と根拠

### 1. 認証方式

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| パスワード認証 | ID/PW認証 | シンプル、導入容易、ユーザー慣れ | 単体ではセキュリティ弱い |
| OAuth 2.0 / OIDC | トークンベース認証 | 外部IdP連携、シングルサインオン | 実装複雑、外部依存 |
| SAML | エンタープライズSSO | 大企業のIdP連携 | 実装複雑、XMLベース |
| 多要素認証（MFA） | 複数要素の組み合わせ | 高セキュリティ | UX低下、導入コスト |
| パスキー/FIDO2 | パスワードレス | 高セキュリティ、フィッシング耐性 | 対応デバイス・ブラウザ限定 |

#### 採用した方針

**セッションベースパスワード認証 + メール検証 + Rate Limiting**

本システムでは、Laravelの標準的なセッションベース認証に、FilamentのemailVerification機能とRate Limitingを組み合わせた認証方式を採用しています。

#### 認証フロー

1. ユーザーがログインフォームにID/パスワードを入力
2. `WithRateLimiting`により2回/試行の制限を適用（ブルートフォース対策）
3. パスワードをbcryptでハッシュ比較
4. 認証成功時に`session()->regenerate()`でセッションIDを再生成（セッション固定化対策）
5. メール未検証の場合は検証メールを送信
6. セッションをデータベースに保存

#### 選択理由

1. **業務要件との適合性**: ERPシステムの主要ユーザーは企業の従業員であり、日常的な業務操作にMFAを毎回要求することはUXを著しく低下させます。パスワード認証は従業員が慣れ親しんだ方式であり、Rate LimitingとemailVerificationの組み合わせで実用的なセキュリティレベルを確保できます。

2. **SME（中小企業）への配慮**: 本システムのターゲットであるSMEは、OAuthやSAML対応のIdPを持たないケースが多いです。外部IdP依存を前提としない自己完結型の認証が適切です。

3. **段階的な拡張可能性**: Laravelの認証基盤は拡張性が高く、将来的にSocialite（OAuth）やMFAパッケージの追加が容易です。現時点で複雑な実装を避けつつ、成長に応じた拡張が可能です。

4. **コスト効率**: 自社でIdPを構築・運用するコスト、外部IdPへの依存コストを避け、Laravel標準機能で実現できる方式を採用しています。

#### 不採用とした選択肢の理由

- **OAuth 2.0 / OIDC**: SMEでの外部IdP依存は導入障壁となります。大企業向けの将来機能として検討の余地はありますが、現時点では過剰仕様と判断しました。
- **SAML**: XMLベースの複雑な実装が必要であり、主にレガシーエンタープライズ環境向けです。モダンなLaravel/Filamentスタックとの親和性が低いと判断しました。
- **MFA**: 将来的な追加オプションとして検討していますが、標準搭載による全ユーザーへの強制は業務効率への影響が大きいと判断しました。
- **パスキー/FIDO2**: 対応デバイス・ブラウザの制約があり、多様な環境で利用されるERPシステムには時期尚早と判断しました。

---

### 2. 認可モデル

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| RBAC | ロールベースアクセス制御 | シンプル、管理容易、標準的 | 非常に柔軟な制御は困難 |
| ABAC | 属性ベースアクセス制御 | 柔軟、コンテキスト考慮可能 | 実装・管理が複雑 |
| ACL | アクセス制御リスト | リソース単位の細かい制御 | スケールしにくい、管理煩雑 |
| ReBAC | 関係ベースアクセス制御 | 関係性表現可能 | 実装複雑、新しいパラダイム |

#### 採用した方針

**RBAC（Spatie/Laravel-Permission + Filament Shield）によるロール・権限管理**

本システムでは、Spatie/Laravel-Permissionをベースに、Filament Shieldプラグインを組み合わせたRBAC（ロールベースアクセス制御）を採用しています。

#### 権限構成

| ロール/属性 | 権限 | 対象リソース | 選択理由 |
| --- | --- | --- | --- |
| Admin | 全リソースのview_any, view, create, update, delete等 | 80以上のFilamentリソース | パネルユーザーのデフォルトロール |
| カスタムロール | リソース単位で個別に権限付与 | 必要なリソースのみ | 職務に応じた最小権限の実現 |
| super_admin | 無制限アクセス | 全リソース | **無効化**（明示的権限管理を強制） |

#### 権限の粒度

各リソースに対して以下の11種類の権限を個別に設定可能:

- `view_any`: 一覧表示
- `view`: 詳細表示
- `create`: 新規作成
- `update`: 編集
- `delete`: 削除
- `delete_any`: 一括削除
- `restore`: 復元
- `restore_any`: 一括復元
- `force_delete`: 完全削除
- `force_delete_any`: 一括完全削除
- `reorder`: 並び替え

#### 選択理由

1. **業務要件との適合性**: ERPシステムは部門・職務に応じた明確な権限分離が必要です。RBACは「営業部長」「経理担当」といった組織構造に直接マッピングでき、管理者が直感的に権限設定できます。

2. **80以上のリソースへの対応**: 本システムはPartner、Invoice、Employee、Inventory等80以上のリソースを持ちます。Filament Shieldは各リソースに対するPolicyを自動生成し、一元的な権限管理UIを提供するため、運用負荷を軽減できます。

3. **super_adminの無効化意図**: 設定で`super_admin.enabled = false`としているのは、「全権限を持つユーザー」の存在を許可しないことで、必ず明示的な権限付与を強制する設計意図があります。これにより監査時の権限証跡が明確になります。

4. **Filament統合**: FilamentのリソースベースのUIアーキテクチャと、Spatie Permissionの権限モデルがFilament Shieldを通じてシームレスに統合されています。権限管理画面からGUIで各リソースの権限を設定でき、非エンジニアの管理者でも運用可能です。

5. **Laravelエコシステムとの親和性**: Spatie/Laravel-Permissionは最も広く使われているLaravelの権限管理パッケージであり、豊富なドキュメント、コミュニティサポート、実績があります。

#### 不採用とした選択肢の理由

- **ABAC**: 属性ベースの動的な権限（「同じ部署のデータのみ閲覧可能」等）は魅力的ですが、80以上のリソースに対するポリシー定義・管理が複雑になります。現時点ではRBACで十分な粒度が得られると判断しました。
- **ACL**: 個々のリソースインスタンスへの権限設定は管理が煩雑になり、ERPのような多数のレコードを扱うシステムには不向きです。
- **ReBAC**: Google Zanzibar等で採用されている先進的なモデルですが、Laravel/Filamentエコシステムでの実績が少なく、学習コスト・実装リスクが高いと判断しました。

---

### 3. セッション管理

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ファイルセッション | ファイルシステムに保存 | シンプル、設定不要 | 負荷分散困難、パフォーマンス限界 |
| データベースセッション | DBテーブルに保存 | 水平スケーリング対応、中央管理 | DB負荷増加 |
| Redisセッション | インメモリストアに保存 | 高速、スケーラブル | インフラ追加必要 |
| JWT（ステートレス） | トークンに情報含む | スケーラブル、マイクロサービス向け | 失効困難、トークンサイズ |

#### 採用した方針

**データベースセッション（120分有効期限）+ セキュアCookie設定**

本システムでは、セッションドライバーとしてdatabase方式を採用し、以下の設定を適用しています:

#### セッション設定

| 項目 | 設定値 | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| ドライバー | database | file, redis, cookie | 水平スケーリング対応、追加インフラ不要 |
| 有効期限 | 120分 | 15分, 60分, 480分 | 業務作業時間とセキュリティのバランス |
| HTTPOnly | true | false | JavaScriptからのCookieアクセス遮断 |
| Same-Site | lax | strict, none | CSRF対策と利便性のバランス |
| セッション再生成 | ログイン時 | 毎リクエスト | セッション固定化攻撃対策 |

#### 選択理由

1. **水平スケーリング対応**: ERPシステムは利用者数の増加に伴いサーバーを増設する可能性があります。ファイルセッションは単一サーバー限定ですが、データベースセッションは複数サーバー間で共有可能です。

2. **Redis不要の運用シンプルさ**: Redisは高速ですが、追加のインフラ管理（Redis serverの監視、可用性確保）が必要です。データベースセッションはERPが元々必要とするDBを流用でき、運用負荷を増やしません。

3. **120分の業務適合性**: 一般的なオフィス業務では、午前/午後それぞれ2時間程度の集中作業時間があります。120分の有効期限は、通常業務中の再認証を避けつつ、昼休みや離席時のセキュリティを確保するバランス点です。

4. **セッション固定化対策**: `session()->regenerate()`をログイン時に実行することで、攻撃者が事前に取得したセッションIDによる乗っ取りを防止しています。

5. **セッション無効化の容易さ**: データベースセッションは、管理者がDBから直接セッションレコードを削除することで即時無効化が可能です。JWTでは有効期限まで無効化できない課題があります。

#### 不採用とした選択肢の理由

- **ファイルセッション**: 単一サーバー環境では動作しますが、将来の拡張性を考慮し不採用としました。
- **Redis**: 高パフォーマンスですが、SME向けERPとして追加インフラの要求は導入障壁となります。将来の大規模環境向けオプションとして検討の余地があります。
- **JWT**: ステートレスな特性はマイクロサービスには適しますが、モノリシックなLaravelアプリケーションでは、即時セッション無効化の困難さがセキュリティリスクとなります。

---

### 4. 暗号化戦略

#### 想定される選択肢（通信）

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| TLS 1.2 | 標準的なTLS | 広く対応 | 1.3より若干弱い |
| TLS 1.3 | 最新TLS | 高セキュリティ、高速 | 古いクライアント非対応 |
| mTLS | 相互TLS | 高セキュリティ | 証明書管理複雑 |

#### 想定される選択肢（保存データ）

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ディスク暗号化 | ボリュームレベル暗号化 | 透過的 | 細かい制御不可 |
| カラムレベル暗号化 | 特定カラムのみ暗号化 | 細かい制御 | 実装複雑、検索不可 |
| アプリケーション暗号化 | アプリで暗号化/復号 | 完全制御 | 実装・運用複雑 |

#### 採用した方針

**AES-256-CBC（アプリケーション暗号化）+ bcrypt（パスワードハッシュ）**

本システムでは、以下の暗号化戦略を採用しています:

#### 暗号化対象

| 対象 | 方式 | アルゴリズム | 鍵管理 | 選択理由 |
| --- | --- | --- | --- | --- |
| アプリケーションデータ | Laravel Encrypter | AES-256-CBC | APP_KEY（環境変数） | Laravel標準、十分な暗号強度 |
| Cookie | EncryptCookiesミドルウェア | AES-256-CBC | APP_KEY | 改ざん防止、情報漏洩防止 |
| パスワード | bcryptハッシュ | bcrypt | N/A（一方向） | 業界標準、レインボーテーブル耐性 |

#### 選択理由

1. **AES-256-CBCの採用理由**: AES-256は米国政府でも採用される暗号化標準であり、現時点で実用的な攻撃方法は知られていません。Laravelが標準でサポートしており、追加の実装コストなく利用できます。

2. **bcryptの採用理由**: パスワードハッシュにはArgon2id等のより新しいアルゴリズムも存在しますが、bcryptは長年の実績があり、Laravelの`hashed`キャストでシームレスに利用できます。ソルト付きで自動的にレインボーテーブル攻撃を防止します。

3. **APP_KEY管理**: 暗号化キーは環境変数`APP_KEY`として管理し、ソースコードには含めません。`APP_PREVIOUS_KEYS`によるキーローテーションにも対応しており、定期的なキー更新が可能です。

4. **Cookie暗号化の理由**: ERPシステムはセッション情報や一時データをCookieに保存する場合があります。EncryptCookiesミドルウェアにより、これらの情報が改ざんされたり、クライアント側で読み取られることを防止します。

#### 不採用とした選択肢の理由

- **カラムレベル暗号化**: 特定のカラム（例：給与情報）を個別に暗号化する方式は、検索やソートが不可能になる制約があります。ERPの一覧画面でのフィルタリング等の業務要件との両立が困難です。
- **KMS（Key Management Service）**: AWS KMS等のマネージドサービスは高度な鍵管理を提供しますが、クラウド依存・コスト増となり、オンプレミス環境やマルチクラウド環境での利用に制約が生じます。

---

### 5. 入力検証・サニタイゼーション

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ホワイトリスト検証 | 許可パターンのみ通過 | 安全 | 厳格すぎる場合あり |
| ブラックリスト検証 | 危険パターンを除外 | 柔軟 | 漏れリスク |
| エスケープ処理 | 出力時にエスケープ | コンテキスト対応 | 漏れリスク |
| パラメータ化クエリ | SQLインジェクション対策 | 確実 | 実装規約必要 |

#### 採用した方針

**多層防御（ホワイトリスト検証 + エスケープ処理 + パラメータ化クエリ）**

本システムでは、Laravel/Filamentフレームワークの標準機能を活用した多層防御を採用しています。

#### 対策対象脅威

| 脅威 | 対策 | 他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| SQLインジェクション | Eloquent ORM（プリペアドステートメント） | 生SQL、クエリビルダー | フレームワーク標準、漏れなし |
| XSS | Blade自動エスケープ + Filamentコンポーネント | 手動エスケープ | フレームワーク標準、漏れなし |
| CSRF | VerifyCsrfTokenミドルウェア + SameSite Cookie | 手動トークン検証 | フレームワーク標準、透過的 |
| Mass Assignment | $fillable属性の明示的定義 | $guarded | ホワイトリスト方式でより安全 |

#### 選択理由

1. **Eloquent ORMの強制理由**: 生のSQLクエリを書く代わりにEloquentを使用することで、すべてのデータベースアクセスがプリペアドステートメントとなり、SQLインジェクションを根本的に防止できます。

2. **Blade自動エスケープ**: `{{ $variable }}`構文で出力する際、自動的にHTMLエスケープが適用されます。`{!! $variable !!}`（非エスケープ）の使用はコードレビューで監視し、必要最小限に限定します。

3. **VerifyCsrfTokenミドルウェア**: AdminPanelProviderのミドルウェアスタックに含まれており、すべてのPOST/PUT/DELETE/PATCHリクエストに対してCSRFトークン検証が自動適用されます。

4. **$fillableによるMass Assignment対策**: Userモデルでは`$fillable`で許可する属性を明示的に列挙しています。これにより、リクエストから直接モデルにデータを渡す際の意図しない属性の書き換えを防止します。

5. **パスワードバリデーションルール**: `Password::default()`を使用することで、Laravelのデフォルトパスワード強度ルール（最小8文字等）が自動適用されます。

#### 不採用とした選択肢の理由

- **ブラックリスト検証**: 危険なパターンを列挙するアプローチは、新たな攻撃パターンに対応できない「いたちごっこ」になります。フレームワーク標準のホワイトリスト方式を採用しました。
- **手動サニタイゼーション**: フレームワーク任せにせず手動でサニタイズする方式は、漏れが生じるリスクがあります。Laravel/Filamentの標準機能を信頼するアプローチを採用しました。

---

### 6. 監査ログ・追跡

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 全イベント記録 | すべての操作をログ | 完全な追跡 | ストレージ・コスト大 |
| 重要イベントのみ | 認証・変更等を選択記録 | バランス | 漏れリスク |
| 最小限 | エラーのみ | コスト小 | 追跡困難 |

#### 採用した方針

**日次ローテーション（14日保持）+ Slack通知（クリティカル）**

本システムでは、以下のログ記録方針を採用しています:

#### ログ記録対象

| イベント種別 | 記録項目 | 保持期間 | 選択理由 |
| --- | --- | --- | --- |
| アプリケーションログ | エラー、警告、情報 | 14日（daily） | 障害調査に十分な期間 |
| 緊急ログ | critical以上 | Slack即時通知 | 即座の対応が必要なイベント |
| 非推奨警告 | 無効化（null） | N/A | 本番環境でのノイズ削減 |

#### 選択理由

1. **14日保持の根拠**: 多くの企業では週次の定例会議でシステム状況を確認します。14日間のログがあれば、前回確認時からの全ログを参照でき、問題の発見・調査が可能です。

2. **日次ローテーションの理由**: 単一ファイルの肥大化を防ぎ、古いログの自動削除により運用負荷を軽減します。dailyドライバーは日付ごとにファイルを分割し、指定日数を超えたファイルを自動削除します。

3. **Slack通知の理由**: criticalレベルのエラー（認証障害、データベース接続エラー等）は即座の対応が必要です。メールよりもリアルタイム性の高いSlackへの通知により、インシデント対応時間を短縮します。

4. **非推奨警告の無効化理由**: 開発環境では有用ですが、本番環境ではノイズとなりログファイルを肥大化させます。環境変数`LOG_DEPRECATIONS_CHANNEL=null`で無効化しています。

5. **操作ログの追加検討**: 現時点ではLaravel標準のログ機能を使用していますが、将来的にはSpatie/Laravel-Activitylog等のパッケージを追加し、「誰が」「いつ」「何を」変更したかを記録する監査証跡機能の追加を検討しています。

#### 不採用とした選択肢の理由

- **全イベント記録**: ERPシステムは多数のユーザーが日常的に操作するため、全リクエストのログ記録はストレージコストが膨大になります。重要なイベントに絞った記録が実用的です。
- **エラーのみ**: 障害発生時の調査には、エラー発生前後のコンテキスト（どの操作でエラーになったか）が重要です。エラーのみでは十分な情報が得られません。

---

### 7. シークレット管理

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 環境変数 | 環境変数で渡す | シンプル | ローテーション困難 |
| Secrets Manager | マネージドシークレット | ローテーション対応 | クラウド依存、コスト |
| Vault | HashiCorp Vault | 高機能、柔軟 | 運用複雑 |
| 設定ファイル（暗号化） | 暗号化した設定ファイル | シンプル | 鍵管理必要 |

#### 採用した方針

**環境変数（.env）によるシークレット管理 + キーローテーション対応**

本システムでは、Laravelの標準的な環境変数ベースのシークレット管理を採用しています:

#### シークレット管理

| シークレット種別 | 保管場所 | ローテーション | 選択理由 |
| --- | --- | --- | --- |
| APP_KEY | .env（環境変数） | APP_PREVIOUS_KEYS対応 | Laravel標準、シンプル |
| DB_PASSWORD | .env（環境変数） | 手動 | デプロイ時に変更可能 |
| API_KEY（外部連携） | .env（環境変数） | 手動 | サービスごとに管理 |

#### 選択理由

1. **Laravel標準方式の採用理由**: .envファイルはGitには含めず（.gitignore）、本番環境では環境変数として注入します。この方式はLaravelのベストプラクティスであり、12 Factor Appの原則に沿っています。

2. **APP_PREVIOUS_KEYSによるローテーション**: 暗号化キー(APP_KEY)を変更する際、旧キーをAPP_PREVIOUS_KEYSに設定することで、旧キーで暗号化されたデータも復号可能な期間を設けられます。ダウンタイムなしのキーローテーションが可能です。

3. **シンプルさの重視**: HashiCorp Vault等の専用シークレット管理ツールは高機能ですが、運用に専門知識が必要です。SME向けERPとして、DevOps専任者がいない組織でも運用可能なシンプルな方式を採用しました。

4. **コンテナ化対応**: Docker/Kubernetes環境では、環境変数をSecrets/ConfigMapとして注入できます。.envベースの管理方式は、コンテナ化への移行を妨げません。

#### 不採用とした選択肢の理由

- **Secrets Manager（AWS/GCP/Azure）**: クラウドベンダーロックインとなり、オンプレミス環境やマルチクラウド環境での利用に制約が生じます。
- **HashiCorp Vault**: 高度な機能（動的シークレット、監査ログ等）は魅力的ですが、別途Vaultクラスターの運用が必要となり、SME向けには過剰です。

---

### 8. ネットワークセキュリティ

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| アプリケーションのみ | アプリレイヤーでの対策のみ | シンプル | インフラ攻撃に脆弱 |
| セキュリティグループ + WAF | L4/L7での防御 | 多層防御 | 設定・コスト |
| フル装備（WAF + DDoS + IDS） | 包括的防御 | 最高レベル | 高コスト、複雑 |

#### 採用した方針

**アプリケーションレイヤーでのセキュリティ対策（インフラは環境依存）**

本システムでは、アプリケーションレイヤーでの対策を実装し、インフラレベルの対策はデプロイ環境に応じて追加する方針としています:

#### セキュリティ構成

| レイヤー | 対策 | 設定 | 選択理由 |
| --- | --- | --- | --- |
| L7（アプリ） | ミドルウェアスタック | CSRF、Cookie暗号化、セッション認証 | Laravelフレームワーク標準 |
| L7（アプリ） | Rate Limiting | 2回/試行 | ブルートフォース対策 |
| L4-L7（インフラ） | 環境依存 | デプロイ先で設定 | オンプレ/クラウドの柔軟性 |

#### 選択理由

1. **環境非依存のアプリケーション**: ERPシステムはオンプレミス、AWS、GCP、Azure等様々な環境にデプロイされる可能性があります。アプリケーションコード自体はインフラに依存せず、ネットワークセキュリティはデプロイ時に環境に応じて構成する方針としています。

2. **AdminPanelProviderのミドルウェアスタック**: 以下のミドルウェアが順番に適用され、リクエストを多層的に検証します:
   - EncryptCookies: Cookie暗号化
   - AddQueuedCookiesToResponse: Cookie追加
   - StartSession: セッション開始
   - AuthenticateSession: セッション認証
   - ShareErrorsFromSession: エラー共有
   - VerifyCsrfToken: CSRFトークン検証
   - SubstituteBindings: ルートモデルバインディング

3. **HTTPOnly Cookie**: JavaScriptからのCookieアクセスを遮断し、XSSによるセッションハイジャックを防止します。

4. **Same-Site Cookie（lax）**: クロスサイトリクエストでのCookie送信を制限し、CSRF攻撃のリスクを軽減します。`strict`ではなく`lax`を選択したのは、外部サイトからのリンク経由でのアクセス（例：メールのリンク）を許容するためです。

#### 不採用とした選択肢の理由

- **WAF組み込み**: アプリケーションコード内にWAF機能を組み込む方式は、インフラレベルのWAF（AWS WAF, Cloudflare等）より性能が劣り、アプリケーションの責務範囲を超えます。
- **DDoS対策組み込み**: DDoS対策はネットワークレイヤーで行うべきであり、アプリケーションで対応するのは限界があります。CloudflareやAWS Shield等のサービスの利用を推奨します。

---

### 9. 脆弱性対策

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 手動パッチ適用 | 手動でパッチ適用 | 制御可能 | 遅延リスク |
| 自動パッチ適用 | 自動でパッチ適用 | 迅速 | 互換性リスク |
| 依存関係スキャン | ライブラリ脆弱性検出 | 早期発見 | 偽陽性 |
| ペネトレーションテスト | 定期的な侵入テスト | 実践的 | コスト |

#### 採用した方針

**Composer依存関係管理 + Laravel/Filamentのセキュリティアップデート追従**

本システムでは、以下の脆弱性対策を採用しています:

#### 対策実施

| 対策種別 | 頻度 | ツール | 選択理由 |
| --- | --- | --- | --- |
| 依存関係管理 | 開発時 | Composer | Laravel標準のパッケージ管理 |
| セキュリティアップデート | リリース時 | composer update | 脆弱性修正の適用 |
| 静的解析 | 開発時 | Laravel Pint | コード品質・セキュリティチェック |
| フレームワークアップデート | 定期 | Filament Upgrade | Filamentのセキュリティ修正適用 |

#### 選択理由

1. **Composerによる依存関係管理**: composer.jsonでパッケージのバージョン制約を管理し、`composer audit`コマンドで既知の脆弱性をチェックできます。

2. **Laravelセキュリティポリシー**: Laravelは重大なセキュリティ問題に対してセキュリティリリースを提供しています。Laravel 11.xを使用しており、LTSサポート期間内です。

3. **Filament Shieldのバージョン固定**: `bezhansalleh/filament-shield: ^4.0`でメジャーバージョンを固定し、互換性を維持しつつセキュリティ修正を受け取れる設定としています。

4. **プラグインアーキテクチャ**: 80以上のプラグイン（Webkul名前空間）は独自のcomposer.jsonを持ち、composer-merge-pluginで統合されます。各プラグインの依存関係も一元的に管理されます。

5. **自動アップグレードスクリプト**: `@php artisan filament:upgrade`がpost-autoload-dumpで実行され、Filamentの最新アセットが自動的に適用されます。

#### 不採用とした選択肢の理由

- **自動パッチ適用**: Dependabotや自動マージは便利ですが、ERPシステムでは互換性の検証なしに依存関係を更新するリスクがあります。手動レビュー後の適用を推奨します。
- **ペネトレーションテスト**: 定期的な実施は推奨しますが、アプリケーションの機能ではなく運用フェーズでの活動として位置づけています。

---

## コンプライアンス考慮

### 準拠規格

| 規格 | 対応状況 | 関連設計 | 選択理由 |
| --- | --- | --- | --- |
| 個人情報保護法 | 対応 | アクセス制御、監査ログ、パスワードハッシュ | 法的義務 |
| OWASP Top 10 | 対応 | SQLi対策、XSS対策、CSRF対策、認証・セッション管理 | 業界標準のセキュリティベストプラクティス |
| Laravel Security Best Practices | 対応 | フレームワーク標準機能の活用 | エコシステムの知見活用 |

---

## 将来の拡張性考慮

### 想定される変更

1. **多要素認証（MFA）の追加**: 将来的にlaravel-fortify等を導入し、TOTP（Google Authenticator等）によるMFAをオプションとして提供する可能性があります。現在の認証フローは拡張可能な設計となっています。

2. **OAuth/OIDC対応**: 大企業向けに、Azure AD、Google Workspace等のIdP連携が必要になる可能性があります。Laravel Socialiteの追加で対応可能です。

3. **監査ログの強化**: Spatie/Laravel-Activitylogを導入し、「誰が」「いつ」「何を」「どのように」変更したかを記録する詳細な監査証跡機能を追加する計画があります。

4. **マルチテナント対応強化**: filament-shieldの`tenant_model`設定を活用し、テナント（会社）ごとのロール・権限分離を強化する可能性があります。

### セキュリティ強化計画

1. **短期**: Rate Limitingの閾値調整、パスワードポリシーの強化（複雑性要件追加）
2. **中期**: MFAオプション追加、監査ログパッケージ導入
3. **長期**: OAuth/OIDC対応、SOC 2準拠に向けた対策強化

---

## 備考

- 本システムはFilamentフレームワークのセキュリティ機能を基盤としており、Laravel標準のセキュリティ機能に加えてFilament固有の保護機能が実装されています
- スーパー管理者機能は意図的に無効化されており、すべてのユーザーに明示的な権限付与が必要です
- 80以上のFilamentリソースに対して個別のPolicy定義が存在し、きめ細かいアクセス制御が可能です
- セキュリティ設計は継続的に見直しを行い、新たな脅威や要件に対応していく必要があります
