# データベース設計意図書

## 概要

本ドキュメントは、AureusERP（Laravel/Filament ベースのオープンソースERP）のデータベース設計における設計意図を記述したものです。既存のデータベース設計書を基に、コードベースから設計意図を類推し、各設計判断の背景、選択理由、不採用とした代替案について詳細に解説します。

本システムは、販売管理、購買管理、在庫管理、プロジェクト管理、人事管理、会計管理、ブログ管理など複数のドメインを持つモジュラー構造のERPアプリケーションであり、データベース設計においても各ドメインの独立性と相互連携を両立させる設計思想が反映されています。

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| 正規化レベル | 第3正規形 / 非正規化 / ハイブリッド | ハイブリッド（基本第3正規形＋一部非正規化） | ERPの多様な業務要件に対応しつつ、クエリパフォーマンスも確保 |
| 主キー戦略 | AUTO_INCREMENT / UUID / 複合キー | AUTO_INCREMENT（bigint unsigned） | Laravelの標準、インデックス効率、シンプルさを重視 |
| 削除方式 | 論理削除 / 物理削除 / ハイブリッド | ハイブリッド（主要エンティティは論理削除） | 監査証跡の保持と復元可能性を重視、関連テーブルは物理削除も許容 |
| 履歴管理 | 履歴テーブル / 変更ログ / イベントソーシング | created_at/updated_at + creator_id による監査証跡 | シンプルさと実装コストのバランス |

---

## 設計判断と根拠

### 1. 正規化レベルの選択

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 第3正規形 | 完全に正規化 | データ整合性が高い | JOIN多発でパフォーマンス低下 |
| 非正規化 | パフォーマンス優先で冗長化 | 読み取り高速 | 更新時の整合性維持が困難 |
| ハイブリッド | 基本正規化＋一部非正規化 | バランス良い | 設計判断が複雑 |

#### 採用した方針
**ハイブリッド方式**（基本的に第3正規形を採用しつつ、パフォーマンスが重要な箇所で一部非正規化）

#### 選択理由

マイグレーションファイルの分析から、以下の設計パターンが確認されました：

1. **基本的な第3正規形の適用**
   - マスタデータ（`products_categories`、`partners_titles`、`employees_departments`等）は独立したテーブルとして正規化
   - 外部キー制約による参照整合性の保証
   - 多対多関係はピボットテーブル（`products_product_tag`、`projects_project_tag`等）で管理

2. **意図的な非正規化の適用**
   - `sales_orders`テーブルに`amount_untaxed`、`amount_tax`、`amount_total`を保持（明細行からの計算値を冗長化）
   - `purchases_orders`テーブルにも同様に`untaxed_amount`、`tax_amount`、`total_amount`を保持
   - `products_categories`テーブルに`full_name`、`parent_path`を保持（階層構造のクエリ高速化）

3. **業務要件への適合性**
   - ERPシステムでは帳票出力やレポート生成時に集計値の高速参照が必要
   - 受注・発注オーダーの金額サマリーは頻繁に参照されるため、計算済み値の保持が合理的

#### 不採用とした選択肢の理由
- **完全な第3正規形**: ERPシステムでは帳票出力時のパフォーマンスが重要であり、毎回JOINして計算するのは非効率
- **過度な非正規化**: データ整合性の維持が困難になり、更新時の不整合リスクが高まる

---

### 2. 主キー戦略

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| AUTO_INCREMENT | 連番の整数ID | シンプル、インデックス効率良い | 分散環境で衝突リスク |
| UUID | ランダムな一意識別子 | 分散環境で安全 | インデックス効率低下、可読性低い |
| ULID | 時系列ソート可能なUUID | 時系列ソート可能 | 実装がやや複雑 |
| 複合キー | 複数カラムの組み合わせ | ビジネスキーを直接表現 | JOINが複雑、変更に弱い |

#### 採用した方針
**AUTO_INCREMENT（bigint unsigned）** - 全テーブルで`$table->id()`を使用

#### 選択理由

1. **Laravel/Eloquent標準との親和性**
   - Laravelの`$table->id()`メソッドは`bigint unsigned AUTO_INCREMENT`を生成
   - Eloquentの規約に従うことで、ORMの恩恵を最大限に活用

2. **インデックス効率**
   - 連番IDはB-Treeインデックスと相性が良く、挿入時のページ分割が発生しにくい
   - bigint（8バイト）はUUID（16バイト）と比較してストレージ効率が良い

3. **運用上の利便性**
   - 連番IDは人間が読みやすく、デバッグや運用時の問い合わせが容易
   - 外部キー参照時のJOINパフォーマンスも優れている

4. **業務キーの分離**
   - `sales_orders.name`（オーダー番号）、`purchases_orders.name`など、業務上のキーは別カラムで管理
   - サロゲートキー（id）とビジネスキー（name等）を分離することで、ビジネスキー変更の柔軟性を確保

#### 不採用とした選択肢の理由
- **UUID**: 現時点では単一DBの構成であり、分散環境での利点を享受できない。インデックス効率低下のデメリットが大きい
- **複合キー**: ORMとの相性が悪く、外部キー参照が複雑になる。将来のスキーマ変更にも弱い
- **ULID**: 実装の複雑さに対して、現時点では十分なメリットがない

---

### 3. インデックス戦略

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 必要最小限 | 主キー＋外部キーのみ | 更新性能維持 | 検索性能に課題 |
| 積極的に作成 | 検索パターンごとに作成 | 検索高速 | 更新性能低下、ストレージ増加 |
| カバリングインデックス活用 | 必要カラムを含める | テーブルアクセス削減 | インデックスサイズ増大 |

#### 採用した方針
**検索頻度の高いカラムに選択的にインデックスを設定** - 業務上重要な検索パターンを考慮

#### 主要インデックス一覧
| テーブル | インデックス | 種別 | 選択理由 |
| --- | --- | --- | --- |
| partners_partners | name | 単一インデックス | 取引先名での検索が頻発 |
| partners_partners | tax_id, phone, mobile | 単一インデックス | 各種識別子での検索対応 |
| partners_partners | reference, company_registry | 単一インデックス | 参照番号・登記番号での検索 |
| purchases_orders | name, state, priority | 単一インデックス | 発注一覧でのフィルタリング |
| purchases_orders | ordered_at, approved_at, planned_at | 単一インデックス | 日付範囲での検索 |
| purchases_orders | user_id, company_id | 単一インデックス | 担当者・会社別の絞り込み |
| sessions | user_id | 単一インデックス | ユーザーセッション検索 |
| sessions | last_activity | 単一インデックス | セッション有効期限チェック |

#### 選択理由

1. **業務パターンに基づく選択**
   - ERP業務で頻繁に使用される検索パターン（名前検索、ステータス絞り込み、日付範囲検索）に対応
   - 一覧画面でのソート・フィルタリングを高速化

2. **外部キーへの自動インデックス**
   - Laravelの`constrained()`メソッドにより、外部キーカラムには自動的にインデックスが設定される
   - JOINパフォーマンスの基盤を確保

3. **更新性能とのバランス**
   - ERPシステムは参照系が多いが、トランザクション処理も重要
   - 過度なインデックスは避け、実際の検索パターンに基づいて選択的に設定

---

### 4. 論理削除 / 物理削除

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 論理削除 | deleted_atカラムで管理 | データ復旧可能、監査対応 | クエリ複雑化、ストレージ増加 |
| 物理削除 | レコードを完全削除 | シンプル、ストレージ効率 | 復旧不可、監査対応困難 |
| アーカイブ方式 | 削除時に別テーブルへ移動 | 本テーブルはシンプル、履歴保持 | 実装複雑 |

#### 採用した方針
**ハイブリッド方式** - 主要エンティティは論理削除、関連・中間テーブルは状況に応じて物理削除も許容

#### テーブル別の適用
| テーブル | 削除方式 | 選択理由 |
| --- | --- | --- |
| users | 論理削除 | ユーザー情報は監査証跡として必須保持 |
| partners_partners | 論理削除 | 取引先情報は過去取引の参照に必要 |
| products_products | 論理削除 | 商品情報は過去明細の参照に必要 |
| sales_orders | 論理削除 | 受注データは監査・会計上保持必須 |
| purchases_orders | 物理削除なし（timestampsのみ） | 発注データは監査上保持必須 |
| inventories_moves | 物理削除なし | 在庫移動履歴は追跡可能性のため保持 |
| employees_employees | 論理削除 | 従業員情報は人事履歴として保持 |
| projects_projects | 論理削除 | プロジェクト履歴の参照に必要 |
| blogs_posts | 論理削除 | コンテンツの復元可能性を確保 |
| accounts_account_moves | 物理削除なし | 会計仕訳は法的に削除不可 |

#### 選択理由

1. **監査要件への対応**
   - ERPシステムでは内部統制・外部監査への対応が求められる
   - 誰が・いつ・何を削除したかの追跡可能性が必要

2. **データ復元の可能性**
   - 誤削除時の復元を可能にすることで、運用リスクを軽減
   - LaravelのSoftDeletesトレイトにより、`withTrashed()`で削除済みデータも参照可能

3. **参照整合性の維持**
   - 受注明細から商品を参照する場合、商品が物理削除されると参照不能になる
   - 論理削除により、過去データの整合性を維持

4. **Laravelエコシステムとの親和性**
   - `softDeletes()`メソッドとSoftDeletesトレイトの標準サポート
   - Filamentの管理画面でも論理削除をサポート

#### 不採用とした選択肢の理由
- **全テーブル物理削除**: 監査要件を満たせず、過去データの参照が不可能になる
- **全テーブルアーカイブ**: 実装複雑化に対してメリットが限定的。Laravelの標準機能で十分対応可能

---

### 5. 履歴・監査データ管理

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 履歴テーブル | 変更前後を別テーブルに保存 | シンプル、検索容易 | ストレージ増加 |
| 変更ログ | JSON等で変更差分を記録 | 柔軟、差分のみ保存 | 検索・復元が複雑 |
| イベントソーシング | イベントの積み重ねで状態管理 | 完全な追跡可能性 | 実装・運用が複雑 |
| created_at/updated_atのみ | 最終更新日時のみ記録 | シンプル | 変更履歴は追えない |

#### 採用した方針
**created_at/updated_at + creator_id + Chatter/LogActivity トレイト** による監査証跡

#### 履歴管理対象
| テーブル | 管理方式 | 保持期間 | 選択理由 |
| --- | --- | --- | --- |
| 全テーブル | created_at/updated_at | 無期限 | 基本的な監査証跡 |
| 主要エンティティ | creator_id | 無期限 | 作成者の追跡 |
| Productモデル等 | HasLogActivity | 無期限 | 変更履歴の詳細追跡 |
| 主要モデル | HasChatter | 無期限 | コメント・活動履歴 |

#### 選択理由

1. **Laravel標準機能の活用**
   - `timestamps()`による`created_at`/`updated_at`の自動管理
   - 追加実装なしで基本的な監査証跡を確保

2. **creator_idによる作成者追跡**
   - 全ての主要テーブルに`creator_id`カラムを設け、作成者を記録
   - データの出所と責任の明確化

3. **HasLogActivityトレイト**
   - Productモデルの`$logAttributes`配列で追跡対象を定義
   - 変更履歴を自動的に記録し、監査要件に対応

4. **HasChatterトレイト**
   - チャット機能による活動履歴の管理
   - コメント、メンション、ファイル添付などの協業機能

5. **シンプルさとコストのバランス**
   - イベントソーシングほど複雑でなく、実装・運用コストを抑制
   - ERPとして必要十分な監査証跡を確保

---

### 6. リレーション設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 外部キー制約あり | DB側で参照整合性を保証 | 整合性保証、不正データ防止 | 柔軟性低下、パフォーマンス影響 |
| 外部キー制約なし | アプリ側で整合性管理 | 柔軟、パフォーマンス良好 | 不整合リスク、アプリ責任増大 |
| 一部のみ制約 | 重要なリレーションのみ制約 | バランス良い | 一貫性が低下 |

#### 採用した方針
**外部キー制約あり** - 全ての関連に対して適切な外部キー制約を設定

#### 主要リレーション
| 親テーブル | 子テーブル | 関係 | ON DELETE | 選択理由 |
| --- | --- | --- | --- | --- |
| companies | sales_orders | 1:N | RESTRICT | 受注は会社必須、会社削除時はブロック |
| partners_partners | sales_orders | 1:N | RESTRICT | 受注は取引先必須、取引先削除時はブロック |
| users | sales_orders (user_id) | 1:N | SET NULL | 担当者削除時もオーダーは保持 |
| users | sales_orders (creator_id) | 1:N | SET NULL | 作成者削除時もオーダーは保持 |
| products_products | sales_order_lines | 1:N | 未指定 | 商品情報は明細に必要 |
| products_categories | products_products | 1:N | CASCADE | カテゴリ削除時は商品も削除 |
| products_products | products_products (parent_id) | 自己参照 | CASCADE | 親商品削除時はバリエーションも削除 |
| unit_of_measures | products_products | 1:N | RESTRICT | 単位は必須、使用中は削除不可 |
| inventories_locations | inventories_moves | 1:N | RESTRICT | ロケーション削除時は移動データ保護 |
| employees_employees | employees_employees (parent_id) | 自己参照 | SET NULL | 上司削除時は関係をクリア |

#### 選択理由

1. **データ整合性の確保**
   - ERPシステムではデータの整合性が業務の正確性に直結
   - DB層での制約により、アプリケーションバグによる不整合を防止

2. **ON DELETE戦略の使い分け**
   - **RESTRICT**: 重要な参照関係（会社、取引先、通貨など）の保護
   - **CASCADE**: 親子関係の強い場合（カテゴリ-商品、親商品-バリエーション）
   - **SET NULL**: 補助的な参照（担当者、作成者など）

3. **参照整合性エラーの早期発見**
   - 外部キー制約違反はDB層でエラーとなり、問題の早期発見が可能
   - アプリケーション層でのチェック漏れを防止

---

### 7. データ型選択

#### 想定される選択肢（代表例）

| 用途 | 選択肢 | メリット | デメリット |
| --- | --- | --- | --- |
| 金額 | DECIMAL / BIGINT（最小単位） | 精度保証 / 計算シンプル | 表示時変換必要 / 精度考慮必要 |
| 日時 | DATETIME / TIMESTAMP | 範囲広い / タイムゾーン対応 | TZ非対応 / 2038年問題 |
| 真偽値 | BOOLEAN / TINYINT | 意味明確 / 互換性高い | DB依存 / 意味曖昧 |
| 長文 | TEXT / VARCHAR(MAX) | 柔軟 / インデックス可 | インデックス不可 / サイズ制限 |

#### 採用した方針
**Laravelの標準データ型マッピング**を基本としつつ、業務要件に応じて適切な型を選択

#### 特筆すべき型選択
| カラム例 | 選択した型 | 想定した他の選択肢 | 選択理由 |
| --- | --- | --- | --- |
| price, cost | decimal(15,4) | float, integer | 金額の精度保証、小数点以下4桁まで対応 |
| amount_untaxed | decimal(15,4) | decimal(10,2), float | 大口取引の金額にも対応、通貨によっては小数点以下複数桁必要 |
| currency_rate | decimal(15,6) | decimal(15,4), float | 為替レートの精度確保、6桁まで対応 |
| state | varchar(255) | enum, tinyint | 柔軟な状態追加が可能、コード可読性向上 |
| images | json | text, 別テーブル | Laravel/MySQLのJSON型サポートを活用、構造化データとして扱える |
| description | text | varchar | 長文対応、サイズ制限なし |
| email | varchar(255) | varchar(100) | RFC 5321準拠の最大長を考慮 |
| is_active | boolean | tinyint(1) | 意味が明確、Laravelのキャストで真偽値として扱える |

---

## テーブル/データモデルの存在理由（業務観点）

### users（ユーザー）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-001 | ユーザー認証・認可 | システムへのログイン管理、権限に基づくアクセス制御 |
| BR-002 | 監査証跡 | 誰がどの操作を行ったかの追跡可能性 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | ユーザー登録時、招待メール受諾時、管理者による手動登録 |
| データ参照先 | ログイン画面、権限チェック、監査ログ、担当者表示 |
| データの業務上の重要性 | ユーザー情報がないとシステムにログインできず、全ての業務操作が不可能になる |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 複数ユーザーが同一システムを利用する際の識別・認証が必要 | 認証がないと不正アクセスや責任追跡が不可能（重大） |
| 2 | ユーザーの権限に応じた機能制限が必要 | 権限管理がないと機密情報漏洩や誤操作のリスク（重大） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 識別・認証 | email（ユニーク）とpassword（ハッシュ化）による認証 | 一意なユーザー識別と安全な認証を実現 |
| 権限管理 | is_activeフラグと論理削除による状態管理、spatie/permissionとの連携 | 柔軟な権限制御とアカウント管理 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | Excel等での手動管理 | 開発コストゼロ | 認証不可、セキュリティリスク大 | 不可 |
| B | 単一ロールのみ | シンプル | 複雑な権限要件に対応不可 | 低 |
| C（採用） | ユーザーテーブル＋spatie/permission | 柔軟な権限管理 | 学習コストあり | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: 複数の部署・役職のユーザーが同一システムを利用するERP業務では、ユーザーごとの認証と権限管理が必須
2. **データ整合性の観点**: emailのユニーク制約により、重複登録を防止し、一意なユーザー識別を保証
3. **将来の業務変化への対応**: 論理削除により退職者データを保持し、過去の監査証跡を維持可能

#### 不採用とした案の理由
- **案A（手動管理）**: ERPシステムでは認証・権限管理が必須であり、手動管理は不可能
- **案B（単一ロール）**: 複数部署、複数会社を横断するERP業務では、細かな権限制御が必要

---

### partners_partners（取引先）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-010 | 取引先マスタ管理 | 顧客・仕入先情報の一元管理 |
| BR-011 | 取引先区分管理 | 顧客・仕入先・両方の区分管理 |
| BR-012 | 親子関係管理 | 本社・支社、グループ会社の関係管理 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 取引先登録画面、受注登録時の新規作成、Web問い合わせ連携 |
| データ参照先 | 受注・発注登録、請求書発行、配送先選択、売掛・買掛管理 |
| データの業務上の重要性 | 取引先情報がないと受注・発注処理ができず、売上・仕入の計上が不可能 |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 顧客情報が分散管理されており、最新情報の把握が困難 | 誤った連絡先への送付、機会損失（中程度） |
| 2 | 同一取引先が顧客・仕入先両方の場合の二重管理 | データ不整合、管理工数増大（中程度） |
| 3 | グループ会社との取引条件の統一管理が困難 | 価格設定ミス、与信管理の漏れ（中程度） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 情報分散 | 単一テーブルでの一元管理、sub_typeによる区分 | 最新情報の一元参照、データ品質向上 |
| 二重管理 | 同一レコードで顧客・仕入先フラグを管理 | 重複排除、管理工数削減 |
| グループ会社管理 | parent_idによる親子関係の表現 | 階層的な取引条件管理が可能 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | Excel管理継続 | 開発コストゼロ | 複数部署での同時更新困難、データ不整合 | 低 |
| B | 顧客・仕入先を別テーブル | 明確な分離 | 両方の場合に重複登録が必要 | 中 |
| C（採用） | 統合パートナーテーブル＋区分フラグ | 一元管理、親子関係表現可能 | テーブル設計がやや複雑 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: 同一取引先が顧客・仕入先両方となるケースは製造業・卸売業で頻出であり、統合管理が合理的
2. **データ整合性の観点**: 住所・連絡先の一元管理により、変更時の更新漏れを防止
3. **将来の業務変化への対応**: parent_idにより、M&Aや組織再編にも柔軟に対応可能

#### 不採用とした案の理由
- **案A（Excel管理継続）**: 受発注システムとの連携が困難、最新情報の共有ができない
- **案B（別テーブル）**: 両方の取引がある場合に二重管理となり、データ整合性維持が困難

---

### products_products（商品）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-020 | 商品マスタ管理 | 商品情報の登録・更新・参照 |
| BR-021 | バリエーション商品管理 | サイズ・色違いなどのバリエーション管理 |
| BR-022 | 販売・購買の商品区分 | 販売可能/購買可能の商品区分管理 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 商品登録画面、仕入時の新規登録、ECサイト連携 |
| データ参照先 | 受注明細、発注明細、在庫管理、価格表、帳票 |
| データの業務上の重要性 | 商品情報がないと受注・発注明細が作成できず、販売・仕入業務が停止 |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 商品情報がExcel等で分散管理されている | 価格・仕様の不整合、在庫管理困難（重大） |
| 2 | バリエーション商品の体系的管理ができていない | 在庫把握困難、販売機会損失（中程度） |
| 3 | 販売用・購買用の商品情報が別管理されている | 原価管理困難、利益把握ができない（重大） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 情報分散 | 単一商品テーブルでの一元管理 | 最新情報の共有、データ品質向上 |
| バリエーション管理 | parent_idによる親子関係＋属性・属性値テーブル | 体系的なバリエーション管理 |
| 販売・購買統合 | enable_sales/enable_purchaseフラグ | 同一商品で販売・購買両方の設定が可能 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | Excel管理継続 | 開発コストゼロ | 在庫・価格管理との連携困難 | 低 |
| B | 販売商品・購買商品を別テーブル | 明確な分離 | 同一商品の二重管理、原価計算困難 | 中 |
| C（採用） | 統合商品テーブル＋属性モデル | 一元管理、バリエーション対応 | スキーマがやや複雑 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: 販売と購買で同一商品を扱うケースが多く、原価・売価の一元管理が必要
2. **データ整合性の観点**: 商品名・仕様の変更が全ての関連データに反映される
3. **将来の業務変化への対応**: 属性モデルにより、新しいバリエーション軸を追加可能

#### 不採用とした案の理由
- **案A（Excel管理継続）**: 在庫管理・受発注システムとの連携が不可能
- **案B（別テーブル）**: 原価と売価の紐付けができず、利益管理が困難

---

### sales_orders / sales_order_lines（受注・受注明細）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-030 | 受注管理 | 顧客からの注文の登録・管理 |
| BR-031 | 見積から受注への変換 | 見積書を受注として確定 |
| BR-032 | 請求連携 | 受注から請求書発行へのフロー |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 受注登録画面、見積確定時、ECサイト連携 |
| データ参照先 | 受注一覧、出荷指示、請求書発行、売上分析 |
| データの業務上の重要性 | 受注データがないと出荷・請求ができず、売上計上が不可能 |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 受注情報が紙やExcelで管理されている | 出荷漏れ、請求遅延、二重入力（重大） |
| 2 | 受注状況のリアルタイム把握が困難 | 納期遅延対応の遅れ、顧客満足度低下（重大） |
| 3 | 受注から請求への連携が手作業 | ミス発生、工数増大（中程度） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 情報分散 | 受注オーダーと明細の正規化モデル | データの一元管理、検索性向上 |
| 状況把握 | stateカラムによるステータス管理 | リアルタイムな進捗把握 |
| 請求連携 | invoice_status、sales_order_invoices連携テーブル | 自動連携による工数削減 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | Excel管理継続 | 開発コストゼロ | リアルタイム性なし、集計困難 | 低 |
| B | 単一テーブルで全情報管理 | シンプル | 明細数の変動に対応困難 | 中 |
| C（採用） | ヘッダー・明細の正規化モデル | 業務フローに適合、拡張性高い | JOINが必要 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: 受注業務は「1受注に対してN明細」という構造であり、正規化モデルが自然
2. **データ整合性の観点**: 金額サマリーを非正規化で保持し、明細との整合性はアプリケーション層で保証
3. **将来の業務変化への対応**: UTM連携、テンプレート機能など、将来の拡張を見据えた設計

#### 不採用とした案の理由
- **案A（Excel管理継続）**: 在庫連携、請求連携ができず、業務効率化の効果がない
- **案B（単一テーブル）**: 明細数が可変のため、カラム数が不定となり非現実的

---

### inventories_moves（在庫移動）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-040 | 入出荷管理 | 入荷・出荷の記録と追跡 |
| BR-041 | 在庫移動管理 | 倉庫間・ロケーション間の移動管理 |
| BR-042 | トレーサビリティ | 在庫の移動履歴追跡 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 入荷検品、出荷処理、棚卸、倉庫間移動 |
| データ参照先 | 在庫一覧、移動履歴、ロット追跡、廃棄処理 |
| データの業務上の重要性 | 在庫移動データがないと正確な在庫数が把握できず、欠品・過剰在庫のリスク |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 在庫の動きがリアルタイムで把握できない | 欠品による販売機会損失、過剰在庫（重大） |
| 2 | どの在庫がどこにあるか追跡できない | 出荷遅延、ピッキングミス（中程度） |
| 3 | 不良品・廃棄の管理ができていない | 在庫差異、損失把握困難（中程度） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| リアルタイム把握 | 全ての在庫移動をmovesテーブルに記録 | 在庫の正確な把握 |
| ロケーション追跡 | source_location_id/destination_location_idによる移動元・先管理 | ピッキング効率向上 |
| 廃棄管理 | is_scraped、scrap_id連携 | 廃棄処理の追跡可能性 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | 在庫数のみ管理（移動履歴なし） | シンプル | 追跡不可、差異原因不明 | 低 |
| B | 移動履歴テーブルのみ | 完全な履歴 | 現在在庫数の計算が必要 | 中 |
| C（採用） | 移動テーブル＋在庫数量テーブル | 履歴と現在値の両方を管理 | やや複雑 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: 在庫管理では「現在の数量」と「どう動いたか」の両方が必要
2. **データ整合性の観点**: 移動履歴から在庫数を再計算可能であり、整合性を検証できる
3. **将来の業務変化への対応**: ロット管理、シリアル番号管理への拡張が容易

#### 不採用とした案の理由
- **案A（在庫数のみ）**: 差異発生時の原因追跡が不可能、監査要件を満たせない
- **案B（履歴のみ）**: 現在在庫数の参照時に毎回集計が必要でパフォーマンス懸念

---

### employees_employees（従業員）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-050 | 従業員マスタ管理 | 従業員の基本情報管理 |
| BR-051 | 組織構造管理 | 部署・上司・コーチの関係管理 |
| BR-052 | 休暇・勤怠連携 | 休暇申請、勤怠管理との連携 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 入社時の従業員登録、人事異動、情報更新 |
| データ参照先 | 組織図、休暇申請、勤怠管理、給与計算、プロジェクトアサイン |
| データの業務上の重要性 | 従業員情報がないと人事・労務管理ができず、給与計算や休暇管理が不可能 |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 従業員情報が人事部門でのみ管理されている | 他部門での情報参照困難、二重入力（中程度） |
| 2 | 組織の階層構造が把握しづらい | 承認フロー設定困難、報告ライン不明確（中程度） |
| 3 | 退職者情報の管理が不十分 | 過去データの参照困難、監査対応の問題（軽度） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 情報分散 | 単一従業員テーブルでの一元管理 | 全部門での情報共有 |
| 組織構造 | parent_id（上司）、coach_id、department_idによる関係表現 | 明確な組織構造の表現 |
| 退職者管理 | 論理削除＋departure_reason_id、departure_date | 退職者情報の保持と追跡 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | Excel管理継続 | 開発コストゼロ | 休暇・勤怠システムとの連携困難 | 低 |
| B | ユーザーテーブルに従業員情報を統合 | シンプル | 非システムユーザーの従業員管理困難 | 中 |
| C（採用） | ユーザーと従業員を分離、user_idで連携 | 柔軟、非ユーザー従業員も管理可能 | 2テーブル管理 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: システムユーザーでない従業員（工場作業員など）も管理対象となるため、分離が必要
2. **データ整合性の観点**: user_idでの連携により、ユーザー・従業員の紐付けを保証
3. **将来の業務変化への対応**: スキル管理、履歴書管理などの拡張機能に対応可能

#### 不採用とした案の理由
- **案A（Excel管理継続）**: 休暇申請、勤怠管理との連携が不可能
- **案B（統合）**: システムを利用しない従業員の管理ができない

---

### accounts_account_moves（会計仕訳）

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-060 | 仕訳管理 | 会計仕訳の登録・承認・転記 |
| BR-061 | 請求書・支払管理 | 請求書発行、支払登録 |
| BR-062 | 会計監査対応 | 仕訳の追跡可能性、改ざん防止 |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | 請求書発行、支払登録、経費精算、決算処理 |
| データ参照先 | 仕訳帳、総勘定元帳、試算表、財務諸表 |
| データの業務上の重要性 | 会計仕訳データがないと財務報告ができず、法的要件を満たせない |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | 手入力による仕訳ミスが発生 | 財務報告の誤り、修正工数（重大） |
| 2 | 仕訳の改ざんリスクがある | 監査指摘、法令違反リスク（重大） |
| 3 | 受発注システムとの二重入力 | 工数増大、データ不整合（中程度） |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| 入力ミス | 受発注からの自動仕訳生成 | ミス削減、工数削減 |
| 改ざん防止 | inalterable_hash、posted_beforeフラグ | 改ざん検知、監査対応 |
| 二重入力 | 受注・発注との連携テーブル | 自動連携による一元管理 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | 外部会計ソフトとのCSV連携 | 既存システム活用 | 二重入力、リアルタイム性なし | 低 |
| B | シンプルな仕訳テーブル | 開発コスト低 | 監査要件を満たせない | 中 |
| C（採用） | 監査対応機能付き仕訳テーブル | 法的要件対応、追跡可能 | 設計複雑 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: ERPとして会計機能を内包することで、受発注から会計までの一気通貫を実現
2. **データ整合性の観点**: inalterable_hashにより仕訳の改ざんを検知可能
3. **将来の業務変化への対応**: 自動仕訳、複数通貨対応など、会計要件の拡張に対応

#### 不採用とした案の理由
- **案A（CSV連携）**: リアルタイム性がなく、データ整合性の維持が困難
- **案B（シンプル）**: 監査要件（改ざん検知など）を満たせない

---

## 将来の拡張性考慮

### 想定される変更
1. **マルチテナント対応の強化**
   - 現状: company_idによる会社単位の分離
   - 拡張: データベース分離、スキーマ分離への移行も可能な設計
   - 対応方針: company_id外部キーの一貫した設計により、将来のテナント分離に対応

2. **多言語・多通貨対応**
   - 現状: currency_id、languageカラムによる基本対応
   - 拡張: 商品名・説明の多言語化、為替レート履歴管理
   - 対応方針: JSON型カラム、翻訳テーブルの追加で対応可能

3. **ワークフロー機能の拡張**
   - 現状: stateカラムによるシンプルな状態管理
   - 拡張: 承認フロー、多段階承認
   - 対応方針: 承認履歴テーブル、ワークフロー定義テーブルの追加

### スケーラビリティ

1. **読み取りスケーリング**
   - インデックス戦略により、読み取りクエリのパフォーマンスを確保
   - 将来的にはリードレプリカの導入も可能

2. **パーティショニング検討対象**
   - inventories_moves: 日付によるレンジパーティショニング
   - accounts_account_moves: 年度によるレンジパーティショニング
   - これらのテーブルは時系列データであり、古いデータのアーカイブも検討可能

3. **シャーディング検討**
   - company_idをシャーディングキーとした水平分割が可能な設計
   - 現状のスケールでは不要だが、将来の大規模展開時に対応可能

---

## 備考

### 設計の一貫性

本データベース設計では、以下の点で一貫性を保っています：

1. **命名規則**: モジュール名_テーブル名（例: partners_partners、products_products）
2. **主キー**: 全テーブルでid（bigint unsigned AUTO_INCREMENT）
3. **監査カラム**: creator_id、created_at、updated_atを全テーブルに設置
4. **論理削除**: 主要エンティティにdeleted_atカラム
5. **会社分離**: マルチテナント対応のためcompany_idを設置

### Laravelエコシステムとの親和性

- Eloquent ORM の規約に準拠した設計
- spatie/laravel-permission との連携を考慮した権限テーブル
- Filament管理画面との親和性（SoftDeletes、HasChatter等）

### 参照した情報源

- マイグレーションファイル（plugins/webkul/*/database/migrations/）
- モデル定義（plugins/webkul/*/src/Models/）
- データベース設計書（docs/code-to-docs/データベース設計書/）
