# API設計書

## 概要

本ドキュメントは、Apache Sparkプロジェクトが提供するREST APIの設計仕様を記載する。SparkにはSparkアプリケーションのモニタリング情報を提供する「Status API（Monitoring API）」と、Standaloneクラスタモードでジョブの投入・管理を行う「REST Submission API」の2系統のAPIが存在する。いずれもJAX-RS（Jersey）またはServletベースで実装されており、JSON形式でデータを返却する。

## 共通仕様

### ベースURL

**Status API（Monitoring API）**
```
http://<spark-host>:<port>/api/v1
```

**REST Submission API（Standalone Cluster Mode）**
```
http://<master-host>:<rest-port>/v1/submissions
```

**Prometheus Metrics API**
```
http://<spark-host>:<port>/metrics/executors/prometheus
```

### 認証方式

SparkのREST APIは、Spark UIのセキュリティ設定に依存する。`spark.ui.filters`や`spark.acls.enable`による認証・認可が有効な場合、HTTPリクエストのRemoteUserに基づいたアクセス制御が適用される。Status APIではBaseAppResourceにてUIのView権限チェックが行われる。REST Submission APIでは`spark.master.rest.filters`によるフィルタベースの認証が利用可能。

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Content-Type | - | application/json（Submission APIのPOSTリクエスト時） |
| Accept | - | application/json |

### 共通エラーレスポンス

**Status API**

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - クエリパラメータ不正（BadParameterException） |
| 403 | Forbidden - UIの閲覧権限なし（ForbiddenException） |
| 404 | Not Found - アプリケーション/リソース未検出（NotFoundException） |
| 503 | Service Unavailable - 機能が利用不可（ServiceUnavailable） |

**REST Submission API**

| ステータスコード | 説明 |
| --- | --- |
| 200 | OK - リクエスト処理成功 |
| 400 | Bad Request - リクエスト不正またはバリデーションエラー |
| 468 | Unknown Protocol Version - サポートされないプロトコルバージョン |
| 500 | Internal Server Error - サーバー内部エラー |

## API一覧

### Status API（Monitoring API）

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Version | /api/v1/version | GET | Sparkバージョン情報を取得 |
| Applications | /api/v1/applications | GET | アプリケーション一覧を取得 |
| Applications | /api/v1/applications/{appId} | GET | 指定アプリケーションの情報を取得 |
| Applications | /api/v1/applications/{appId}/{attemptId} | GET | 指定アプリケーション試行の情報を取得 |
| Jobs | /api/v1/applications/{appId}/jobs | GET | ジョブ一覧を取得 |
| Jobs | /api/v1/applications/{appId}/jobs/{jobId} | GET | 指定ジョブの情報を取得 |
| Stages | /api/v1/applications/{appId}/stages | GET | ステージ一覧を取得 |
| Stages | /api/v1/applications/{appId}/stages/{stageId} | GET | 指定ステージの情報を取得 |
| Stages | /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId} | GET | 指定ステージ試行の情報を取得 |
| Stages | /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskSummary | GET | タスクメトリクス分布を取得 |
| Stages | /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskList | GET | タスク一覧を取得 |
| Stages | /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskTable | GET | タスクテーブルデータを取得（DataTables用） |
| Executors | /api/v1/applications/{appId}/executors | GET | アクティブExecutor一覧を取得 |
| Executors | /api/v1/applications/{appId}/allexecutors | GET | 全Executor一覧を取得（終了済み含む） |
| Executors | /api/v1/applications/{appId}/executors/{executorId}/threads | GET | Executorのスレッドダンプを取得 |
| Threads | /api/v1/applications/{appId}/threads | GET | タスクのスレッドダンプを取得 |
| Processes | /api/v1/applications/{appId}/allmiscellaneousprocess | GET | 雑多なプロセス一覧を取得 |
| Storage | /api/v1/applications/{appId}/storage/rdd | GET | RDDストレージ一覧を取得 |
| Storage | /api/v1/applications/{appId}/storage/rdd/{rddId} | GET | 指定RDDのストレージ情報を取得 |
| Environment | /api/v1/applications/{appId}/environment | GET | アプリケーション環境情報を取得 |
| Logs | /api/v1/applications/{appId}/logs | GET | イベントログをダウンロード |
| SQL | /api/v1/applications/{appId}/sql | GET | SQL実行一覧を取得 |
| SQL | /api/v1/applications/{appId}/sql/{executionId} | GET | 指定SQL実行の情報を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/statistics | GET | ストリーミング統計情報を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/receivers | GET | レシーバー一覧を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/receivers/{streamId} | GET | 指定レシーバーの情報を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/batches | GET | バッチ一覧を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/batches/{batchId} | GET | 指定バッチの情報を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/batches/{batchId}/operations | GET | バッチの出力操作一覧を取得 |
| Streaming | /api/v1/applications/{appId}/streaming/batches/{batchId}/operations/{outputOpId} | GET | 指定出力操作の情報を取得 |
| Prometheus | /metrics/executors/prometheus | GET | Executorメトリクスをprometheus形式で取得 |

### REST Submission API（Standalone Cluster Mode）

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Submission | /v1/submissions/create | POST | アプリケーションを投入 |
| Submission | /v1/submissions/kill/{submissionId} | POST | 指定サブミッションを終了 |
| Submission | /v1/submissions/killall | POST | 全サブミッションを終了 |
| Submission | /v1/submissions/status/{submissionId} | GET | サブミッションのステータスを取得 |
| Submission | /v1/submissions/clear | POST | 完了済みドライバー・アプリケーションをクリア |
| Submission | /v1/submissions/readyz | GET | Masterの準備状態を確認 |

## 各APIエンドポイント定義

### Version

#### 1. Sparkバージョン取得

Sparkのバージョン情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/version` |
| 認証 | 不要 |
| 権限 | なし |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "spark": "4.0.0"
}
```

---

### Applications

#### 2. アプリケーション一覧取得

登録されたアプリケーションの一覧を取得する。HistoryServerまたはSparkUI経由でアクセスする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| status | JList[ApplicationStatus] | - | - | フィルタ条件（RUNNING, COMPLETED） |
| minDate | SimpleDateParam | - | 2010-01-01 | 開始日時の下限 |
| maxDate | SimpleDateParam | - | 3000-01-01 | 開始日時の上限 |
| minEndDate | SimpleDateParam | - | 2010-01-01 | 終了日時の下限 |
| maxEndDate | SimpleDateParam | - | 3000-01-01 | 終了日時の上限 |
| limit | Integer | - | - | 返却件数の上限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": "app-20240101120000-0000",
    "name": "MySparkApp",
    "coresGranted": 4,
    "maxCores": 8,
    "coresPerExecutor": 2,
    "memoryPerExecutorMB": 1024,
    "attempts": [
      {
        "attemptId": "1",
        "startTime": "2024-01-01T12:00:00.000GMT",
        "endTime": "2024-01-01T13:00:00.000GMT",
        "lastUpdated": "2024-01-01T13:00:00.000GMT",
        "duration": 3600000,
        "sparkUser": "user1",
        "completed": true,
        "appSparkVersion": "4.0.0",
        "startTimeEpoch": 1704106800000,
        "endTimeEpoch": 1704110400000,
        "lastUpdatedEpoch": 1704110400000
      }
    ]
  }
]
```

---

#### 3. アプリケーション情報取得

指定したアプリケーションの詳細情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": "app-20240101120000-0000",
  "name": "MySparkApp",
  "coresGranted": 4,
  "maxCores": 8,
  "coresPerExecutor": 2,
  "memoryPerExecutorMB": 1024,
  "attempts": [...]
}
```

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（アプリケーションが存在しない場合）

---

#### 4. アプリケーション試行情報取得

指定したアプリケーションの特定の試行の情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/{attemptId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| attemptId | string | ○ | 試行ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "attemptId": "1",
  "startTime": "2024-01-01T12:00:00.000GMT",
  "endTime": "2024-01-01T13:00:00.000GMT",
  "lastUpdated": "2024-01-01T13:00:00.000GMT",
  "duration": 3600000,
  "sparkUser": "user1",
  "completed": true,
  "appSparkVersion": "4.0.0"
}
```

---

### Jobs

#### 5. ジョブ一覧取得

指定アプリケーションのジョブ一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/jobs` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| status | JList[JobExecutionStatus] | - | - | フィルタ条件（RUNNING, SUCCEEDED, FAILED, UNKNOWN） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "jobId": 0,
    "name": "count at MyApp.scala:42",
    "description": "My Job Description",
    "submissionTime": "2024-01-01T12:00:00.000GMT",
    "completionTime": "2024-01-01T12:05:00.000GMT",
    "stageIds": [0, 1],
    "jobGroup": "group1",
    "jobTags": ["tag1"],
    "status": "SUCCEEDED",
    "numTasks": 100,
    "numActiveTasks": 0,
    "numCompletedTasks": 100,
    "numSkippedTasks": 0,
    "numFailedTasks": 0,
    "numKilledTasks": 0,
    "numCompletedIndices": 100,
    "numActiveStages": 0,
    "numCompletedStages": 2,
    "numSkippedStages": 0,
    "numFailedStages": 0,
    "killedTasksSummary": {}
  }
]
```

---

#### 6. ジョブ情報取得

指定したジョブの詳細情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/jobs/{jobId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| jobId | int | ○ | ジョブID（数値） |

**レスポンス（成功時）**

ステータスコード: `200 OK`（レスポンス形式はジョブ一覧の要素と同一）

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（ジョブが存在しない場合）

---

### Stages

#### 7. ステージ一覧取得

指定アプリケーションのステージ一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| status | JList[StageStatus] | - | - | フィルタ条件（ACTIVE, COMPLETE, PENDING, FAILED） |
| details | boolean | - | false | タスク詳細を含めるか |
| withSummaries | boolean | - | false | タスクメトリクスサマリーを含めるか |
| quantiles | string | - | 0.0,0.25,0.5,0.75,1.0 | 分位数指定（カンマ区切り） |
| taskStatus | JList[TaskStatus] | - | - | タスクステータスでフィルタ（RUNNING, SUCCESS, FAILED, KILLED, UNKNOWN） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "status": "COMPLETE",
    "stageId": 0,
    "attemptId": 0,
    "numTasks": 10,
    "numActiveTasks": 0,
    "numCompleteTasks": 10,
    "numFailedTasks": 0,
    "numKilledTasks": 0,
    "numCompletedIndices": 10,
    "submissionTime": "2024-01-01T12:00:00.000GMT",
    "firstTaskLaunchedTime": "2024-01-01T12:00:01.000GMT",
    "completionTime": "2024-01-01T12:00:30.000GMT",
    "executorDeserializeTime": 100,
    "executorRunTime": 5000,
    "resultSize": 1024,
    "jvmGcTime": 50,
    "memoryBytesSpilled": 0,
    "diskBytesSpilled": 0,
    "inputBytes": 10240,
    "inputRecords": 1000,
    "outputBytes": 5120,
    "outputRecords": 500,
    "shuffleReadBytes": 0,
    "shuffleWriteBytes": 0,
    "name": "count at MyApp.scala:42",
    "details": "org.apache.spark...",
    "schedulingPool": "default",
    "rddIds": [0],
    "accumulatorUpdates": [],
    "killedTasksSummary": {},
    "resourceProfileId": 0,
    "isShufflePushEnabled": false,
    "shuffleMergersCount": 0
  }
]
```

---

#### 8. ステージ情報取得

指定したステージの全試行情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages/{stageId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| stageId | int | ○ | ステージID（数値） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| details | boolean | - | true | タスク詳細を含めるか |
| taskStatus | JList[TaskStatus] | - | - | タスクステータスでフィルタ |
| withSummaries | boolean | - | false | タスクメトリクスサマリーを含めるか |
| quantiles | string | - | 0.0,0.25,0.5,0.75,1.0 | 分位数指定 |

**レスポンス（成功時）**

ステータスコード: `200 OK`（StageDataの配列）

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（ステージが存在しない場合）

---

#### 9. ステージ試行情報取得

指定したステージの特定の試行の詳細情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| stageId | int | ○ | ステージID（数値） |
| stageAttemptId | int | ○ | ステージ試行ID（数値） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| details | boolean | - | true | タスク詳細を含めるか |
| taskStatus | JList[TaskStatus] | - | - | タスクステータスでフィルタ |
| withSummaries | boolean | - | false | タスクメトリクスサマリーを含めるか |
| quantiles | string | - | 0.0,0.25,0.5,0.75,1.0 | 分位数指定 |

**レスポンス（成功時）**

ステータスコード: `200 OK`（単一のStageDataオブジェクト）

---

#### 10. タスクメトリクス分布取得

指定したステージ試行のタスクメトリクスの統計的分布を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskSummary` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| stageId | int | ○ | ステージID |
| stageAttemptId | int | ○ | ステージ試行ID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| quantiles | string | - | 0.05,0.25,0.5,0.75,0.95 | 分位数指定（カンマ区切り） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "quantiles": [0.05, 0.25, 0.5, 0.75, 0.95],
  "duration": [10.0, 50.0, 100.0, 200.0, 500.0],
  "executorDeserializeTime": [...],
  "executorRunTime": [...],
  "executorCpuTime": [...],
  "resultSize": [...],
  "jvmGcTime": [...],
  "memoryBytesSpilled": [...],
  "diskBytesSpilled": [...],
  "peakExecutionMemory": [...],
  "inputMetrics": {
    "bytesRead": [...],
    "recordsRead": [...]
  },
  "outputMetrics": {
    "bytesWritten": [...],
    "recordsWritten": [...]
  },
  "shuffleReadMetrics": {
    "readBytes": [...],
    "readRecords": [...]
  },
  "shuffleWriteMetrics": {
    "writeBytes": [...],
    "writeRecords": [...],
    "writeTime": [...]
  }
}
```

---

#### 11. タスク一覧取得

指定したステージ試行のタスク一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskList` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| stageId | int | ○ | ステージID |
| stageAttemptId | int | ○ | ステージ試行ID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| offset | int | - | 0 | 結果の開始位置 |
| length | int | - | 20 | 返却件数 |
| sortBy | TaskSorting | - | ID | ソート基準 |
| status | JList[TaskStatus] | - | - | タスクステータスでフィルタ |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "taskId": 0,
    "index": 0,
    "attempt": 0,
    "partitionId": 0,
    "launchTime": "2024-01-01T12:00:00.000GMT",
    "resultFetchStart": null,
    "duration": 1500,
    "executorId": "1",
    "host": "worker-1",
    "status": "SUCCESS",
    "taskLocality": "PROCESS_LOCAL",
    "speculative": false,
    "accumulatorUpdates": [],
    "errorMessage": null,
    "taskMetrics": {
      "executorDeserializeTime": 10,
      "executorRunTime": 1400,
      "resultSize": 512,
      "jvmGcTime": 5,
      "memoryBytesSpilled": 0,
      "diskBytesSpilled": 0,
      "peakExecutionMemory": 1048576,
      "inputMetrics": { "bytesRead": 1024, "recordsRead": 100 },
      "outputMetrics": { "bytesWritten": 512, "recordsWritten": 50 },
      "shuffleReadMetrics": { "remoteBlocksFetched": 0, "localBlocksFetched": 0, "fetchWaitTime": 0, "remoteBytesRead": 0, "localBytesRead": 0, "recordsRead": 0 },
      "shuffleWriteMetrics": { "bytesWritten": 0, "writeTime": 0, "recordsWritten": 0 }
    },
    "executorLogs": {},
    "schedulerDelay": 50,
    "gettingResultTime": 10
  }
]
```

---

#### 12. タスクテーブルデータ取得

DataTablesライブラリ用のサーバーサイドページング対応タスクテーブルデータを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/stages/{stageId}/{stageAttemptId}/taskTable` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| stageId | int | ○ | ステージID |
| stageAttemptId | int | ○ | ステージ試行ID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| details | boolean | - | true | タスク詳細を含めるか |
| numTasks | string | - | - | DataTables用の総レコード数 |
| search[value] | string | - | - | 検索文字列 |
| start | string | - | - | ページ開始インデックス |
| length | string | - | - | ページサイズ |
| columnNameToSort | string | - | - | ソート列名 |
| order[0][dir] | string | - | - | ソート方向（asc/desc） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "aaData": [...],
  "recordsTotal": "100",
  "recordsFiltered": "50"
}
```

---

### Executors

#### 13. アクティブExecutor一覧取得

現在アクティブなExecutorの一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/executors` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": "1",
    "hostPort": "worker-1:45678",
    "isActive": true,
    "rddBlocks": 5,
    "memoryUsed": 1048576,
    "diskUsed": 0,
    "totalCores": 4,
    "maxTasks": 4,
    "activeTasks": 2,
    "failedTasks": 0,
    "completedTasks": 50,
    "totalTasks": 52,
    "totalDuration": 60000,
    "totalGCTime": 500,
    "totalInputBytes": 10240,
    "totalShuffleRead": 5120,
    "totalShuffleWrite": 2048,
    "isBlacklisted": false,
    "maxMemory": 536870912,
    "addTime": "2024-01-01T12:00:00.000GMT",
    "removeTime": null,
    "removeReason": null,
    "executorLogs": {
      "stdout": "http://worker-1:8081/logPage/?...",
      "stderr": "http://worker-1:8081/logPage/?..."
    },
    "memoryMetrics": {
      "usedOnHeapStorageMemory": 524288,
      "usedOffHeapStorageMemory": 0,
      "totalOnHeapStorageMemory": 536870912,
      "totalOffHeapStorageMemory": 0
    },
    "peakMemoryMetrics": {
      "JVMHeapMemory": 268435456,
      "JVMOffHeapMemory": 67108864
    },
    "attributes": {},
    "resources": {},
    "resourceProfileId": 0,
    "isExcluded": false,
    "excludedInStages": []
  }
]
```

---

#### 14. 全Executor一覧取得

終了済みを含む全Executorの一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/allexecutors` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`（Executor一覧と同じ形式。isActiveがfalseのものも含まれる）

---

#### 15. Executorスレッドダンプ取得

指定したExecutorのスレッドダンプを取得する。HistoryServerからは利用不可。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/executors/{executorId}/threads` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| executorId | string | ○ | ExecutorID（数値 or "driver"） |

**バリデーション**

- executorIdは "driver" または数値のみ受け付ける
- Executorがアクティブでない場合は400エラー

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "threadId": 1,
    "threadName": "main",
    "threadState": "RUNNABLE",
    "stackTrace": { "elems": ["at java.lang.Thread.run()"] },
    "blockedByThreadId": null,
    "blockedByLock": "",
    "holdingLocks": [],
    "synchronizers": [],
    "monitors": [],
    "lockName": null,
    "lockOwnerName": null,
    "suspended": false,
    "inNative": false,
    "isDaemon": false,
    "priority": 5
  }
]
```

**レスポンス（エラー時）**

- `400 Bad Request` - executorIdの形式が不正
- `404 Not Found` - Executorが存在しない、またはスレッドダンプが利用不可
- `503 Service Unavailable` - HistoryServer経由でのアクセス

---

#### 16. タスクスレッドダンプ取得

実行中のタスクのスレッドダンプを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/threads` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| taskId | long | ○ | - | タスクID |
| executorId | string | ○ | - | ExecutorID |

**レスポンス（成功時）**

ステータスコード: `200 OK`（単一のThreadStackTraceオブジェクト）

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（タスクが指定Executor上で実行されていない場合）

---

### Processes

#### 17. 雑多なプロセス一覧取得

アプリケーションに関連する雑多なプロセス（非Executorプロセス）の一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/allmiscellaneousprocess` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": "process-1",
    "hostPort": "host:1234",
    "isActive": true,
    "totalCores": 2,
    "addTime": "2024-01-01T12:00:00.000GMT",
    "removeTime": null,
    "processLogs": {}
  }
]
```

---

### Storage

#### 18. RDDストレージ一覧取得

キャッシュされたRDDの一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/storage/rdd` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 0,
    "name": "MyRDD",
    "numPartitions": 10,
    "numCachedPartitions": 10,
    "storageLevel": "Memory Deserialized 1x Replicated",
    "memoryUsed": 1048576,
    "diskUsed": 0,
    "dataDistribution": null,
    "partitions": null
  }
]
```

---

#### 19. RDDストレージ情報取得

指定したRDDのストレージ情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/storage/rdd/{rddId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| rddId | int | ○ | RDDID（数値） |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（RDDが存在しない場合）

---

### Environment

#### 20. アプリケーション環境情報取得

アプリケーションのランタイム環境情報（JVM情報、Sparkプロパティ、システムプロパティ等）を取得する。機密情報はredactされる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/environment` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "runtime": {
    "javaVersion": "17.0.1",
    "javaHome": "/usr/lib/jvm/java-17",
    "scalaVersion": "2.13.12"
  },
  "sparkProperties": [
    ["spark.app.name", "MyApp"],
    ["spark.master", "spark://master:7077"]
  ],
  "hadoopProperties": [],
  "systemProperties": [],
  "metricsProperties": [],
  "classpathEntries": [],
  "resourceProfiles": [
    {
      "id": 0,
      "executorResources": {},
      "taskResources": {}
    }
  ]
}
```

---

### Logs

#### 21. イベントログダウンロード

アプリケーションのイベントログをZIPファイルとしてダウンロードする。HistoryServerからのみ利用可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/logs` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |
| レスポンス形式 | application/octet-stream |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

- Content-Disposition: `attachment; filename=eventLogs-{appId}.zip`
- Content-Type: `application/octet-stream`

**レスポンス（エラー時）**

ステータスコード: `503 Service Unavailable`（イベントログが利用不可の場合）

---

### SQL

#### 22. SQL実行一覧取得

Spark SQLの実行一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/sql` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| details | boolean | - | true | ノード・エッジ詳細を含めるか |
| planDescription | boolean | - | true | 物理実行プランの説明を含めるか |
| offset | int | - | 0 | 結果の開始位置 |
| length | int | - | 20 | 返却件数 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 0,
    "status": "COMPLETED",
    "description": "SELECT * FROM table",
    "planDescription": "== Physical Plan ==\n...",
    "submissionTime": "2024-01-01T12:00:00.000GMT",
    "duration": 5000,
    "runningJobIds": [],
    "successJobIds": [0, 1],
    "failedJobIds": [],
    "nodes": [
      {
        "nodeId": 0,
        "nodeName": "Scan",
        "wholeStageCodegenId": 1,
        "metrics": [
          { "name": "number of output rows", "value": "1000" }
        ]
      }
    ],
    "edges": [
      { "fromId": 1, "toId": 0 }
    ]
  }
]
```

---

#### 23. SQL実行情報取得

指定したSQL実行の詳細情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/sql/{executionId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| executionId | long | ○ | SQL実行ID（数値） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| details | boolean | - | true | ノード・エッジ詳細を含めるか |
| planDescription | boolean | - | true | 物理実行プランの説明を含めるか |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（SQL実行が存在しない場合）

---

### Streaming

#### 24. ストリーミング統計情報取得

Spark Streamingの統計情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/statistics` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "startTime": "2024-01-01T12:00:00.000GMT",
  "batchDuration": 1000,
  "numReceivers": 2,
  "numActiveReceivers": 2,
  "numInactiveReceivers": 0,
  "numTotalCompletedBatches": 500,
  "numRetainedCompletedBatches": 100,
  "numActiveBatches": 1,
  "numProcessedRecords": 50000,
  "numReceivedRecords": 50100,
  "avgInputRate": 100.0,
  "avgSchedulingDelay": 5,
  "avgProcessingTime": 800,
  "avgTotalDelay": 805
}
```

---

#### 25. レシーバー一覧取得

Spark Streamingのレシーバー一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/receivers` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "streamId": 0,
    "streamName": "SocketReceiver",
    "isActive": true,
    "executorId": "1",
    "executorHost": "worker-1",
    "lastErrorTime": null,
    "lastErrorMessage": null,
    "lastError": null,
    "avgEventRate": 100.5,
    "eventRates": [[1704106800000, 100.0], [1704106801000, 101.0]]
  }
]
```

---

#### 26. レシーバー情報取得

指定したレシーバーの情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/receivers/{streamId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| streamId | int | ○ | ストリームID（数値） |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（レシーバーが存在しない場合）

---

#### 27. バッチ一覧取得

Spark Streamingのバッチ一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/batches` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| status | JList[BatchStatus] | - | 全ステータス | バッチステータスでフィルタ（COMPLETED, QUEUED, PROCESSING） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "batchId": 1704106800000,
    "batchTime": "2024-01-01T12:00:00.000GMT",
    "status": "COMPLETED",
    "batchDuration": 1000,
    "inputSize": 100,
    "schedulingDelay": 5,
    "processingTime": 800,
    "totalDelay": 805,
    "numActiveOutputOps": 0,
    "numCompletedOutputOps": 2,
    "numFailedOutputOps": 0,
    "numTotalOutputOps": 2,
    "firstFailureReason": null
  }
]
```

---

#### 28. バッチ情報取得

指定したバッチの情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/batches/{batchId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| batchId | long | ○ | バッチID（ミリ秒エポック時刻） |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（バッチが存在しない場合）

---

#### 29. バッチ出力操作一覧取得

指定したバッチの出力操作一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/batches/{batchId}/operations` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "outputOpId": 0,
    "name": "foreachRDD",
    "description": "foreachRDD at MyApp.scala:50",
    "startTime": "2024-01-01T12:00:00.000GMT",
    "endTime": "2024-01-01T12:00:00.800GMT",
    "duration": 800,
    "failureReason": null,
    "jobIds": [10, 11]
  }
]
```

---

#### 30. バッチ出力操作情報取得

指定した出力操作の情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/applications/{appId}/streaming/batches/{batchId}/operations/{outputOpId}` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| appId | string | ○ | アプリケーションID |
| batchId | long | ○ | バッチID |
| outputOpId | int | ○ | 出力操作ID（数値） |

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`（出力操作が存在しない場合）

---

### Prometheus

#### 31. Prometheusメトリクス取得

Executorのメトリクスをprometheus形式のテキストで取得する。Experimental機能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /metrics/executors/prometheus` |
| 認証 | UI認証設定に依存 |
| 権限 | UI閲覧権限 |
| レスポンス形式 | text/plain |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```
spark_info{version="4.0.0", revision="abcdef"} 1.0
metrics_executor_rddBlocks{application_id="app-001", application_name="MyApp", executor_id="1"} 5
metrics_executor_memoryUsed_bytes{...} 1048576
metrics_executor_diskUsed_bytes{...} 0
metrics_executor_totalCores{...} 4
metrics_executor_maxTasks{...} 4
metrics_executor_activeTasks{...} 2
metrics_executor_failedTasks_total{...} 0
metrics_executor_completedTasks_total{...} 50
metrics_executor_totalTasks_total{...} 52
metrics_executor_totalDuration_seconds_total{...} 60.0
metrics_executor_totalGCTime_seconds_total{...} 0.5
metrics_executor_totalInputBytes_bytes_total{...} 10240
metrics_executor_totalShuffleRead_bytes_total{...} 5120
metrics_executor_totalShuffleWrite_bytes_total{...} 2048
metrics_executor_maxMemory_bytes{...} 536870912
```

---

### REST Submission API

#### 32. アプリケーション投入

Standaloneクラスタにアプリケーションを投入する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /v1/submissions/create` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**リクエストボディ**

```json
{
  "action": "CreateSubmissionRequest",
  "clientSparkVersion": "4.0.0",
  "appResource": "hdfs:///path/to/app.jar",
  "mainClass": "com.example.MyApp",
  "appArgs": ["arg1", "arg2"],
  "sparkProperties": {
    "spark.app.name": "MyApp",
    "spark.master": "spark://master:7077",
    "spark.driver.memory": "2g",
    "spark.driver.cores": "2"
  },
  "environmentVariables": {
    "SPARK_ENV_VAR": "value"
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| action | string | ○ | "CreateSubmissionRequest" |
| clientSparkVersion | string | ○ | クライアントのSparkバージョン |
| appResource | string | ○ | アプリケーションJARのパス |
| mainClass | string | ○ | メインクラス名 |
| appArgs | string[] | - | アプリケーション引数 |
| sparkProperties | Map[string,string] | - | Sparkプロパティ |
| environmentVariables | Map[string,string] | - | 環境変数 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "CreateSubmissionResponse",
  "serverSparkVersion": "4.0.0",
  "submissionId": "driver-20240101120000-0000",
  "success": true,
  "message": "Driver successfully submitted"
}
```

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`

```json
{
  "action": "ErrorResponse",
  "serverSparkVersion": "4.0.0",
  "message": "Malformed request: ..."
}
```

---

#### 33. サブミッション終了

指定したサブミッション（ドライバー）を終了させる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /v1/submissions/kill/{submissionId}` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| submissionId | string | ○ | サブミッションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "KillSubmissionResponse",
  "serverSparkVersion": "4.0.0",
  "submissionId": "driver-20240101120000-0000",
  "success": true,
  "message": "Kill request for driver-20240101120000-0000 submitted"
}
```

**レスポンス（エラー時）**

ステータスコード: `400 Bad Request`（サブミッションIDが未指定の場合）

---

#### 34. 全サブミッション終了

全てのサブミッション（ドライバー）を終了させる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /v1/submissions/killall` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "KillAllSubmissionResponse",
  "serverSparkVersion": "4.0.0",
  "success": true,
  "message": "All drivers killed"
}
```

---

#### 35. サブミッションステータス取得

指定したサブミッションの状態を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /v1/submissions/status/{submissionId}` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| submissionId | string | ○ | サブミッションID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "SubmissionStatusResponse",
  "serverSparkVersion": "4.0.0",
  "submissionId": "driver-20240101120000-0000",
  "success": true,
  "driverState": "RUNNING",
  "workerId": "worker-20240101120000-0000",
  "workerHostPort": "worker-1:7078",
  "message": null
}
```

---

#### 36. 完了済みクリア

完了済みのドライバーとアプリケーションをクリアする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /v1/submissions/clear` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "ClearResponse",
  "serverSparkVersion": "4.0.0",
  "success": true,
  "message": ""
}
```

---

#### 37. Masterヘルスチェック

Masterの準備状態を確認する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /v1/submissions/readyz` |
| 認証 | フィルタ設定に依存 |
| 権限 | なし |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "action": "ReadyzResponse",
  "serverSparkVersion": "4.0.0",
  "success": true,
  "message": ""
}
```

**レスポンス（エラー時）**

ステータスコード: `503 Service Unavailable`

```json
{
  "action": "ErrorResponse",
  "serverSparkVersion": "4.0.0",
  "message": "Master is not ready."
}
```

---

## 備考

- Status API（Monitoring API）は全てGETメソッドのみ対応しており、読み取り専用のAPIである
- Status APIのパスで `{appId}` の後に `{attemptId}` を指定するパターンは全エンドポイントで利用可能（例: `/api/v1/applications/{appId}/{attemptId}/jobs`）
- REST Submission APIはStandaloneクラスタモードでのみ利用可能であり、YARN/Mesos/Kubernetesモードでは利用できない
- Prometheusメトリクスエンドポイントは `@Experimental` アノテーションが付与されており、将来変更される可能性がある
- レスポンスのJSON変換はJacksonライブラリ（JacksonMessageWriter）によって自動的に行われる
- 機密情報（パスワード等を含むSparkプロパティ）は環境情報APIにおいてredact（伏せ字処理）される
- REST Submission APIのプロトコルバージョンは現在 `v1` であり、バージョン不一致時にはステータスコード468が返される
